#!/usr/bin/env python
#-*- coding: latin-1 -*-

# mdt25.py 081204
# Copyright 2008 Gregorio Díaz-Marta Mateos
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with This program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""Download mdt25 rasters from IDEE WCS.

Usage:

python mtd25.py x1 y1 x2 y2 dst
python mtd25.py x y -w width [-h height] dst
python mtd25.py x y -r radius dst

Several 25 km x 25 km files are created and lower left coordinates added to
the file names.

Format is AsciiGrid with float32 pixel values and EPSG:23030 srs but
projection information is not stored. If gdal python bindings are available
then projection information is added to the files and GeoTIFF files with int16
pixel values and a VRT mosaic of them are generated.

"""

import os
import sys
import urllib
import getopt

try:
    from helper import add_coords
    from gdalhelper import gdal_translate, add_srs, gdal
    from gdal_vrtmerge import gdal_vrtmerge
    gdal_available = True
except ImportError, e:
    print e
    gdal_available = False

def usage():
    """Print usage advices."""
    print __doc__

def processargs(args):
    """Process command line arguments.

    Return a tuple (x1, y1, x2, y2, dstname).

    """

    opts, args = getopt.gnu_getopt(args, 'h:w:r:')
    optdict =dict(opts)

    if '-r' in optdict:
        radius = int(optdict['-r'])
        x = int(args[0])
        y = int(args[1])
        x1 = x - radius
        y1 = y - radius
        x2 = x + radius
        y2 = y + radius
        dstname = args[2]

    elif '-w' in optdict:
        width = int(optdict['-w'])
        x1 = int(args[0])
        y1 = int(args[1])
        x2 = x1 + width
        if '-h' in optdict:
            height = int(optdict['-h'])
        else:
            height = width
        y2 = y1 + height
        dstname = args[2]

    else:
        x1 = int(args[0])
        y1 = int(args[1])
        x2 = int(args[2])
        y2 = int(args[3])
        dstname = args[4]

    return x1, y1, x2, y2, dstname

def to_gtiff(srcname):
    """Translate an IDEE mdt25 raster into GeoTIFF."""
    name, ext = os.path.splitext(srcname)
    dstname = os.path.extsep.join((name, 'tif'))
    gdal_translate(srcname, dstname, ot=gdal.GDT_Int16, a_srs='EPSG:23030')

def download_single_mdt25(x1, y1, x2, y2, dstname):
    """Download a mdt25 raster from IDEE WCS.

    Format is AsciiGrid with float32 pixel values and EPSG:23030 srs but
    projection information is not stored. If gdal python bindings are
    available then projection information is added and also a GeoTIFF with
    int16 pixel values is generated.

    """

    url = 'http://www.idee.es/wcs/IDEE-WCS-UTM30N/wcsServlet?' + \
          'SERVICE=WCS&' + \
          'REQUEST=GetCoverage&' + \
          'VERSION=1.1.1c1&' + \
          'CRS=EPSG:23030&' + \
          'BBOX=%d,%d,%d,%d&' + \
          'COVERAGE=MDT25_peninsula_ZIP&' + \
          'RESX=25&RESY=25&' + \
          'FORMAT=AsciiGrid&' + \
          'EXCEPTIONS=XML'

    url = url % (x1, y1, x2, y2)
    urllib.urlretrieve(url, dstname)

    if gdal_available:
        add_srs(dstname, 'EPSG:23030')
        to_gtiff(dstname)

def download_mdt25(x1, y1, x2, y2, dstname, size=25000):
    """Download mdt25 rasters from IDEE WCS.

    Several size x size files are created and lower left coordinates added to
    the file names.

    Format is AsciiGrid with float32 pixel values and EPSG:23030 srs but
    projection information is not stored. If gdal python bindings are
    available then projection information is added to the files and GeoTIFF
    files with int16 pixel values and a VRT mosaic of them are generated.

    """

    names = []
    for xx1 in range(x1, x2, size):
        xx2 = min(x2, xx1 + size)
        for yy1 in range(y1, y2, size):
           yy2 = min(y2, yy1 + size)
           name = add_coords(dstname, xx1, yy1)
           download_single_mdt25(xx1, yy1, xx2, yy2, name)
           if gdal_available:
               name, ext = os.path.splitext(name)
               name = os.path.extsep.join((name, 'tif'))
               names.append(name)

    if gdal_available:
        name, ext = os.path.splitext(dstname)
        name = os.path.extsep.join((name, 'vrt'))
        gdal_vrtmerge(names, out_file=name)

if __name__ == '__main__':
    try:
        x1, y1, x2, y2, dstname = processargs(sys.argv[1:])
    except Exception, e:
        print e
        usage()
        sys.exit(1)

    download_mdt25(x1, y1, x2, y2, dstname, size=25000)
