#!/usr/bin/env python
#-*- coding: latin-1 -*-

# Copyright 2008 Gregorio Díaz-Marta Mateos
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with This program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""Download pnoa images from IDEE WMS.

Since the size of files served by the WMS is limited several files are downloadad if needed and a VRT mosaic of them is created.

Note that some features like creation of VRT files or addition of srs information won't work if GDAL python bindings are not available.

Usage:

python pnoa.py x y [x2 y2] [options] dstfile

x, y:
    Lower-left coordinates of the region (or center coordinates if -r is used)
    in EPSG:23030 (UTM ED50 Spain zone 30)

x2, y2:
    Upper-right coordinates of the region in EPSG:23030 (UTM ED50 Spain zone
    30). Mandatory unless -w or -r are used.

dstfile:
    Name of the destination file. Note that depending of region size and the
    options used several files can be created an named after this name.

Options:

-w width:
    Width of the region in meters. If -h is not used then height = width is
    assumed. Don't use -w with x2, y2.

-h height:
    Height of the region in meters. Use only with -w.

-r radius:
    Radius of the maximum circle that fits in the region. Don't use with -w.
    Note that x, y are interpreted as center coordinates in this case.

-m mpp:
    Resolution in meters per pixel. If not used then resolution is computed in
    such a way that the specified region fits in a single image of the
    maximum available size (2000 px x 1700 px).

-g:
    Create wld files (ESRI world files) for each image.

"""

import os
import sys
import urllib
import getopt
from helper import add_coords, gen_wld

try:
    from gdalhelper import add_srs
    from gdal_vrtmerge import gdal_vrtmerge
    gdal_available = True
except ImportError:
    gdal_available = False

MaxXSize = 2000  # Maximum X size in pixels
MaxYSize = 1700  # Maximum Y size in pixels

def usage():
    """Print usage advices."""
    print __doc__

def processargs(args):
    """Process command line arguments."""
    opts, args = getopt.gnu_getopt(args, 'h:w:r:m:g')
    optdict =dict(opts)

    if '-r' in optdict:
        radius = int(optdict['-r'])
        x = int(args[0])
        y = int(args[1])
        x1 = x - radius
        y1 = y - radius
        x2 = x + radius
        y2 = y + radius
        dstname = args[2]

    elif '-w' in optdict:
        width = int(optdict['-w'])
        x1 = int(args[0])
        y1 = int(args[1])
        x2 = x1 + width
        if '-h' in optdict:
            height = int(optdict['-h'])
            y2 = y1 + height
        else:
            y2 = y1 + width
        dstname = args[2]

    else:
        x1 = int(args[0])
        y1 = int(args[1])
        x2 = int(args[2])
        y2 = int(args[3])
        dstname = args[4]

    if '-m' in optdict:
        res = float(optdict['-m'])
    else:
        res = None

    if '-g' in optdict:
        wld = True
    else:
        wld = False
        

    return x1, y1, x2, y2, dstname, res, wld

def download_pnoa(x1, y1, x2, y2, dstname, res=None, wld=False):
    """Download a pnoa image from IDEE WMS.

    res: resolution in meters per pixel.
    wld: whether we want to generate a wld file.

    """

    # m = px * res
    # px = m / res

    dx = float(x2 - x1)
    dy = float(y2 - y1)
    if res is None:
        res = max(dx / MaxXSize, dy / MaxYSize)

    width = round(dx / res)
    height = round(dy / res)
    url = 'http://www.idee.es/wms/PNOA/PNOA?' + \
          'REQUEST=GetMap&' + \
          'VERSION=1.1.1&' + \
          'SERVICE=WMS&' + \
          'SRS=EPSG:23030&' + \
          'BBOX=%d,%d,%d,%d&' + \
          'WIDTH=%d&HEIGHT=%d&' + \
          'LAYERS=pnoa&' + \
          'STYLES=default&' + \
          'FORMAT=image/jpeg'

    url = url % (x1, y1, x2, y2, width, height)
    urllib.urlretrieve(url, dstname)


    # Add srs and geotransform.
    gt = (x1, res, 0.0, y2, 0.0, -res)
    if gdal_available:
        add_srs(dstname, 'EPSG:23030', gt=gt)
    if wld:
        gen_wld(gt, dstname)
        

def download_pnoa_mosaic(x1, y1, x2, y2, dstname, res, wld):
    """Download pnoa images from IDEE WMS.

    Returns a list of file names.

    res: resolution in meters per pixel.
    wld: whether we want to generate a wld file.

    """

    names = []
    for xx1 in range(x1, x2, int(round(MaxXSize * res))):
        xx2 = min(x2, xx1 + MaxXSize * res)
        for yy1 in range(y1, y2, int(round(MaxYSize * res))):
           yy2 = min(y2, yy1 + MaxYSize * res)
           name = add_coords(dstname, xx1, yy1)
           download_pnoa(xx1, yy1, xx2, yy2, name, res, wld)
           names.append(name)
    return names

if __name__ == '__main__':
    try:
        x1, y1, x2, y2, dstname, res, wld = processargs(sys.argv[1:])
    except Exception, e:
        print e
        usage()
        sys.exit(1)
    if res is None:
        names = download_pnoa(x1, y1, x2, y2, dstname, res, wld),
    else:
        names = download_pnoa_mosaic(x1, y1, x2, y2, dstname, res, wld)
    if gdal_available and len(names) > 1:
        name, ext = os.path.splitext(dstname)
        name = os.path.extsep.join((name, 'vrt'))
        gdal_vrtmerge(names, out_file=name)


