#!/usr/bin/env python
# -*- coding: latin-1 -*-

# CurveEditorWindow.py
# Copyright 2008 Gregorio Díaz-Marta Mateos
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


import pygtk
pygtk.require('2.0')
import gtk
import matplotlib
matplotlib.use('GtkAgg')
matplotlib.interactive(True)
import pylab
import CurveEditor as ce
import windpower as wp
import os


class CurveEditorWindow:

    def newbutton_clicked(self, widget, data=None):
        self.newCurve()

    def openbutton_clicked(self, widget, data=None):
        self.openCurve()

    def savebutton_clicked(self, widget, data=None):
        filename = self.fileName()
        if filename is None:
            self.saveasbutton_clicked(widget)
        else:
            self.saveCurve(filename)

    def saveasbutton_clicked(self, widget, data=None):
        chooser = gtk.FileChooserDialog(title='Guardar como...,',
                                        action=gtk.FILE_CHOOSER_ACTION_SAVE,
                                        buttons=(gtk.STOCK_CANCEL,
                                                 gtk.RESPONSE_CANCEL,
                                                 gtk.STOCK_SAVE,
                                                 gtk.RESPONSE_OK))
        filename = self.fileName()
        if filename is None:
            chooser.set_current_name('new_curve.txt')
        else:
            chooser.set_filename(filename)
        chooser.set_do_overwrite_confirmation(True)
        response = chooser.run()
        filename = chooser.get_filename()
        chooser.destroy()
        if response != gtk.RESPONSE_OK:
            return
        self.saveCurve(filename)

    def addbutton_clicked(self, widget, data=None):
        self.editor.add_row()

    def removebutton_clicked(self, widget, data=None):
        self.editor.remove_row()

    def plotbutton_clicked(self, widget, data=None):
        filename = self.fileName()
        if filename is None:
            thelabel = ''
        else:
            thelabel = os.path.basename(filename)

        self.editor.curve().plot(marker='o', linewidth=1.5, label=thelabel)
        pylab.grid(True)
        pylab.xlabel('Speed [m/s]')
        pylab.ylabel('Power [kW]')
        pylab.legend(loc='best')

    def destroy(self, widget, data=None):
        gtk.main_quit()

    def newCurve(self):
        """Open a new curve."""
        self.setCurve(wp.PowerCurve())

    def openCurve(self):
        """Open a curve."""
        dialog = gtk.FileChooserDialog(title='Open curve',
                                       action=gtk.FILE_CHOOSER_ACTION_OPEN,
                                       buttons=(gtk.STOCK_CANCEL,
                                                gtk.RESPONSE_CANCEL,
                                                gtk.STOCK_OPEN,
                                                gtk.RESPONSE_OK))
        dialog.set_default_response(gtk.RESPONSE_OK)
        response = dialog.run()
        filename = dialog.get_filename()
        dialog.destroy()
        if response == gtk.RESPONSE_OK:
            self.loadCurve(filename)

    def loadCurve(self, filename):
        """Load a curve from a file."""
        curve = wp.readCurve(filename)
        self.setCurve(curve)
        self.setFileName(filename)

    def setCurve(self, curve):
        """Set the curve."""
        self.editor.setCurve(curve)
        self.setHasChanges(False)

    def fileName(self, default=None):
        """Return the file name."""
        filename = self._fileName
        if filename is None:
            filename = default
        return filename

    def setFileName(self, filename):
        """Set the file name and update the window title."""
        self._fileName = filename
        self.updateWindowTitle()

    def setHasChanges(self, hasChanges):
        self._hasChanges = hasChanges
        self.updateWindowTitle()

    def hasChanges(self):
        """Return whether there are unsaved changes."""
        try:
            haschanges = self._hasChanges
        except AttributeError:
            haschanges = False
        return haschanges

    def updateWindowTitle(self):
        """Set the window title based on internal state."""
        filename = self.fileName()
        if filename is None:
            basename = 'No file'
            dirname = ''
        else:
            basename = os.path.basename(filename)
            dirname = os.path.dirname(filename)
        title = basename + ' (' + dirname + ')' + ' - PowerCurved'
        if self.hasChanges():
            title = '*' + title
        self.window.set_title(title)

    def saveCurve(self, filename):
        curve = self.editor.curve()
        curve.save(filename)
        self.setFileName(filename)
        self.setHasChanges(False)

    def curve_changed_cb(self, event):
        self.setHasChanges(True)

    def makeToolBar(self):
        newbuttonTip = 'New curve'
        openbuttonTip = 'Open file'
        savebuttonTip = 'Save current file'
        saveasbuttonTip = 'Save to a different file'
        addbuttonTip = 'Add'
        removebuttonTip = 'Remove'
        quitbuttonTip = 'Quit program'
        plotbuttonTip = 'Plot curve'

        toolbar = gtk.Toolbar()

        newbutton = gtk.ToolButton(gtk.STOCK_NEW)
        newbutton.connect('clicked', self.newbutton_clicked)
        self.tooltips.set_tip(newbutton, newbuttonTip, tip_private=None)
        toolbar.insert(newbutton, -1)

        openbutton = gtk.ToolButton(gtk.STOCK_OPEN)
        openbutton.set_is_important(True)
        openbutton.connect('clicked', self.openbutton_clicked)
        self.tooltips.set_tip(openbutton, openbuttonTip, tip_private=None)
        toolbar.insert(openbutton, -1)

        savebutton = gtk.ToolButton(gtk.STOCK_SAVE)
        savebutton.set_is_important(True)
        savebutton.connect('clicked', self.savebutton_clicked)
        self.tooltips.set_tip(savebutton, savebuttonTip, tip_private=None)
        toolbar.insert(savebutton, -1)

        saveasbutton = gtk.ToolButton(gtk.STOCK_SAVE_AS)
        saveasbutton.connect('clicked', self.saveasbutton_clicked)
        self.tooltips.set_tip(saveasbutton, saveasbuttonTip, tip_private=None)
        toolbar.insert(saveasbutton, -1)

        separator1 = gtk.SeparatorToolItem()
        toolbar.insert(separator1, -1)

        addbutton = gtk.ToolButton(gtk.STOCK_ADD)
        addbutton.connect('clicked', self.addbutton_clicked)
        self.tooltips.set_tip(addbutton, addbuttonTip, tip_private=None)
        toolbar.insert(addbutton, -1)

        removebutton = gtk.ToolButton(gtk.STOCK_REMOVE)
        removebutton.connect('clicked', self.removebutton_clicked)
        self.tooltips.set_tip(removebutton, removebuttonTip, tip_private=None)
        toolbar.insert(removebutton, -1)

        separator1 = gtk.SeparatorToolItem()
        toolbar.insert(separator1, -1)

        plotbutton = gtk.ToolButton(label='Plot')
        plotbutton.set_is_important(True)
        plotbutton.connect('clicked', self.plotbutton_clicked)
        self.tooltips.set_tip(plotbutton, plotbuttonTip, tip_private=None)
        toolbar.insert(plotbutton, -1)

        separator2 = gtk.SeparatorToolItem()
        separator2.set_expand(True)
        separator2.set_draw(False)
        toolbar.insert(separator2, -1)

        quitbutton = gtk.ToolButton(gtk.STOCK_QUIT)
        quitbutton.connect("clicked", self.destroy)
        self.tooltips.set_tip(quitbutton, quitbuttonTip, tip_private=None)
        toolbar.insert(quitbutton, -1)

        return toolbar

    def __init__(self):
        appName = 'PowerCurved'

        self.tooltips = gtk.Tooltips()

        self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
        self.window.set_title(appName)
        self.window.connect("destroy", self.destroy)

        mainbox = gtk.VBox()
        self.window.add(mainbox)

        toolbar = self.makeToolBar()
        mainbox.pack_start(toolbar, expand = False)

        self.editor = ce.CurveEditor(wp.PowerCurve())
        mainbox.pack_start(self.editor)

        self.setFileName(None)
        self.setHasChanges(False)
        self.window.show_all()

        self.editor.curve_changed.register(self.curve_changed_cb)
