#!/usr/bin/env python
# -*- coding: latin-1 -*-

# CurveInfoEditor.py
# Copyright 2008 Gregorio Díaz-Marta Mateos
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


import pygtk
pygtk.require('2.0')
import gtk
import windpower as wp
from Observable import ObservableEvent


class CurveInfoEditor(gtk.Table):
    """A curve info widget."""
    def __init__(self, curve):
        gtk.Table.__init__(self, 6, 2)

        self.set_has_changes(False)

        self.changed = ObservableEvent()

        self.has_explicit_rated_power = gtk.CheckButton('Explicit rated power')
        self.attach(self.has_explicit_rated_power, 0, 2, 0, 1)
        self.has_explicit_rated_power.connect(
            'toggled', self.has_explicit_rated_power_toggled)

        label = gtk.Label('Rated power:')
        self.attach(label, 0, 1, 1, 2)
        self.rated_power = gtk.Entry()
        self.attach(self.rated_power, 1, 2, 1, 2)
        self.rated_power.connect('changed', self.rated_power_changed)
        self.rated_power.connect('focus-out-event',
                                 self.rated_power_focus_out)

        self.has_explicit_cut_in = gtk.CheckButton('Explicit cut in')
        self.attach(self.has_explicit_cut_in, 0, 2, 2, 3)
        self.has_explicit_cut_in.connect(
            'toggled', self.has_explicit_cut_in_toggled)

        label = gtk.Label('Cut in:')
        self.attach(label, 0, 1, 3, 4)
        self.cut_in = gtk.Entry()
        self.attach(self.cut_in, 1, 2, 3, 4)
        self.cut_in.connect('changed', self.cut_in_changed)
        self.cut_in.connect('focus-out-event', self.cut_in_focus_out)

        self.has_explicit_cut_out = gtk.CheckButton('Explicit cut out')
        self.attach(self.has_explicit_cut_out, 0, 2, 4, 5)
        self.has_explicit_cut_out.connect(
            'toggled', self.has_explicit_cut_out_toggled)

        label = gtk.Label('Cut out:')
        self.attach(label, 0, 1, 5, 6)
        self.cut_out = gtk.Entry()
        self.attach(self.cut_out, 1, 2, 5, 6)
        self.cut_out.connect('changed', self.cut_out_changed)
        self.cut_out.connect('focus-out-event', self.cut_out_focus_out)

        self.set_curve(curve)

    def has_explicit_rated_power_toggled(self, togglebutton):
        has_explicit_rated_power = togglebutton.get_active()
        self.rated_power.set_editable(has_explicit_rated_power)
        curve = self.get_curve()
        if has_explicit_rated_power:
            curve.setRatedPower(self.get_rated_power())
        else:
            curve.unsetRatedPower()
        self.changed.happen()

    def rated_power_changed(self, editable):
        self.set_has_changes(True)

    def rated_power_focus_out(self, widget, event):
        if self.get_has_changes():
            self.set_has_changes(False)
            curve = self.get_curve()
            curve.setRatedPower(self.get_rated_power())
            self.changed.happen()

    def has_explicit_cut_in_toggled(self, togglebutton):
        has_explicit_cut_in = togglebutton.get_active()
        self.cut_in.set_editable(has_explicit_cut_in)
        curve = self.get_curve()
        if has_explicit_cut_in:
            curve.setCutIn(self.get_cut_in())
        else:
            curve.unsetCutIn()
        self.changed.happen()

    def cut_in_changed(self, editable):
        self.set_has_changes(True)

    def cut_in_focus_out(self, widget, event):
        if self.get_has_changes():
            self.set_has_changes(False)
            curve = self.get_curve()
            curve.setCutIn(self.get_cut_in())
            self.changed.happen()

    def has_explicit_cut_out_toggled(self, togglebutton):
        has_explicit_cut_out = togglebutton.get_active()
        self.cut_out.set_editable(has_explicit_cut_out)
        curve = self.get_curve()
        if has_explicit_cut_out:
            curve.setCutOut(self.get_cut_out())
        else:
            curve.unsetCutOut()
        self.changed.happen()

    def cut_out_changed(self, editable):
        self.set_has_changes(True)

    def cut_out_focus_out(self, widget, event):
        if self.get_has_changes():
            self.set_has_changes(False)
            curve = self.get_curve()
            curve.setCutOut(self.get_cut_out())
            self.changed.happen()

    def update(self):
        self.set_curve(self.get_curve())

    def set_curve(self, curve):
        self._curve = curve

        has_explicit_rated_power = curve.hasExplicitRatedPower()
        try:
            rated_power = curve.ratedPower()
        except ValueError:
            rated_power = ''

        has_explicit_cut_in = curve.hasExplicitCutIn()
        try:
            cut_in = curve.cutIn()
        except ValueError:
            cut_in = ''

        has_explicit_cut_out = curve.hasExplicitCutOut()
        try:
            cut_out = curve.cutOut()
        except ValueError:
            cut_out = ''

        self.has_explicit_rated_power.set_active(has_explicit_rated_power)
        self.rated_power.set_text(str(rated_power))
        self.has_explicit_cut_in.set_active(has_explicit_cut_in)
        self.cut_in.set_text(str(cut_in))
        self.has_explicit_cut_out.set_active(has_explicit_cut_out)
        self.cut_out.set_text(str(cut_out))

        # XXX Setting the curve shouldn't set_has_changes(True), but it does
        # so we need to set it to False again.
        self.set_has_changes(False)

    def get_curve(self):
        return self._curve

    def get_has_explicit_rated_power(self):
        return self.has_explicit_rated_power.get_active()

    def get_rated_power(self):
        try:
            output = float(self.rated_power.get_text())
        except ValueError:
            output = None
        return output

    def get_has_explicit_cut_in(self):
        return self.has_explicit_cut_in.get_active()

    def get_cut_in(self):
        try:
            output = float(self.cut_in.get_text())
        except ValueError:
            output = None
        return output

    def get_has_explicit_cut_out(self):
        return self.has_explicit_cut_out.get_active()

    def get_cut_out(self):
        try:
            output = float(self.cut_out.get_text())
        except ValueError:
            output = None
        return output

    def get_has_changes(self):
        return self._has_changes

    def set_has_changes(self, has_changes):
        self._has_changes = has_changes
