#!/usr/bin/env python
# -*- coding: latin-1 -*-

# Observable.py
# Copyright 2008 Gregorio Díaz-Marta Mateos
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


_next_handler = 0

def new_handler():
    global _next_handler
    output = _next_handler
    _next_handler += 1
    return output


class ObservableEvent:
    """Just an observable event...

    Instances mantain a list of listeners (functions) which are notified
    (called) on event occurrences.

    """

    def __init__(self):
        self._listeners = {}

    def listeners(self):
        return self._listeners

    def register(self, func, *params):
        """Register a listener.

        Signature of func:
            def func(event, *params)

        """

        h = new_handler()
        self.listeners()[h] = (func, params)
        return h

    def unregister(self, handler):
        """Unregister a listener (denoted by handler)."""
        del self.listeners()[handler]

    def happen(self):
        """Make the event happen and their listeners be notified."""
        for h in self._listeners:
            self.notify(h)

    def notify(self, handler):
        """Notify the listener that the event has happen."""
        func, params = self.listeners()[handler]
        func(self, *params)

