#!/usr/bin/env python
#-*- coding: latin-1 -*-

# windpower.py
# Copyright 2008 Gregorio Díaz-Marta Mateos
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


import math
import numpy
import pylab

class PowerCurve(dict):
    """A wind turbine power curve."""

    def __init__(self):
        dict.__init__(self)
        self.setRatedPower(None)
        self.setCutIn(None)
        self.setCutOut(None)

    def power(self, speed):
        """Power given a wind speed."""
        return self._power1(speed)

    def _power1(self, speed):
        # Test speed range
        cutin, cutout = self.cutIn(), self.cutOut()
        if not (cutin <= speed <= cutout):
            return 0

        # XXX What a mess!
        speeds = sorted(self.keys())
        lspeeds = [s0 for s0 in speeds if s0 <= speed]
        if lspeeds == []:
            return 0
        s0 = lspeeds[-1]
        if s0 == speed:
            return self[speed]
        j = speeds.index(s0)
        if j == len(speeds)-1:
            return 0
        s1 = speeds[j + 1]
        w0 = self[s0]
        w1 = self[s1]
        w = w0 + (w1-w0)*(speed-s0)/(s1-s0)
        return w

    def avgPower(self, k, A, hours=False):
        """Average power given a Weibull distribution."""
        def func(x):
            return (self.power(x) * weibull(x, k, A, False))
        a = self.cutIn()
        b = self.cutOut()
        output = integrate(func, a, b)
        if hours:
            output = output * 365 * 24 / self.ratedPower()
        return output

    def plot(self, **kwargs):
        """Plot the power curve."""
        sortedKeys = sorted(self.keys())
        values = [self[key] for key in sortedKeys]
        response = pylab.plot(sortedKeys, values, **kwargs)
        return response

    def ratedPower(self):
        """Return the rated power."""
        if self._ratedPower is None:
            output = max(self.values())
        else:
            output = self._ratedPower
        return output

    def setRatedPower(self, power):
        """Explicitly set the rated power."""
        self._ratedPower = power

    def unsetRatedPower(self):
        """Unset the explicit rated power."""
        self._ratedPower = None

    def hasExplicitRatedPower(self):
        """True if an explicit rated power has been set."""
        return not self._ratedPower is None

    def cutIn(self):
        """Return the cut in wind speed."""
        if self._cutIn is None:
            return min(self.keys())
        else:
            return self._cutIn

    def cutOut(self):
        """Return the cut out wind speed."""
        if self._cutOut is None:
            return max(self.keys())
        else:
            return self._cutOut

    def setCutIn(self, speed):
        """Explicitly set the cut in wind speed."""
        self._cutIn = speed

    def setCutOut(self, speed):
        """Explicitly set the cut out wind speed."""
        self._cutOut = speed

    def unsetCutIn(self):
        """Unset the explicit cut in wind speed."""
        self._cutIn = None

    def unsetCutOut(self):
        """Unset the explicit cut out wind speed."""
        self._cutOut = None

    def hasExplicitCutIn(self):
        """True if an explicit cut in wind speed has been set."""
        return not self._cutIn is None

    def hasExplicitCutOut(self):
        """True if an explicit cut out wind speed has been set."""
        return not self._cutOut is None

    def save(self, filename):
        """Save the curve to a file."""
        f = open(filename, 'w')
        line = ''
        if self.hasExplicitRatedPower():
            line += '%s\t%s\n' % ('ratedpower', self.ratedPower())
        if self.hasExplicitCutIn():
            line += '%s\t%s\n' % ('cutin', self.cutIn())
        if self.hasExplicitCutOut():
            line += '%s\t%s\n' % ('cutout', self.cutOut())
        f.write(line)
        speeds = sorted(self.keys())
        for s in speeds:
            line = '%s\t%s\n' % (s, self[s])
            f.write(line)
        f.close()


def weibull(x, k, A, cumulative=True):
    """Weibull distribution function."""
    x, k, A = map(float, (x, k, A))
    if cumulative:
        return 1 - math.exp(-(x/A)**k)
    else:
        return (k/A)*(x/A)**(k-1)*math.exp(-(x/A)**k)

def readCurve(fileName):
    """Read a power curve from a file."""
    output = PowerCurve()
    f = open(fileName)
    for line in f:
        key, value = line.split()
        if key == 'ratedpower':
            output.setRatedPower(float(value))
        elif key == 'cutin':
            output.setCutIn(float(value))
        elif key == 'cutout':
            output.setCutOut(float(value))
        else:
            speed, power = map(float, line.split())
            output[speed] = power
    f.close()
    return output

def integrate(func, a, b, dx=0.01):
    """ Integrate func between a and b."""
    # return sum(map(lambda x:dx*x, func(numpy.arange(a,b,dx))))
    return sum(map(lambda x:func(x)*dx, numpy.arange(a,b,dx)))
