/**************************************************************************************

	PROTUX - THE FREE PROFESSIONAL AUDIO TOOLS FOR LINUX
	AUTHOR : See AUTHORS file for details

	This software is distributed under the terms of the GNU General Public License
	as specified in the COPYING file.

***************************************************************************************/
#include <cmath>
#include <mustux.h>
#include "AudioClip.hh"
#include "ColorManager.hh"


AudioClip::AudioClip(Track* pParentTrack,
		Audio *pAudioSource,
		long long pSourceFirstBlock,
		long long pLenght,
		long long pTrackInsertBlock)
	: MustuxDrawableRegion( pParentTrack->get_drawarea() ) ,
	MustuxObject()
	{
	PENTERCONS;
	parentTrack = pParentTrack;
	parentSong  = parentTrack->parentSong;
	audioSource = pAudioSource;
	clipName = audioSource->get_pure_filename();
	blockSize = audioSource->file->blockSize;
	channels = audioSource->file->channels;
	rate = audioSource->file->rate;
	bitDepth = audioSource->file->bitDepth;
	isTake=false;
	maxSample = (int) ( pow(2,bitDepth) / 256 );
	gain = 1.0;
	progress = 0;
	fadeOutBlocks=0;
	fadeInBlocks=0;

	set_blur(false);

	sRate.setNum(rate);
	sBitDepth.setNum(bitDepth);
	sourceType=(isTake?"CAP":"SRC");
	sclipGain.setNum((double)gain,'f',1);
	sclipGain = "G:"+sclipGain;
	clipInfo = sRate +  "  " + sBitDepth + "   " + sourceType + "  " + sclipGain + "   " + clipName;

	sourceFirstBlock= pSourceFirstBlock;
	long long lenght = pLenght;
	sourceLastBlock = sourceFirstBlock + lenght;
	trackFirstBlock = pTrackInsertBlock;
	trackLastBlock = trackFirstBlock + lenght;
	isMuted=false;
	PEXITCONS;
	}


AudioClip::~AudioClip()
	{
	PENTERDES;
	PEXITDES;
	}


Audio* AudioClip::get_audio_source()
	{
	return audioSource;
	}




void AudioClip::update()
	{
	int baseX = parentSong->block_to_xpos(trackFirstBlock);
	int zoomStep = parentSong->get_zoom_step();
	int clipXWidth = ( sourceLastBlock - sourceFirstBlock )  / zoomStep;
	drawArea->update(baseX, parentTrack->real_baseY(), clipXWidth, parentTrack->height);
	}



void AudioClip::clear()
	{
	int baseX = parentSong->block_to_xpos(trackFirstBlock);
	int zoomStep = parentSong->get_zoom_step();
	int clipXWidth = ( sourceLastBlock - sourceFirstBlock )  / zoomStep;
	drawArea->clear(baseX, parentTrack->real_baseY(), clipXWidth, parentTrack->height);
	}


int AudioClip::draw()
	{
	QPainter* painter = drawArea->painter;
	int hzoom = parentSong->get_hzoom();
	int zoomStep = parentSong->get_zoom_step();
	int baseY  = this->get_baseY();
	int height = this->get_height();
	int baseX =  this->get_baseX();
	int clipXWidth = this->get_width();
	int trackWidth = drawArea->width();

	long long firstBlock = sourceFirstBlock;

	if ( baseX < 0)
		{
		firstBlock += (-1*baseX) * zoomStep;
		clipXWidth -= (-1*baseX);
		baseX = 0;
		}

	if (clipXWidth > (trackWidth - baseX))
		clipXWidth = trackWidth - baseX;

	if (clipXWidth<0)
		return 1;

	// clip info area bg
	if (parentTrack->isActive)
		painter->fillRect( baseX , baseY, clipXWidth, 16, CM_COLOR(CLIP_INFO_AREA_BG_ACTIVE));
	else
		painter->fillRect( baseX , baseY, clipXWidth, 16, CM_COLOR(CLIP_INFO_AREA_BG));


	// Draw Clip Info Area
	if (clipXWidth>70)
		{
		painter->setPen(CM_COLOR(DARK_TEXT));
		drawArea->painter->setFont( QFont( "Helvetica", 7 ) );
		for (int i=0; i<channels; i++)
			painter->drawEllipse( baseX + 5 + 4*i, baseY + 5, 8 , 8);
		if (isMuted)
			painter->drawText( baseX + 23, baseY + 12,"M");
		QRect r = QRect(baseX + 20, baseY + 2, clipXWidth -20,20);
		painter->drawText( r, Qt::AlignAuto, clipInfo);
		}


	// Check Cross Fades
	AudioClip* pClip = this->prev;
	AudioClip* nClip = this->next;
	long long pTLB = pClip ? pClip->trackLastBlock : 0;
	long long nTFB = nClip ? nClip->trackFirstBlock : 0;
	bool crossAtLeft  = ((pClip) && (pTLB > trackFirstBlock));
	bool crossAtRight = ((nClip) && (nTFB < trackLastBlock));
	int lCrossX = crossAtLeft  ? parentSong->block_to_xpos(pTLB) - parentSong->block_to_xpos(trackFirstBlock) : 0;
	int rCrossX = crossAtRight ? parentSong->block_to_xpos(trackLastBlock) - parentSong->block_to_xpos(nTFB)  : 0;


	// main bg (dont paint under crossfades)
	if (isMuted)
		painter->fillRect( baseX + lCrossX, baseY+16, clipXWidth - rCrossX - lCrossX, height-16, CM_COLOR(CLIP_BG_MUTED));
	else if (isSelected)
		painter->fillRect( baseX + lCrossX, baseY+16, clipXWidth - rCrossX - lCrossX, height-16, CM_COLOR(CLIP_BG_SELECTED));
	else
		painter->fillRect( baseX + lCrossX, baseY+16, clipXWidth - rCrossX - lCrossX, height-16 , CM_COLOR(CLIP_BG));

	// bg under RIGHT crossfade region  ( only RIGHT !)
	painter->fillRect( baseX + clipXWidth - rCrossX , baseY+16, rCrossX, height-16, QColor(120,150,130));

	// Calculate fades in/out and clip gain
	int xrbfi = baseX ;
	int xrbfo = baseX + clipXWidth;
	int gy = (int) ( (float ) baseY + height - gain * ( height - 16 ) );
	if (parentSong->get_editing_mode()==Song::EDITING_MODE_NORMAL)
		{
		if (fadeInBlocks>0)
			xrbfi = parentSong->block_to_xpos(trackFirstBlock + fadeInBlocks);
		if (fadeOutBlocks>0)
			xrbfo = parentSong->block_to_xpos(trackFirstBlock + ( sourceLastBlock - sourceFirstBlock)  - fadeOutBlocks);
		}
	if (xrbfi < 0 )
		xrbfi = 0;
	if (xrbfo > (baseX + clipXWidth))
		xrbfo = baseX + clipXWidth;

	// Draw the peaks
	int peaksToFill = clipXWidth;
	char* peakBuffer = new char[ peaksToFill * channels ];
	Peak* pk = audioSource->peak;
	if (!pk) // There is no peak for this audio Clip. Maybe this is just a recording box... No need to worry ...
		return -1;

	int peaksFilled = pk->fill_buffer(peakBuffer, firstBlock, peaksToFill, hzoom);
	const int chHeight = (height-16) / channels;
	int dy = 0;
	int yc = 0;
	long long ppos=0;
	char sample;
	if (peaksFilled > 0)
		{
		if ( hzoom <= Peak::MAX_ZOOM_USING_SOURCEFILE ) // MICRO VIEW MODE
			{
			//Draw axis
			painter->setPen(QColor(150,190,200));
			for (int k=0; k<channels; k++)
				{
				int yc = chHeight/2 + chHeight*k + 16;
				painter->drawLine(baseX, baseY + yc, clipXWidth , baseY + yc);
				}
			const float yFactor = (float) chHeight / maxSample;
			// Set peaks foreground color
			int* lastdy = new int[channels];
			for (int k=0; k<channels; k++)
				lastdy[k]=0;
			// peaks outside cross regions
			painter->setPen(CM_COLOR(CLIP_PEAK_MICROVIEW));
			int lastX = baseX + peaksFilled - rCrossX;
			for (int x=baseX; x<lastX; x++)
				{
				float facI = 1.0;
				float facO = 1.0;
				if ((x<xrbfi) && (fadeInBlocks!=0))
					facI = (float)(x-baseX)/(xrbfi-baseX);
				if ((x>xrbfo) && (fadeOutBlocks!=0))
					facO = 1.0 - (float) (x - xrbfo)/(lastX-xrbfo);
				for (int k=0; k<channels; k++)
					{
					yc = chHeight/2 + chHeight*k + 16;
					sample = (char) ( gain * facO * facI * peakBuffer[ppos] * gain * -1); //either micro or macroview is painted upsidedown. macroview has only positive peaks, so we invert microview??
					dy = (int) (yFactor * (float) sample ) ;
					painter->drawLine( x-1 , baseY + yc + lastdy[k] , x , baseY + yc + dy );
					lastdy[k]=dy;
					ppos++;
					}
				}

			// peaks under right cross region ( only right needed )
			if (crossAtRight)
				{
				painter->setPen(Qt::black);
				for (int x=baseX + peaksFilled - rCrossX; x < baseX + peaksFilled; x++)
					{
					float facI = 1.0;
					float facO = 1.0;
					if ((x<xrbfi) && (fadeInBlocks!=0))
						facI = (float)(x-baseX)/(xrbfi-baseX);
					if ((x>xrbfo) && (fadeOutBlocks!=0))
						facO = 1.0 - (float) (x - xrbfo)/(lastX-xrbfo);
					for (int k=0; k<channels; k++)
						{
						yc = chHeight/2 + chHeight*k + 16;
						sample = (char) ( gain * facO * facI * peakBuffer[ppos] * gain * -1); //either micro or macroview is painted upsidedown. macroview has only positive peaks, so we convert microview??
						dy = (int) (yFactor * (float) sample ) ;
						painter->drawLine( x-1 , baseY + yc + lastdy[k] , x , baseY + yc + dy );
						lastdy[k]=dy;
						ppos++;
						}
					}
				}
			delete lastdy;
			}
		else
			{
			// peaks outside cross regions
			const float yFactor = (float) 2.0 * chHeight / maxSample;
			painter->setPen(CM_COLOR(CLIP_PEAK_MACROVIEW));
			int lastX = baseX + peaksFilled - rCrossX;
			for (int x=baseX; x<lastX; x++)
				{
				float facI = 1.0;
				float facO = 1.0;
				if ((x<xrbfi) && (fadeInBlocks!=0))
					facI = (float)(x-baseX)/(xrbfi-baseX);
				if ((x>xrbfo) && (fadeOutBlocks!=0))
					facO = 1.0 - (float) (x - xrbfo)/(baseX + peaksFilled - xrbfo);
				for (int k=0; k<channels; k++)
					{
					yc = chHeight*(k+1)+ 16;
					sample = (char) ( gain * facI * facO * peakBuffer[ppos] );
					if (sample>=127)
						painter->setPen(CM_COLOR(CLIP_PEAK_OVERLOADED_SAMPLE));
					else
						painter->setPen(CM_COLOR(CLIP_PEAK_MACROVIEW));
					dy = (int) (yFactor * (float) sample ) ;
					painter->drawLine( x , baseY + yc, x , baseY + yc - dy );
					ppos++;
					}
				}
			// peaks under right cross fade region ( only right needed )
			if (crossAtRight)
				{
				painter->setPen(Qt::black);
				for (int x=baseX + peaksFilled - rCrossX; x < baseX + peaksFilled; x++)
					{
					float facI = 1.0;
					float facO = 1.0;
					if ((x<xrbfi) && (fadeInBlocks!=0))
						facI = (float)(x-baseX)/(xrbfi-baseX);
					if ((x>xrbfo)  && (fadeOutBlocks!=0))
						facO = (float)(baseX + peaksFilled - x)/rCrossX;
					for (int k=0; k<channels; k++)
						{
						yc = chHeight*(k+1)+ 16;
						sample = (char) ( gain * facI * facO * peakBuffer[ppos] );
						dy = (int) (yFactor * (float) sample ) ;
						painter->drawLine( x , baseY + yc, x , baseY + yc - dy );
						ppos++;
						}
					}
				}
			}
		}
	else
		{
		progress = pk->get_progress();
		painter->setPen(Qt::black);
		QRect r(baseX + clipXWidth/10, get_baseY() + chHeight/2,180,20);
		drawArea->painter->setFont( QFont( "Helvetica", 12 ) );
		QString si; si.setNum(progress);
		QString buildProcess = "Building Peaks: " + si + "%";
		painter->drawText(r, Qt::AlignAuto, buildProcess);
		}
	delete peakBuffer;

	// draw fades in/out and clip gain (shown only in normal editing mode)
	if (parentSong->get_editing_mode()==Song::EDITING_MODE_NORMAL)
		{
		painter->setPen(CM_COLOR(MAGENTA));
		if (fadeInBlocks>0)
			painter->drawLine(baseX, baseY + height, xrbfi, gy);
		if (fadeOutBlocks>0)
			painter->drawLine(baseX + clipXWidth - 1, baseY + height ,xrbfo, gy );
		if (gain<1.0)
			painter->drawLine(xrbfi, gy,   xrbfo , gy );
		}

	// Black Contour of clip Info area
	painter->setPen(CM_COLOR(DARK_TEXT)); // CHANGE TO CLIP_COUNTOUR
	painter->drawRect(baseX, baseY , clipXWidth , 16);
	painter->drawRect(baseX, baseY , clipXWidth , height);
	painter->setPen(Qt::blue);
	if (crossAtLeft)
			painter->drawArc(baseX - lCrossX, baseY + 16, lCrossX * 2 , (height-16)*2 , 90*16, -90*16);
	if (crossAtRight)
			painter->drawArc(baseX + clipXWidth - rCrossX, baseY + 16, rCrossX * 2, (height-16) * 2, 90*16, 90*16);
	return 1;
	}


void AudioClip::set_track_first_block(long long newTrackFirstBlock)
	{
	long long lenght = sourceLastBlock - sourceFirstBlock;
	trackFirstBlock=newTrackFirstBlock;
	trackLastBlock=trackFirstBlock + lenght;
	}


int AudioClip::play()
	{
	return 1;
	}

int AudioClip::split()
	{
	return 1;
	}

void AudioClip::set_muted(bool b)
	{
	isMuted=b;
	}





void AudioClip::set_left_edge(long long block)
	{
	long long olen = sourceLastBlock - sourceFirstBlock;
	long long otfb = trackFirstBlock;
	trackFirstBlock = block;
	long long lenght = olen + otfb - block;
	sourceFirstBlock = sourceLastBlock - lenght;
	}


void AudioClip::set_right_edge(long long block)
	{
	long long lenght = sourceLastBlock - sourceFirstBlock;
	long long olen = lenght;
	long long otlb = trackLastBlock;
	trackLastBlock = block;
	lenght = olen + block - otlb;
	sourceLastBlock = sourceFirstBlock + lenght;
	}


void AudioClip::set_first_source_block(long long block)
	{
	// TODO
	}


void AudioClip::set_last_source_block(long long block)
	{
	sourceLastBlock = block;
	long long lenght = sourceLastBlock - sourceFirstBlock;
	trackLastBlock = trackFirstBlock + lenght;
	}


long long AudioClip::get_lenght()
	{
	return sourceLastBlock - sourceFirstBlock;
	}


void AudioClip::set_blur(bool stat)
	{
	ColorManager::set_blur(ColorManager::CLIP_BG                    , stat );
	ColorManager::set_blur(ColorManager::CLIP_BG_ACTIVE             , stat );
	ColorManager::set_blur(ColorManager::CLIP_BG_SELECTED           , stat );
	ColorManager::set_blur(ColorManager::CLIP_BG_MUTED              , stat );
	ColorManager::set_blur(ColorManager::CLIP_INFO_AREA_BG          , stat );
	ColorManager::set_blur(ColorManager::CLIP_INFO_AREA_BG_ACTIVE   , stat );
	ColorManager::set_blur(ColorManager::CLIP_PEAK_MICROVIEW        , stat );
	ColorManager::set_blur(ColorManager::CLIP_PEAK_MACROVIEW        , stat );
	ColorManager::set_blur(ColorManager::CLIP_PEAK_OVERLOADED_SAMPLE, stat );
	ColorManager::set_blur(ColorManager::DARK_TEXT                  , stat );
	}

int AudioClip::get_baseX()
	{
	return parentSong->block_to_xpos(trackFirstBlock);
	}

int AudioClip::get_baseY()
	{
	return parentTrack->real_baseY() + 3;
	}

int AudioClip::get_width()
	{
	long long blockWidth = sourceLastBlock - sourceFirstBlock;
	int xwidth = (int) ( blockWidth / parentSong->get_zoom_step() );
	return xwidth ;
	}


int AudioClip::get_height()
	{
	return parentTrack->height - 8;
	}


void AudioClip::set_fade_in(long long b)
	{
	fadeInBlocks=b;
	}


void AudioClip::set_fade_out(long long b)
	{
	fadeOutBlocks=b;
	}


float AudioClip::get_fade_factor_for(long long pos) // pos : position in the track
	{
	float fi,fo;
	if (fadeInBlocks==0)
		fi = 1.0;
	else
		{
		if (pos < trackFirstBlock) // that happens in the edge, the current playing pos , when rounded to locate the clip, may lead to a position "considered" inside the clip, but it is not.
			fi = 0.0;
		else if (pos > trackFirstBlock + fadeInBlocks)
			fi = 1.0;
		else
			fi = (float)((double)(pos - trackFirstBlock)/fadeInBlocks);
		}
	if (fadeOutBlocks==0)
		fo = 1.0;
	else
		{
		long long trackLastBlock = trackFirstBlock + sourceLastBlock - sourceFirstBlock;
		if (pos> trackLastBlock)
			fo = 0.0;
		else if ( pos < trackLastBlock - fadeOutBlocks)
			fo = 1.0;
		else
			fo = (float)((double)(trackLastBlock-pos)/fadeOutBlocks);
		}
	// TODO calculate cross-fades-gains
	return fi*fo;
	}


void AudioClip::set_gain(float g)
	{
	g = g < 0.0 ? 0.0 : g;
	g = g > 1.0 ? 1.0 : g;
	gain = g;
	}
// eof


