/**************************************************************************************

	PROTUX - THE FREE PROFESSIONAL AUDIO TOOLS FOR LINUX
	AUTHOR : See AUTHORS file for details

	This software is distributed under the terms of the GNU General Public License
	as specified in the COPYING file.

***************************************************************************************/
#include "CursorManager.hh"
#include "Mixer.hh"


static const int LOCATION_INFO_X = 5;
static const int LOCATION_INFO_Y = 5;
static const int LOCATION_INFO_W = 120;
static const int LOCATION_INFO_H = 25;


CursorManager::CursorManager(Song* pAssocSong)
	{
	PENTERCONS;
	assocSong = pAssocSong; 
	drawArea = assocSong->get_tracks_area();
	floatX=0;
	playX=0;
	workX=0;
	mode=MODE_NOTDEFINED;
	followPlayTimer = new QTimer(this);
	connect( followPlayTimer , SIGNAL(timeout()), this, SLOT(update_play_cursor()));
	workCursorColor = new QColor(150,50,50);
	cursorMap[CURSOR_FLOAT] = new QCursor( QPixmap(RESOURCES_DIR "cursorFloat.xpm") );
	cursorMap[CURSOR_FLOAT_OVER_CLIP] = new QCursor( QPixmap(RESOURCES_DIR "/cursorFloatOverClip.xpm") );
	cursorMap[CURSOR_FLOAT_OVER_TRACK] = new QCursor( QPixmap(RESOURCES_DIR "/cursorFloatOverTrack.xpm") );
	cursorMap[CURSOR_HOLD_UD] = new QCursor( QPixmap(RESOURCES_DIR "/cursorHoldUd.xpm") );
	cursorMap[CURSOR_HOLD_LR] = new QCursor( QPixmap(RESOURCES_DIR "/cursorHoldLr.xpm") );
	cursorMap[CURSOR_HOLD_LRUD] = new QCursor( QPixmap(RESOURCES_DIR "/cursorHoldLrud.xpm") );
	cursorMap[CURSOR_DRAG] = new QCursor( QPixmap(RESOURCES_DIR "/cursorDrag.xpm") );
	cursorMap[CURSOR_SELECT] = new QCursor( QPixmap(RESOURCES_DIR "/cursorSelect.xpm") );
	cursorMap[CURSOR_MAGIC_ZOOM] = new QCursor( QPixmap(RESOURCES_DIR "/cursorMagicZoom.xpm") );
	set_cursor(CURSOR_FLOAT);
	dataBox = assocSong->parentProject->get_parent_interface()->dataBox;
	PEXITCONS;
	}


CursorManager::~CursorManager()
	{
	PENTERDES;
	delete followPlayTimer;
	PEXITDES;
	}


void CursorManager::start()
	{
	PENTER2;
	set_mode(MODE_FLOAT);
	drawArea->setMouseTracking(true);
	isEnabled=false;
	enable();
	PEXIT2;
	}


void CursorManager::set_mode(int pMode, int pOrigX, int pOrigY,
	int pCursorW, int pCursorH,
	int pOffsetX, int pOffsetY)
	{
	mode = pMode;
	xOrig = pOrigX;
	yOrig = pOrigY;
	xOffset = pOffsetX;
	yOffset = pOffsetY;
	cursorW = pCursorW;
	cursorH = pCursorH;
	}


void CursorManager::set_cursor(int cMode)
	{
	if ((cMode >= 0)  && (cMode <= CURSOR_MAGIC_ZOOM))
		{
		assocSong->get_tracks_area()->setCursor( *(cursorMap[cMode]) );
		currentCursorMapIndex=cMode;
		}
	}


int CursorManager::get_current_cursor_map_index()
	{
	return currentCursorMapIndex;
	}


void CursorManager::recreate()
	{
	}


void CursorManager::enable()
	{
	if (!isEnabled)
		{
		isEnabled=true;
		toggle_work_cursor();
		switch (mode)
			{
			case MODE_NONE:
				{
				return;
				}
			case MODE_FLOAT:
				{
				toggle_float_cursor(floatX, drawArea->height());
				break;
				}
			case MODE_DRAG:
				{
				toggle_drag_cursor();
				break;
				}
			case MODE_PLAY:
				{
				toggle_play_cursor(playX, drawArea->height());
				followPlayTimer->start(25); // GlobalProperties::get_int("PLAY_CURSOR_REFRESH_RATE");
				break;
				}
			case MODE_NOTDEFINED:
				{
				break;
				}
			}
		}
	}


void CursorManager::disable()
	{
	if (isEnabled)
		{
		isEnabled=false;

		toggle_work_cursor();
		switch (mode)
			{
			case MODE_FLOAT:
				{
				toggle_float_cursor(floatX, drawArea->height());
				break;
				}
			case MODE_DRAG:
				{
				toggle_drag_cursor();
				break;
				}
			case MODE_PLAY:
				{
				followPlayTimer->stop();
				toggle_play_cursor(playX, drawArea->height());
				break;
				}
			case MODE_NOTDEFINED:
				{
				break;
				}
			}
		}
	}


void CursorManager::update(int x, int y)
	{
	switch (mode)
		{
		case MODE_FLOAT:
			{
			update_float_cursor(x);
			break;
			}
		case MODE_DRAG:
			{
			update_drag_cursor(x,y);
			break;
			}
		case MODE_PLAY:
			{
			update_play_cursor();
			break;
			}
		case MODE_NOTDEFINED:
			{
			break;
			}
		}

	long long bpos = assocSong->xpos_to_block(x);
	if (bpos<0) bpos=0;
	QString smpte = "SMPTE: " + block_to_smpte(bpos,assocSong->rate);
	QString sbpos; sbpos.setNum((double)bpos,'f',0); sbpos = "BLOCK: "+ sbpos;
	QPainter* painter = assocSong->trackPanelsArea->fastPainter;
	painter->fillRect(0, 0, assocSong->get_track_panels_area()->width(), 30,Qt::black); //30 == locatorheight. Get it from Song!
	painter->setPen(  QColor(245,255,240) );
	painter->setFont( QFont( "Helvetica", 7));
	painter->drawText(LOCATION_INFO_X, LOCATION_INFO_Y+7, smpte);
	painter->drawText(LOCATION_INFO_X, LOCATION_INFO_Y+21, sbpos  );
	// dataBox->update( LOCATION_INFO_X, LOCATION_INFO_Y, LOCATION_INFO_W,LOCATION_INFO_H);
	}



inline QString CursorManager::block_to_smpte(long long bpos, int sampleRate)
	{
	float f =  sampleRate / 30;
	float flo = (float) bpos /  f;
	long long framepos = ( long long ) flo;
	int pfr = (int) framepos % 30;
	int pho = (int) (framepos/108000) % 60;
	int pmi = (int) (framepos/1800) % 60;
	int pse = (int) (framepos/30) % 60;
	if (pho<0) pho=0;
	if (pmi<0) pmi=0;
	if (pse<0) pse=0;
	if (pfr<0) pfr=0;
	QString spho; spho.setNum(pho);
	QString spmi; spmi.setNum(pmi); if (pmi<10) spmi.prepend("0");
	QString spse; spse.setNum(pse); if (pse<10) spse.prepend("0");
	QString spfr; spfr.setNum(pfr); if (pfr<10) spfr.prepend("0");
	QString spos = spho + ":" + spmi + ":" + spse + "," + spfr;
	return spos;
	}

int CursorManager::get_orig_x()
	{
	return xOrig;
	}

int CursorManager::get_orig_y()
	{
	return yOrig;
	}


int CursorManager::get_x_offset()
	{
	return xOffset;
	}

int CursorManager::get_y_offset()
	{
	return yOffset;
	}

void CursorManager::set_x_offset(int x)
	{
	xOffset=x;
	}

void CursorManager::set_y_offset(int y)
	{
	yOffset=y;
	}

void CursorManager::set_work_at(int x)
	{
	update_work_cursor(x);
	}

void CursorManager::set_work_at(long long bp)
	{
	update_work_cursor(assocSong->block_to_xpos(bp));
	}

// PROTECTED


void CursorManager::update_float_cursor(int x)
	{
	toggle_float_cursor(floatX, drawArea->height());
	floatX = x;
	toggle_float_cursor(floatX, drawArea->height());
	}


void CursorManager::update_drag_cursor(int x, int y)
	{
	toggle_drag_cursor();
	floatX = x;
	floatY = y;
	toggle_drag_cursor();
	}


void CursorManager::update_work_cursor(int x)
	{
	workX= x;
	}


void CursorManager::update_play_cursor()
	{
	if (assocSong->get_status()==Mixer::STOPPED)
		{
		disable();
		set_mode(CursorManager::MODE_FLOAT);
		enable();
		}
	else
		{
		int fullW = drawArea->width();
		int fullH = drawArea->height();
		int x = assocSong->get_playing_xpos();
		if ( (x >= 0 ) && ( x <= fullW ) )
			{
			toggle_play_cursor(playX, fullH);
			playX = x;
			toggle_play_cursor(playX, fullH);
			}
		else if ( x > fullW )
			{
			//playX = fullW - 1;
			assocSong->scroll_right( fullW);
			assocSong->recreate_tracks(); // LG : I added this because during play, when turning page, the mta wasnt being recreated.
			}
		}
	}


void CursorManager::toggle_play_cursor(int x, int h)
	{
	drawArea->painter->setRasterOp(Qt::XorROP);
	drawArea->painter->setPen(QColor(0,255,0));
	drawArea->painter->drawLine( x  , 0, x  , h);
	drawArea->painter->setRasterOp(Qt::CopyROP);
	drawArea->update(x  - 1, 0, 2,-1);
	}



void CursorManager::toggle_float_cursor(int x, int h)
	{
	drawArea->painter->setRasterOp(Qt::XorROP);
	drawArea->painter->setPen(QColor(250,255,0));
	drawArea->painter->drawLine( x , 0, x, h);
	drawArea->painter->setRasterOp(Qt::CopyROP);
	drawArea->update(x,0,2,-1);
	}


void CursorManager::toggle_drag_cursor()
	{
	QPainter* p = drawArea->painter;
	p->setRasterOp(Qt::XorROP);
	p->setPen(QColor(155,105,155));
	int x0 = floatX - xOffset;
	int x1 = x0 + cursorW;
	int y0 = floatY - yOffset;
	int xc = floatX;
	int fullH = drawArea->height();
	p->drawLine( x0,  y0, x0 + cursorW, y0);
	p->drawLine( x0,  y0 + cursorH , x0 + cursorW, y0 + cursorH);
	p->drawLine( x0, 0, x0, fullH);
	p->drawLine( xc, 0, xc, fullH);
	p->drawLine( x1, 0, x1, fullH);
	p->setRasterOp(Qt::CopyROP);
	drawArea->update(x0, floatY  - yOffset, cursorW + 1, cursorH + 1 );
	drawArea->update(x0 - 1 , 0 , 1, fullH);
	drawArea->update(x1 - 1 , 0 , 1, fullH);
	}


void CursorManager::toggle_work_cursor()
	{
	drawArea->painter->setRasterOp(Qt::XorROP);
	drawArea->painter->setPen(*workCursorColor);
	int h = assocSong->get_floorY();
	int y=0;
	while (y<h-5)
		{
		drawArea->painter->drawLine( workX, y, workX, y+5);
		y+=10;
		}
	drawArea->painter->setRasterOp(Qt::CopyROP);
	drawArea->update(workX-1,0,2,-1);
	}

void CursorManager::set_work_cursor_color(int r, int g, int b)
	{
	workCursorColor->setRgb(r,g,b);
	}

// eof

