/**************************************************************************************

	PROTUX - THE FREE PROFESSIONAL AUDIO TOOLS FOR LINUX
	AUTHOR : See AUTHORS file for details

	This software is distributed under the terms of the GNU General Public License
	as specified in the COPYING file.

***************************************************************************************/

#include <sys/vfs.h>
#include "Interface.hh"
#include "ProjectManager.hh"
#include "ColorManager.hh"
#include <qframe.h>
#include <qdockwindow.h>

static const int TOPPANEL_FIXED_HEIGHT = 100;
static const int MINIMUM_INFO_LCD_WIDTH = 510;
static const int MINIMUM_BUS_MONITOR_WIDTH = 80;
static const int MINIMUM_BUS_MONITOR_HEIGHT = 90;
static const int MINIMUM_FLOATING_BUS_MONITOR_HEIGHT = 120;

Interface::Interface(QWidget *parent) : MustuxInterface( parent )
	{
	PENTERCONS;
	resize(MINIMUM_INTERFACE_WIDTH,MINIMUM_INTERFACE_HEIGHT);
	setMinimumSize(MINIMUM_INTERFACE_WIDTH,MINIMUM_INTERFACE_HEIGHT);
	setIcon(QPixmap (RESOURCES_DIR "/protux.xpm") );

	ColorManager::init();

	//find a better to use init stuff like this
	QString s = GlobalProperties::get("USE_PLUGHW");
	if (s=="YES")
		MustuxAudioDeviceMapper::set_plughw_use(true);
  	else
		MustuxAudioDeviceMapper::set_plughw_use(false);

	MustuxAudioDeviceMapper::probe_busses_valid_modes();
	//
	tuner = (MustuxTuner*) 0;
	defaultLcd = (MustuxLcd*) 0;

	// top level container
	mainVBoxLayout = new QVBoxLayout(this, 6);
	mainVBoxLayout->setResizeMode(QLayout::FreeResize);

	/*QSplitter**/topSplitter = new QWidget/*QSplitter*/(this);
	lay = new QHBoxLayout(topSplitter);
	topSplitter->setFixedHeight(TOPPANEL_FIXED_HEIGHT);

 	// 1st Area (TOP) infoLcd , dataBox and the Docked bus monitor goes into the splitter
	infoLcd = new MustuxLcd(MINIMUM_INFO_LCD_WIDTH, TOPPANEL_FIXED_HEIGHT, topSplitter, "infoLcd");
	infoLcd->setMinimumWidth(MINIMUM_INFO_LCD_WIDTH);
	lay->insertWidget( 0, infoLcd);

	dataBox=new MustuxDrawable(200,TOPPANEL_FIXED_HEIGHT, topSplitter,"dataBox");
	dataBox->set_background_color(160,170,150); // in Project::show_info, there is a hard dependency to this values. please, fix me thru color manager or dataBox->get_background_color
	lay->insertWidget( 1, dataBox, 4);

	busMonitor = new BusMonitor(MINIMUM_BUS_MONITOR_WIDTH,MINIMUM_BUS_MONITOR_HEIGHT,topSplitter,this);
	lay->insertWidget( 2, busMonitor);
	isBusMonitorDocked = true;
	busMonitorWindow = new QWidget( (QWidget*) 0);
	busMonitorWindow->hide();


	mainVBoxLayout->insertWidget( 0, topSplitter );

	// 2nd Area (MIDDLE) the current project
	projectManager = new ProjectManager(this);

	QString sCurrentProject = GlobalProperties::get("CURRENT_PROJECT");
	if ((!sCurrentProject) || (sCurrentProject==""))
		sCurrentProject="Untitled";
	if (projectManager->projectExists(sCurrentProject))
		{
		if (projectManager->load_project(sCurrentProject)<0)
			{
			PWARN("Cannot load project %s. Continuing anyway...", sCurrentProject.ascii());
			}
		}
	else
        	{
		if (projectManager->create_new_project("Untitled", 1)<0)
			{
			PWARN("Cannot load project Untitled. Continuing anyway...");
			}
		}
        if (projectManager->currentProject)
		{
		mainVBoxLayout->insertWidget( 1 , projectManager->currentProject);
		// FOR LATER (ADAPT ME. SPECTROGRAM WILL BE INSIDE BUSMONITOR) projectManager->currentProject->set_output_spectrogram(spectrogram);
		projectManager->currentProject->show();
		}


	// 3rd Area (BOTTOM) : Status stuff
	QWidget* statusContainer = new QWidget(this);
	statusContainer->setFixedHeight(20);
	QHBoxLayout* statusContainerLayout = new QHBoxLayout( statusContainer );
	/* FOR LATER      QLabel* l1 = new QLabel("PILL",statusContainer);
	QToolTip::add( l1,  "This will be the Protux InternaL Language console (Accepting suggestions to change the language name :-)");
	statusContainerLayout->addWidget( l1 );
	QLineEdit* prompt = new QLineEdit(statusContainer);
	statusContainerLayout->addWidget( prompt );
	*/
	ramUsage = new QLabel("",statusContainer);
	QToolTip::add( ramUsage,  "How much RAM protux is using right now");
	ramUsage->setFrameStyle( QFrame::Panel | QFrame::Sunken );
	ramUsage->setLineWidth( 1 );
	ramUsage->setFixedWidth(100);
	statusContainerLayout->addWidget( ramUsage );

	snapStatus = new QLabel("",statusContainer);
	QToolTip::add( snapStatus,  "Snap status : >|< Snap to work cursor  >[]< Snap to clips borders");
	snapStatus->setFrameStyle( QFrame::Panel | QFrame::Sunken );
	snapStatus->setLineWidth( 1 );
	snapStatus->setFixedWidth(90);
	statusContainerLayout->addWidget( snapStatus );

	freeTime = new QLabel("",statusContainer);
	QToolTip::add( freeTime,  "How much record time you still have");
	freeTime->setFrameStyle( QFrame::Panel | QFrame::Sunken );
	freeTime->setLineWidth( 1 );
	freeTime->setFixedWidth(100);
	statusContainerLayout->addWidget( freeTime );

	diskUsage = new QLabel("",statusContainer);
	QToolTip::add( diskUsage,  "How many disk space (Mb) protuxproject dir is using");
	diskUsage->setFrameStyle( QFrame::Panel | QFrame::Sunken );
	diskUsage->setLineWidth( 1 );
	diskUsage->setFixedWidth(100);
	statusContainerLayout->addWidget( diskUsage );

	diskSpace = new QLabel("",statusContainer);
	QToolTip::add( diskSpace,  "How much free space there is on the same HD where protuxprojects dir is");
	diskSpace->setFrameStyle( QFrame::Panel | QFrame::Sunken );
	diskSpace->setLineWidth( 1 );
	diskSpace->setFixedWidth(100);
	statusContainerLayout->addWidget( diskSpace );
	mainVBoxLayout->insertWidget( 2, statusContainer );
	mainVBoxLayout->activate();

        // Other inits
	busMonitor->init();

	//this is the same as "update_properties" exept the call to MustuxAudioDeviceMapper::probe_busses_valid_modes(); and the "plughw" detection is removed
	jmb->set_clear_time(GlobalProperties::get_int("JMB_CLEAR_TIME"));
	jmb->set_hold_sensitiveness(GlobalProperties::get_int("JMB_HOLD_TIMEOUT"));
	jmb->set_double_fact_interval(GlobalProperties::get_int("JMB_DOUBLE_FACT_TIMEOUT"));
	projectManager->currentProject->update_properties();
	//


	helpWindow = new Help(RESOURCES_DIR "/help.text", (QWidget*) 0);
	globalPropertiesDialog = new GlobalPropertiesDialog(this);
	statusBarTimer = new QTimer(this);
 	set_default_lcd(infoLcd);
	infoLcd->roll_message("Welcome to Protux " VERSION "       Press H for Quick Help                 ");
	jmb->init(RESOURCES_DIR "/jmb.map");
	//well, and if we have ((jmb) global) properties, then we have to SET them ! ;-)
	connect( statusBarTimer , SIGNAL(timeout()), this, SLOT(update_system_resources()));
	statusBarTimer->start(1000);
	showMaximized();
	show_splash_screen();
	PEXITCONS;
	}


Interface::~Interface()
	{
	PENTERDES;
	busMonitor->finish();
	statusBarTimer->stop();
	delete busMonitor;
	delete globalPropertiesDialog;
	delete helpWindow;
	delete projectManager;
	PEXITDES;
	}


int Interface::process_jmb_action(MustuxJogMouseBoardMessage* jmbMesg)
	{
	int r=-1;
	int action = jmbMesg->action;
	bool isBeginning = jmbMesg->moment;
	QString actionName = jmbMesg->label;
	int parameter = jmbMesg->parameter;
	int collNumber = jmbMesg->collectedNumber;

	// Interface scope actions
	if (actionName == "QUIT")
		{
		if (projectManager->currentProject)
			projectManager->currentProject->save();
		QApplication::exit();
		}
	else if (actionName == "ABOUT")
		{
		about_protux();
		}
	else if (actionName == "PROJECT MANAGER")
		{
		projectManager->show();
		projectManager->raise();
		r=1;
		}
	else if (actionName == "TOGGLE SNAP ON/OFF")
		{
		Song* so = projectManager->currentProject->get_current_song();
		so->toggle_snap();
		update_system_resources();
		}
	else if (actionName == "LOCK/UNLOCK EVERYTHING")
		{
		if (jmb->is_locked())
			jmb->unlock();
		else
			jmb->lock();
		r=1;
		}

	else if (actionName == "HELP")
		{
		help();
		r=1;
		}

	else if (actionName == "TUNER")
		{
		if (tuner)
			{
			delete tuner;
			tuner = (MustuxTuner*) 0;
			}
		else
			{
			tuner = new MustuxTuner(1,true);
			tuner->init_frontend();
			tuner->show();
			}

		r=1;
		}

	else if (actionName == "GLOBAL PROPERTIES")
		{
		globalPropertiesDialog->show();
		r=1;
		}
	else
		{
		if (projectManager->currentProject)
			{
			// PROJECT SCOPE ACTIONS
			if (actionName == "SAVE CURRENT PROJECT")
				{
				r = projectManager->currentProject->save();
				}
			else if (actionName == "RENDER SONG")
				{
				r = projectManager->currentProject->get_current_song()->render();
				}
			else if (actionName == "USE SONG")
				{
				int sn = (collNumber>=0) ? (sn = collNumber - 1) : 0;
				projectManager->currentProject->set_current_song(sn);
				QString sp; sp.setNum(sn + 1); sp="USING SONG " +sp;
				info(sp);
				}
			else if (actionName == "RENDER SONG")
				{
				r = projectManager->currentProject->get_current_song()->render();
				}
			else
				{
				// SONG SCOPE ACTIONS
				Song* currentSong = projectManager->currentProject->get_current_song();
				r = currentSong->process_jmb_action(jmbMesg);
				}
			}
		else
			{
			info("OPEN OR CREATE A PROJECT FIRST!!");
			r=-1;
			}
		}
	return r;
	}




void Interface::show_splash_screen()
	{
	splashScreen = new SplashScreen(RESOURCES_DIR "/protuxSplash.jpg");
	}


void Interface::hide_splash_screen() // deprecated
	{
	}


void Interface::help()
	{
	PENTER;
	if (helpWindow->isShown())
		{
		helpWindow->hide();
		}
	else
		{
		helpWindow->show();
		}
	PEXIT;
	}

void Interface::quick_help()
	{
	PENTER;
	PEXIT;
	}

void Interface::show_jmb_map()
	{
	PENTER;
	PEXIT;
	}

void Interface::about_protux()
	{
	PENTER;
	QString authors="The Protux Team : Luciano Giordana, Reinhard Amersberger, Tapio Laxstrom, David Bain, Nathan Langford, Fabio dos Santos, Martin Herren, Eduardo Pimenta, Valery Beluntsov, Marcio Enrico, Remon Sijrier, Gustavo Lemos, Alexandre beraldo, and all the people from Free Software world, who made important technologies on which Protux is based ( Gcc, QT, KDE, XFree, Linux, and so on...)                                              ";
	infoLcd->roll_message(authors,80,2);
	show_splash_screen();
	PEXIT;
	}

MustuxLcd* Interface::get_default_lcd()
	{
	return infoLcd;
	}


void Interface::update_properties()
	{
	PENTER;
	jmb->set_clear_time(GlobalProperties::get_int("JMB_CLEAR_TIME"));
	jmb->set_hold_sensitiveness(GlobalProperties::get_int("JMB_HOLD_TIMEOUT"));
	jmb->set_double_fact_interval(GlobalProperties::get_int("JMB_DOUBLE_FACT_TIMEOUT"));
	QString s = GlobalProperties::get("USE_PLUGHW");
	if (s=="YES")
		MustuxAudioDeviceMapper::set_plughw_use(true);
  	else
		MustuxAudioDeviceMapper::set_plughw_use(false);
	MustuxAudioDeviceMapper::probe_busses_valid_modes();
	projectManager->currentProject->update_properties();
	PEXIT;
	}


void Interface::info(QString message)
	{
	if (defaultLcd)
		defaultLcd->print(message,0);
	}

void Interface::info(QString message, int line)
	{
	defaultLcd->print(message,line);
	}

BusMonitor* Interface::get_bus_monitor()
	{
	return busMonitor;
	}


void Interface::set_current_project(Project *proj)
	{
	PENTER2;
	mainVBoxLayout->insertWidget( 1 , projectManager->currentProject);
	mainVBoxLayout->activate();
	// FOR LATER (ADAPT ME. SPECTROGRAM WILL BE INSIDE BUSMONITOR) projectManager->currentProject->set_output_spectrogram(spectrogram);
	projectManager->currentProject->show();
	GlobalProperties::set("CURRENT_PROJECT", (char*) projectManager->currentProject->get_title().ascii());
	GlobalProperties::save();
	PEXIT2;
	}

Project* Interface::get_current_project()
	{
	return projectManager->currentProject;
	}

ProjectManager* Interface::get_project_manager()
	{
	return projectManager;
	}


void Interface::update_system_resources()
	{
	if ((projectManager->currentProject) && (projectManager->currentProject->get_current_song()->is_snap_on()))
		snapStatus->setText("SNAP ON");
	else
		snapStatus->setText("SNAP OFF");
	struct statfs fs;
	statfs("/", &fs);
	float space = (float) fs.f_bfree * fs.f_bsize / 1048576;
	QString s; s.setNum(space,'f',2);
	diskSpace->setText(s+" Mb");

	/*
	int statfs(const char *path, struct statfs *buf);
	DESCRIPTION
statfs  returns  information  about a mounted file system.  path is the
path name of any file within the mounted filesystem.  buf is a  pointer
to a statfs structure defined as follows:

	struct statfs {
		long    f_type;     // type of filesystem (see below)
		long    f_bsize;    // optimal transfer block size
		long    f_blocks;   // total data blocks in file system
		long    f_bfree;    // free blocks in fs
		long    f_bavail;   // free blocks avail to non-superuser
		long    f_files;    // total file nodes in file system
		long    f_ffree;    // free file nodes in fs
		fsid_t  f_fsid;     // file system id
		long    f_namelen;  // maximum length of filenames
		long    f_spare[6]; // spare for later
	};
	*/
	}


void Interface::resizeEvent(QResizeEvent* e)
	{
	projectManager->currentProject->recreate();
	}


void Interface::busmonitor_dock()
	{
	PENTER2;
	busMonitor->reparent(topSplitter, busMonitor->geometry().topLeft(), TRUE);
	//Reset busMonitor maximum height to fit busMonitor in lay
	busMonitor->setMinimumHeight(MINIMUM_BUS_MONITOR_HEIGHT);
	lay->insertWidget( 2, busMonitor);
	busMonitorWindow->hide();
	isBusMonitorDocked = true;
	PEXIT2;
	}


void Interface::busmonitor_undock()
	{
	PENTER2;
	busMonitor->reparent(busMonitorWindow->parentWidget(), busMonitor->geometry().topLeft(), true);
	//Make the floating busMonitor a bit heigher by default
	busMonitor->setMinimumHeight(MINIMUM_FLOATING_BUS_MONITOR_HEIGHT);
	lay->remove(busMonitor);
	isBusMonitorDocked = false;
	PEXIT2;
	}

bool Interface::is_busmonitor_docked()
	{
	return isBusMonitorDocked;
	}
// EOF

