/**************************************************************************************

	PROTUX - THE FREE PROFESSIONAL AUDIO TOOLS FOR LINUX
	AUTHOR : See AUTHORS file for details

	This software is distributed under the terms of the GNU General Public License
	as specified in the COPYING file.

***************************************************************************************/
#include <stdio.h>
#include <math.h>
#include <qwidget.h>
#include <mustux.h>
#include "Peak.hh" 
#include <qtimer.h>

// TODO : Make peak building/saving/loading proces independend of bit-depth. It now only takes care of 16 bits audio ??

/**
*  Usage:
*  2 ways:
*  1. use Peak::build to build/load a peakfile
*  2. use Peak::process_samples_from_buffer(..) to generate a peakfile on the fly. All samples from
*     an audiofile must be passed to this method. Call Peak::init() before using
*     "Peak::process_samples_from_buffer()". Call Peak::build when all samples are processed.
*
*  Internals:
*  Peak has an internal PeakBuildThread who takes care of building a peakfile.
*  Calling Peak::build(...) starts a new PeakBuildThread, if a peakfile exist it will
*  be loaded else the file is (re)scanned to make one. In case of buildOnTheFly, RAMBufferHzoomNine
*  and RAMBufferHzoomTen are used to calculate the values of RAMBuffer.
*
*  The Build proces:
*  The audio file is scanned with a resolution of 8 (zoomStep[4], which means every 8th sample
*  is used), a finer grained scanning of the file seems not to result in a better waveform
*  drawing. These values are stored in tempRAMBuffer. The two first levels (zoomStep[9 and 10])
*  of RAMBuffer are recalculated from tempRAMBuffer. The other levels (zoomStep[11 .. 31]) are
*  calculated by: RAMBuffer[hzoom][pos] = max(RAMBuffer[hzoom-2][pos0],RAMBuffer[hzoom-2][pos1]) which
*  means for each new value 2 old values are proccesed.
*/
const int Peak::MAX_ZOOM_USING_SOURCEFILE = 8;
const int Peak::zoomStep[MAX_ZOOM_LEVELS] = {
						1,    2,    4,     6,     8,    12,    16,    20,
						24,   32,   48,    64,    96,   128,   192,
						256,  384,  512,   768,  1024,  1536,  2048,  3072,
					4096, 6144, 8192, 12288, 16384, 24576, 32768, 65536
					};

PeakBuildThread::PeakBuildThread  (Peak* pPeak)
	{
	peak = pPeak;
	}


void PeakBuildThread::run()
	{
	// THIS GOES TO MUSTUX DEBUGGER SOON : KEEP IT HERE printf("---> Starting PeakBuildThread::run()\n");
	if(peak->threadedBuild() < 1)
		{
		// THIS GOES TO MUSTUX DEBUGGER SOON : WILL BE SOMETHING LIKE PERROR_THREAD
		printf("---> PeakBuildThread: Failed to load/build peak file\n");
		}
	// THIS GOES TO MUSTUX DEBUGGER SOON : KEEP IT HERE printf("---> Leaving PeakBuildThread::run()\n");
	}


Peak::Peak(Song* pParentSong)
	{
	PENTERCONS;
	parentSong = pParentSong;
	buildOnTheFly = false;
	audioSource = (Audio*) 0;
	for (int hzoom = 0; hzoom<MAX_ZOOM_LEVELS; hzoom++)
		{
		RAMBuffer[hzoom] = (char*) 0;
		RAMBufferSize[hzoom]=0;
		}
	PEXITCONS;
	}


Peak::~Peak()
	{
	PENTERDES;
	PEXITDES;
	}

void Peak::init(Audio* anAudioSource)
	{
	audioSource = anAudioSource;
	MustuxAudioFileFormat* file = audioSource->file;
	channels = file->channels;
	RAMBufferHzoomNine = (char*) 0;
	RAMBufferHzoomTen = (char*) 0;
	RAMBufferHzoomNineSize = 0;
	RAMBufferHzoomTenSize = 0;
	posInRAMBufferHzoomNine = 0;
	posInRAMBufferHzoomTen = 0;
	resolutionHzoomNine = 0;
	resolutionHzoomTen = 0;
	buildOnTheFly = true;
	peakValueHzoomNine = (char*) 0;
	peakValueHzoomTen = (char*) 0;
	peakValueHzoomNine = new char[channels];
	peakValueHzoomTen = new char[channels];
	for(int i=0; i<channels; i++)
		{
		peakValueHzoomNine[i] = 0;
		}
	for(int i=0; i<channels; i++)
		{
		peakValueHzoomTen[i] = 0;
		}
	}


int Peak::build(Audio* anAudioSource, QProgressBar *pMonitorProgressBar )
	{
	PENTER;
	audioSource = anAudioSource;
	monitorProgressBar = pMonitorProgressBar;	//FIXME not used anymore??
	peakBuildThread = new PeakBuildThread(this);
	peaksAvailable = false;
	peakBuildThread->start();
	PEXIT;
	return 1;
	}

int Peak::threadedBuild()
	{
	// THIS GOES TO MUSTUX DEBUGGER SOON : KEEP IT HERE printf("---> Starting Peak::ThreadedBuild()\n");
	MustuxAudioFileFormat* file = audioSource->file;
	file->rewind_audio();
	int channels = file->channels;
	int blockSize = file->blockSize;
	int headerSize = file->headerSize;
	long long totalBlocks = file->totalBlocks;

	//important: do not change tempStep into a value other then: 0, 1, 2, 4, 6. The lower this value,
	//the more time is needed for the build process, but it is done more accurately (which doesn't
	//make sense for the waveform-drawing as the monitor-resolution is the limiting factor in this case).
	int tempStep=4;
	totalRAMBufferSize=0;


	for (int hzoom=MAX_ZOOM_USING_SOURCEFILE+1;   hzoom<MAX_ZOOM_LEVELS;  hzoom++)
		{
		long long howManyZoomStepsForThisZoomLevel = totalBlocks / zoomStep[hzoom];
		RAMBufferSize[hzoom] = (howManyZoomStepsForThisZoomLevel + 1) * channels;
		RAMBuffer[hzoom] = new char[RAMBufferSize[hzoom]];
		totalRAMBufferSize += RAMBufferSize[hzoom];
		//  LG: I WILL ADD A SPECIAL MACRO FOR DEBUGGIN THREAD.. LEaVE IT COMMENTED MEANWHILE. printf("---> Peak::ThreadedBuild() : Created RAM buffer for HZOOM = %d of %d shorts \n ", hzoom , RAMBufferSize[hzoom] );
		}

	// try to load a previously saved image before rescan the file...
	parentSong->set_progress_timer(true);
	if (load()>0)
		{
		//  LG: I WILL ADD A SPECIAL MACRO FOR DEBUGGIN THREAD.. LEaVE IT COMMENTED MEANWHILE. printf("---> Peak::ThreadedBuild() : Loaded peaks from previously saved image");
		peaksAvailable = true;
		parentSong->set_progress_timer(false);
		// THIS GOES TO MUSTUX DEBUGGER SOON : KEEP IT HERE printf("---> Leaving Peak::ThreadedBuild()\n");
		return 1;
		}

	QString afn = audioSource->get_pure_filename();
	long long blockPosCounter = 0;
	int ch = 0;
	long howManyZoomSteps = totalBlocks / zoomStep[tempStep];
	long long scanningBlockPos = headerSize;
	long long stepInBlocks = zoomStep[tempStep];
	short sho;
	progress = 0;
	int step;
	int posInTempBuffer;
	char representativeSample;
	char val;
	long long pl;
	//  LG: I WILL ADD A SPECIAL MACRO FOR DEBUGGIN THREAD.. LEaVE IT COMMENTED MEANWHILE. printf("---> Peak::ThreadedBuild() : Rescanning peaks ...\n");
	if(!buildOnTheFly)	//No data in tempRAMBuffer/onTheFlyBuffers yet, rescanning file needed
		{
		tempRAMBuffer = (char*) 0;
		tempRAMBuffer = new char[((howManyZoomSteps + 1) * channels)];
		char* p = tempRAMBuffer;

		//read peak information into tempRAMBuffer.
		do
			{
			// FIX ME ( ogg files and this next block_seek call)
			// this block_seek is positioning the files always in position 0
			// no matter what the scanningBlockPos value is
			// It is strange since in ogg seeking test, it worked fine
			int r = file->block_seek(scanningBlockPos);
			//printf("scanningBlockPos=%d   currentPos=%d\n",channels,scanningBlockPos,(int)file->block_tell());
			for  (ch=0; ch<channels; ch++)
				{
				file->read_fragment((char*)&sho, sizeof(short));
				if (sho < 0) sho *= -1;
				*(p++) = sho / 256;	//AudioClip::draw makes use of 'char' to draw the waveform. MAX_VALUE sho = 65536. dividing it by 256 makes it storable in a char!
				}			//This means RAMBuffer also uses 'char' to store the peakValues.
			blockPosCounter++;
			scanningBlockPos+=stepInBlocks;

			if (blockPosCounter % 50000==0)
				{
				progress = (int)(96 * (blockPosCounter * zoomStep[tempStep]) / totalBlocks);
				if (monitorProgressBar) monitorProgressBar->setProgress(progress);
				}

			}
		while ((blockPosCounter < (totalBlocks/zoomStep[tempStep])) && !file->eof());

		//from here: fill RAMBuffer with (processed) data from tempRAMBuffer
		//process the first 2 zoomlevels > MAX_ZOOM_USING_SOURCEFILE and place the values from tempRAMBuffer into RAMBUffer
		for (int hzoom=MAX_ZOOM_USING_SOURCEFILE+1; hzoom<11; hzoom++)
			{
			blockPosCounter = 0;
			step = 0;
			pl = 0;
			do
				{
				for  (ch=0; ch<channels; ch++)
					{
					representativeSample = 0;
					posInTempBuffer = step;
					for(int i=0; i<(zoomStep[hzoom]/zoomStep[tempStep]); i++)
						{
							val = tempRAMBuffer[posInTempBuffer];
							if (val > representativeSample) representativeSample = val;
							posInTempBuffer += channels;
						}
					RAMBuffer [ hzoom ] [ pl ] = representativeSample;
					pl++;
					step++;
					}
				step -= channels;
				step += ((zoomStep[hzoom] * channels) / zoomStep[tempStep]);
				blockPosCounter++;
				}
			while (blockPosCounter < (totalBlocks / zoomStep[hzoom]));
			progress += 2;
			if (monitorProgressBar) monitorProgressBar->setProgress(progress);
			}
		delete tempRAMBuffer;
		}
	else
		{
		memcpy(RAMBuffer[9], RAMBufferHzoomNine, RAMBufferSize[9]*sizeof(char));
		memcpy(RAMBuffer[10], RAMBufferHzoomTen, RAMBufferSize[10]*sizeof(char));
		}

	int zoomLevel = 9;
	int posInRAMBuffer;
	progress = 100;
	//process the remaining higher zoomlevels
	for (int hzoom=11; hzoom<MAX_ZOOM_LEVELS; hzoom++)
		{
		blockPosCounter = 0;
		step = 0;
		pl = 0;
		do
			{
			for  (ch=0; ch<channels; ch++)
				{
				representativeSample = 0;
				posInRAMBuffer = step;
				for(int i=0; i<2; i++)	//Read text at top of file for an explenation of this value (i<2)
					{
						val = RAMBuffer [zoomLevel] [posInRAMBuffer];
						if (val > representativeSample) representativeSample = val;
						posInRAMBuffer += channels;
					}
				RAMBuffer [ hzoom ] [ pl ] = representativeSample;
				pl++;
				step++;
				}
			step += channels;
			blockPosCounter++;
			}
		while (blockPosCounter < (totalBlocks / zoomStep[hzoom]));
		zoomLevel++;
		}
	if (monitorProgressBar) monitorProgressBar->reset();
	peaksAvailable = true;
	parentSong->set_progress_timer(false);

	if(buildOnTheFly)
		{
		delete RAMBufferHzoomTen;
		delete RAMBufferHzoomNine;
		}


	//  LG: I WILL ADD A SPECIAL MACRO FOR DEBUGGIN THREAD.. LEaVE IT COMMENTED MEANWHILE. printf("---> Peak::ThreadedBuild() : Rescanning Peaks : Completed.\n");
	if (save()<0)
		{
		// THIS GOES TO MUSTUX DEBUGGER SOON : KEEP IT HERE printf("---> Peak::ThreadedBuild() : WARNING : Cannot save Peak image (RAM Buffers) to a peak file\n");
		return -1;
		}
	// THIS GOES TO MUSTUX DEBUGGER SOON : KEEP IT HERE printf("---> Leaving Peak::ThreadedBuild()\n");
	return 1;
	}


int Peak::process_samples_from_buffer(char* buffer, int bufferSize)
	{
	int resizeValueNine = 1000000 * channels;
	int resizeValueTen = 666670 * channels;

	//TODO find the best way to dynamically realocate arrays.
	if ((posInRAMBufferHzoomNine + (int)(bufferSize/32)) > RAMBufferHzoomNineSize)
		{
		RAMBufferHzoomNine = (char*) realloc (RAMBufferHzoomNine, (RAMBufferHzoomNineSize + resizeValueNine*sizeof(char)));
		RAMBufferHzoomNineSize += resizeValueNine;
		RAMBufferHzoomTen = (char*) realloc (RAMBufferHzoomTen, (RAMBufferHzoomTenSize + resizeValueTen*sizeof(char)));
		RAMBufferHzoomTenSize += resizeValueTen;
		PMESG3("Buffer's are resized");
		}
	int i=0;
	short sho;
	char* posBuffer = buffer;
	while(i<bufferSize)
		{
		for(int j=0; j<channels; j++)
			{
			char* p = posBuffer + j*sizeof(short);
			sho = *(short*)p;
			sho /= 256;
			i += sizeof(short);
			if(sho < 0) sho *= -1;
			if(sho > peakValueHzoomNine[j])
				peakValueHzoomNine[j] = sho;
			if(sho > peakValueHzoomTen[j])
				peakValueHzoomTen[j] = sho;
			}
		if(resolutionHzoomNine == 32)
			{
			for(int k=0; k<channels; k++)
				{
				RAMBufferHzoomNine[posInRAMBufferHzoomNine] = peakValueHzoomNine[k];
				posInRAMBufferHzoomNine++;
				peakValueHzoomNine[k] = 0;
				}
			resolutionHzoomNine = 0;
			}
		if(resolutionHzoomTen == 48)
			{
			for(int k=0; k<channels; k++)
				{
				RAMBufferHzoomTen[posInRAMBufferHzoomTen] = peakValueHzoomTen[k];
				posInRAMBufferHzoomTen++;
				peakValueHzoomTen[k] = 0;
				}
			resolutionHzoomTen = 0;
			}
		posBuffer += sizeof(short)*channels;
		resolutionHzoomNine++;
		resolutionHzoomTen++;
		}
	}


int Peak::unbuild()
	{
	PENTER;
	//audioSourceFile->close_file();
	for (int i=0; i<MAX_ZOOM_LEVELS; i++)
		if (RAMBuffer[i])
			delete RAMBuffer[i];
	peaksAvailable = false;
	PEXIT;
	return 1;
	}



int Peak::fill_buffer(char* bufferToFill, long long startBlock, int numBlocksToFill,  int hzoom)
	{
	int blocksFilled;
	if(peaksAvailable)
		{
		MustuxAudioFileFormat* file = audioSource->file;
		int blockSize = file->blockSize;
		int channels = file->channels;
		int headerSize = file->headerSize;
		//PDEBUG("blockSize=%d headerSize=%d",blockSize,headerSize);
		if (hzoom > MAX_ZOOM_USING_SOURCEFILE)
			{
			long long offsetInBytes = (startBlock / zoomStep[hzoom]) * blockSize;
			long long offsetInShorts = offsetInBytes / sizeof(short) ;
			char* pi = RAMBuffer[hzoom] + offsetInShorts;
			char* pf = RAMBuffer[hzoom] + RAMBufferSize[hzoom];
			long long numShortsToFill = numBlocksToFill * channels;
			if (numShortsToFill >= (pf-pi))
				numShortsToFill = pf-pi;
			if (numShortsToFill<0)
				return 0;
			memcpy(bufferToFill, pi, numShortsToFill*sizeof(char));
			blocksFilled = numShortsToFill / channels;
			}
		else
			{
			long long stepInBytes = blockSize * zoomStep[hzoom];
			long long originalPos = file->byte_tell();
			long long numShortsToFill = numBlocksToFill * channels * zoomStep[hzoom];
			short sho;
			char* p = bufferToFill;
			char* buf;
			buf = (char*) malloc (numShortsToFill*sizeof(short));
			file->block_seek(startBlock);
			int r = file->read_fragment(buf,(numShortsToFill*sizeof(short)));

			char* posBuffer = buf;
			blocksFilled = 0;
			do
				{
				for (int ch=0; ch < channels; ch++)
					{
					char* s = posBuffer + ch*sizeof(short);
					*(p++) = *(short*)s / 256;
					}
				blocksFilled++;
				posBuffer += stepInBytes;
				}
			while(blocksFilled < numBlocksToFill);
			delete buf;
			file->byte_seek(originalPos);
			}
		}
	else
		{
		//Hmm, is there a posibility no peak file exist and will not be rebuild automatically? If so, we have to call audioSource->rebuild_peaks(); here
		return -1;
		}
	return blocksFilled;
	}


void Peak::show_info()
	{
	PMESG("Peakfiles information\n");
	//TODO
	}



int Peak::save()
	{
	PENTER;
	QString peakFilename = audioSource->rootDir + "/"+audioSource->get_pure_filename() + ".peak";
	PMESG("Saving Peak to %s", (const char*) peakFilename.ascii());
	size_t size ;
	FILE* file = fopen((const char*)peakFilename.ascii(),"w+");
	char* buf;

	if (!file)
		{
		PERROR("Cannot save Peak image to %s", (const char*) peakFilename.ascii());
		PEXIT;
		return -1;
		}
	//store the values from RAMBuffer in a (temp) buf, and write buf at ones to the hard disk, delete buf afterwards.
	buf = new char[totalRAMBufferSize*sizeof(char)];
	char* p = buf;
	for (int hzoom = MAX_ZOOM_USING_SOURCEFILE+1; hzoom<MAX_ZOOM_LEVELS; hzoom++)
		{
		for (int j=0; j<RAMBufferSize[hzoom]; j++)
			{
			char sho = RAMBuffer[hzoom][j];
			*(p++) = sho;
			}
		}
	fwrite(buf,sizeof(char),totalRAMBufferSize,file);
	PMESG("RAM Buffers saved. Total size of file is %d",totalRAMBufferSize * sizeof(char) );
	fclose(file);
	delete buf;
	PEXIT;
	return 1;
	}




int Peak::load()
	{
	PENTER;
	QString peakFilename = audioSource->rootDir + "/"+ audioSource->get_pure_filename() + ".peak";
	PMESG("Looking for peak image file %s", (const char*) peakFilename.ascii());
	FILE* pfile = fopen((const char*)peakFilename.ascii(),"r+");
	char* buf;
	if (!pfile)
		{
		PEXIT;
		return -1;
		}
	PMESG("Found it ! Loading it into RAM Buffers... ");
	//reading peakfile into (temp) buf, so the file reading can be done in one big slurp.
	buf =  new char[totalRAMBufferSize*sizeof(char)];
	fread(buf,sizeof(char),totalRAMBufferSize,pfile);
	char* p = buf;
	//putting the values in the right manner into RAMBuffer, deleting buf afterwards
	for (int hzoom = MAX_ZOOM_USING_SOURCEFILE+1; hzoom<MAX_ZOOM_LEVELS; hzoom++)
		for (int j=0; j<RAMBufferSize[hzoom]; j++)
			RAMBuffer[hzoom][j] = *(p++);
	PMESG("RAM Buffers sucessfully loaded.");
	fclose(pfile);
	delete buf;
	PEXIT;
	return 1;
	}


int Peak::get_progress()
	{
	return progress;
	}
// eof
