/**************************************************************************************

	PROTUX - THE FREE PROFESSIONAL AUDIO TOOLS FOR LINUX
	AUTHOR : See AUTHORS file for details

	This software is distributed under the terms of the GNU General Public License
	as specified in the COPYING file.

***************************************************************************************/
#include "ProjectManager.hh"

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <pthread.h>
#include <math.h>

#include <qvariant.h>
#include <qheader.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qwhatsthis.h>
#include <qimage.h>
#include <qpixmap.h>

#include <mustux.h>

#include "FileBrowser.hh"
#include "DevelGuardian.hh"
#include "Interface.hh"
#include "Mixer.hh"

static const int MIN_SAMPLE = -32768;
static const int MAX_SAMPLE =  32767;

ProjectManager::ProjectManager( Interface* pParentInterface,  const char* name ) : QDialog( (QWidget*) 0, name)
	{
	PENTERCONS;
	parentInterface = pParentInterface;

	if ( !name )
		setName( "ProjectManager" );
	resize( 780, 500 );
	setMaximumSize( QSize( 1024, 768 ) );
	setCaption( "<PM> Project Manager" );

	mainTabPane = new QTabWidget( this );
	mainTabPane->setGeometry( QRect( 0, 0, 780, 500 ) );
	connect( mainTabPane, SIGNAL( currentChanged(QWidget*) ), this, SLOT(tab_changed()));

	// Album Project tab -------------------------------------------------
	tab = new QWidget( mainTabPane);
	QWidget* privateLayoutWidget = new QWidget( tab );
	privateLayoutWidget->setGeometry( QRect( 10, 300, 710, 24 ) );
	QHBoxLayout* hlayout1 = new QHBoxLayout( privateLayoutWidget, 0, 6);
	lblAlbumTitle = new QLabel( privateLayoutWidget);
	lblAlbumTitle->setMinimumSize( QSize( 65, 0 ) );
	lblAlbumTitle->setText( tr( "CURRENT PROJECT : Title" ) );
	hlayout1->addWidget( lblAlbumTitle );
	fieldAlbumTitle = new QLineEdit( privateLayoutWidget);
	fieldAlbumTitle->setText( tr( "" ) );
	hlayout1->addWidget( fieldAlbumTitle );
	QWidget* privateLayoutWidget_2 = new QWidget( tab);
	privateLayoutWidget_2->setGeometry( QRect( 10, 330, 500, 27 ) );
	QHBoxLayout* hlayout2 = new QHBoxLayout( privateLayoutWidget_2, 0, 6);
	lblNOfSongs = new QLabel( privateLayoutWidget_2);
	lblNOfSongs->setText( tr( "Number of Songs" ) );
	hlayout2->addWidget( lblNOfSongs );
	fieldNOfSongs = new QLineEdit( privateLayoutWidget_2);
	fieldNOfSongs->setMaximumSize(QSize(60,20));
	//fieldNOfSongs->setEnabled( FALSE );
	//fieldNOfSongs->setText( tr( "1" ) );
	hlayout2->addWidget( fieldNOfSongs );
	lblStatus = new QLabel( privateLayoutWidget_2);
	lblStatus->setText( tr( "Status" ) );
	hlayout2->addWidget( lblStatus );
	cmbStatus = new QComboBox( FALSE, privateLayoutWidget_2);
	cmbStatus->insertItem( tr( "Planning" ) );
	cmbStatus->insertItem( tr( "Rehearsals" ) );
	cmbStatus->insertItem( tr( "Recording" ) );
	cmbStatus->insertItem( tr( "Finishing" ) );
	cmbStatus->insertItem( tr( "Tournee" ) );
	cmbStatus->setMaximumSize( QSize( 140, 28 ) );
	hlayout2->addWidget( cmbStatus );
	lblTotalDur = new QLabel( privateLayoutWidget_2);
	lblTotalDur->setText( tr( "Dur: ???" ) );
	hlayout2->addWidget( lblTotalDur );
	QWidget* privateLayoutWidget_3 = new QWidget( tab);
	privateLayoutWidget_3->setGeometry( QRect( 10, 360, 710, 24 ) );
	QHBoxLayout* hlayout3 = new QHBoxLayout( privateLayoutWidget_3, 0, 6);
	lblEngineer = new QLabel( privateLayoutWidget_3);
	lblEngineer->setMinimumSize( QSize( 95, 0 ) );
	lblEngineer->setText( tr( "Protux Engineer" ) );
	hlayout3->addWidget( lblEngineer );
	fieldEngineer = new QLineEdit( privateLayoutWidget_3);
	hlayout3->addWidget( fieldEngineer );
	QWidget* privateLayoutWidget_4 = new QWidget( tab );
	privateLayoutWidget_4->setGeometry( QRect( 10, 420, 710, 24 ) );
	QHBoxLayout* hlayout4 = new QHBoxLayout( privateLayoutWidget_4, 0, 6);
	btnRender = new QPushButton( privateLayoutWidget_4);
	btnRender->setText( tr( "Render ..." ) );
	connect( btnRender, SIGNAL(clicked()), this, SLOT(btnRender_clicked()));
	//QToolTip::add( btnRender, tr( "Render all underendered elements. Do this before burning a CD" ) );
	hlayout4->addWidget( btnRender );
	btnMaster = new QPushButton( privateLayoutWidget_4);
	btnMaster->setEnabled( FALSE );
	btnMaster->setText( tr( "Master ..." ) );
	connect( btnMaster, SIGNAL(clicked()), this, SLOT(btnMaster_clicked()));
	//QToolTip::add( btnMaster, tr( "Opens the masterization dialog" ) );
	hlayout4->addWidget( btnMaster );
	btnBurnCD = new QPushButton( privateLayoutWidget_4);
	btnBurnCD->setEnabled( FALSE );
	connect( btnBurnCD, SIGNAL(clicked()), this, SLOT(btnBurnCD_clicked()));
	//QFont btnBurnCD_font(  btnBurnCD->font() );
	//btnBurnCD_font.setFamily( "Helvetica" );
	//btnBurnCD_font.setPointSize( 10 );
	//btnBurnCD_font.setBold( TRUE );
	//btnBurnCD->setFont( btnBurnCD_font );
	btnBurnCD->setText( tr( "Burn CD" ) );
	hlayout4->addWidget( btnBurnCD );
	QWidget* privateLayoutWidget_5 = new QWidget( tab );
	privateLayoutWidget_5->setGeometry( QRect( 10, 390, 710, 24 ) );
	QHBoxLayout* hlayout5 = new QHBoxLayout( privateLayoutWidget_5, 0, 6);
	btnCreateProject = new QPushButton( privateLayoutWidget_5 );
	btnCreateProject->setText( tr( "CREATE NEW" ) );
	hlayout5->addWidget( btnCreateProject );
	connect( btnCreateProject, SIGNAL(clicked()), this, SLOT(btnCreateProject_clicked()));
	btnLoadProject = new QPushButton( privateLayoutWidget_5 );
	btnLoadProject->setText( tr( "LOAD" ) );
	connect( btnLoadProject, SIGNAL(clicked()), this, SLOT(btnLoadProject_clicked()));
	//QToolTip::add( btnSaveProject, tr( "Load this album project (Save your current project first !!!)" ) );
	hlayout5->addWidget( btnLoadProject );
	btnSaveProject = new QPushButton( privateLayoutWidget_5 );
	btnSaveProject->setText( tr( "SAVE" ) );
	connect( btnSaveProject, SIGNAL(clicked()), this, SLOT(btnSaveProject_clicked()));
	//QToolTip::add( btnSaveProject, tr( "Save all Song projects under this album project" ) );
	hlayout5->addWidget( btnSaveProject );
	btnCopyToProject = new QPushButton( privateLayoutWidget_5 );
	btnCopyToProject->setText( tr( "Save as" ) );
	connect( btnCopyToProject, SIGNAL(clicked()), this, SLOT(btnCopyToProject_clicked()));
	//QToolTip::add( btnCopyToProject, tr( "Copy this project to another project" ) );
	hlayout5->addWidget( btnCopyToProject );
	fieldSaveProjAs = new QLineEdit( privateLayoutWidget_5 );
	hlayout5->addWidget( fieldSaveProjAs );
	btnDeleteProject = new QPushButton( privateLayoutWidget_5 );
	connect( btnDeleteProject, SIGNAL(clicked()), this, SLOT(btnDeleteProject_clicked()));
	btnDeleteProject->setText( tr( "DELETE" ) );
	//QToolTip::add( btnDeleteProject, tr( "Delete current project (DANGER)" ) );
	hlayout5->addWidget( btnDeleteProject );
	listAlbumProjects = new QListView( tab );
	listAlbumProjects->addColumn( tr( "Album Name" ) );
	listAlbumProjects->addColumn( tr( "N. of Songs" ) );
	listAlbumProjects->addColumn( tr( "Space Alloc" ) );
	listAlbumProjects->addColumn( tr( "Status" ) );
	listAlbumProjects->setColumnAlignment( 1, Qt::AlignHCenter );
	listAlbumProjects->setColumnAlignment( 3, Qt::AlignHCenter );
	listAlbumProjects->setGeometry( QRect( 10, 10, 710, 280 ) );
	connect(listAlbumProjects, SIGNAL(selectionChanged()), this, SLOT(listAlbumProject_projectSelected()));
	mainTabPane->insertTab( tab, tr( "Album Projects" ) );


	// Songs tab ----------------------------------------------------------
	tab_2 = new QWidget( mainTabPane );
	QWidget* privateLayoutWidget_6 = new QWidget( tab_2);
	privateLayoutWidget_6->setGeometry( QRect( 10, 430, 710, 22 ) );
	QHBoxLayout* hlayout6 = new QHBoxLayout( privateLayoutWidget_6, 0, 6);
	lblMasterGain = new QLabel( privateLayoutWidget_6 );
	lblMasterGain->setMinimumSize( QSize( 69, 0 ) );
	lblMasterGain->setText( tr( "Master Gain" ) );
	hlayout6->addWidget( lblMasterGain );
	fieldMasterGain = new QLineEdit( privateLayoutWidget_6 );
	fieldMasterGain->setText( tr( "0.0" ) );
	hlayout6->addWidget( fieldMasterGain );
	lblNumTracks = new QLabel( privateLayoutWidget_6 );
	lblNumTracks->setMinimumSize( QSize( 70, 0 ) );
	lblNumTracks->setText( tr( "Num. Tracks" ) );
	hlayout6->addWidget( lblNumTracks );
	fieldNumTracks = new QLineEdit( privateLayoutWidget_6);
	fieldNumTracks->setText( tr( "" ) );
	fieldNumTracks->setEnabled(false);
	//QToolTip::add( fieldNumTracks, tr( "To add/remove tracks, use standard JMB actions , such <INSERT> or <DELETE>, right inside MTA" ) );
	hlayout6->addWidget( fieldNumTracks );
	lblSongDur = new QLabel( privateLayoutWidget_6);
	lblSongDur->setMinimumSize( QSize( 56, 0 ) );
	lblSongDur->setText( tr( "Duration" ) );
	lblSongDur->setScaledContents( FALSE );
	hlayout6->addWidget( lblSongDur );
	fieldSongDur = new QLineEdit( privateLayoutWidget_6 );
	fieldSongDur->setText( tr( "??" ) );
	fieldNumTracks->setEnabled(false);
	hlayout6->addWidget( fieldSongDur );
	QWidget* privateLayoutWidget_7 = new QWidget( tab_2);
	privateLayoutWidget_7->setGeometry( QRect( 10, 400, 710, 24 ) );
	QHBoxLayout* hlayout7 = new QHBoxLayout( privateLayoutWidget_7, 0, 6);
	lblArtist = new QLabel( privateLayoutWidget_7);
	lblArtist->setMinimumSize( QSize( 60, 0 ) );
	lblArtist->setText( tr( "Artists" ) );
	hlayout7->addWidget( lblArtist );
	fieldArtists = new QLineEdit( privateLayoutWidget_7);
	hlayout7->addWidget( fieldArtists );
	QWidget* privateLayoutWidget_8 = new QWidget( tab_2 );
	privateLayoutWidget_8->setGeometry( QRect( 10, 370, 710, 22 ) );
	QHBoxLayout* hlayout8 = new QHBoxLayout( privateLayoutWidget_8, 0, 6);
	TextLabel2 = new QLabel( privateLayoutWidget_8 );
	//TextLabel2->setText( tr( "Song : Au revoir" ) );
	hlayout8->addWidget( TextLabel2 );

	QWidget* privateLayoutWidget_9 = new QWidget( tab_2 );
	privateLayoutWidget_9->setGeometry( QRect( 10, 320, 710, 24 ) );
	QHBoxLayout* hlayout9 = new QHBoxLayout( privateLayoutWidget_9, 0, 6);

	btnSaveSong = new QPushButton( privateLayoutWidget_9 );
	btnSaveSong->setMinimumSize( QSize( 80, 0 ) );
	btnSaveSong->setText( tr( "SAVE" ) );
	//QToolTip::add( btnSaveSong, tr( "Save this song project" ) );
	hlayout9->addWidget( btnSaveSong );

	btnSaveSongAs = new QPushButton( privateLayoutWidget_9);
	btnSaveSongAs->setMinimumSize( QSize( 80, 0 ) );
	btnSaveSongAs->setText( tr( "Rename" ) );
	//QToolTip::add( btnSaveSongAs, tr( "Add a new song which is a copy of current song" ) );
	hlayout9->addWidget( btnSaveSongAs );
	lblOr = new QLabel( privateLayoutWidget_9 );
	lblOr->setText( tr( "or" ) );
	hlayout9->addWidget( lblOr );
	btn = new QPushButton( privateLayoutWidget_9 );
	btn->setMinimumSize( QSize( 80, 0 ) );
	btn->setText( tr( "COPY as" ) );
	//QToolTip::add( btn, tr( "add an empty song project" ) );
	hlayout9->addWidget( btn );
	fieldCopyTo = new QLineEdit( privateLayoutWidget_9 );
	fieldCopyTo->setMinimumSize( QSize( 410, 0 ) );
	fieldCopyTo->setMaximumSize( QSize( 300, 30 ) );
	hlayout9->addWidget( fieldCopyTo );
	listSongs = new QListView( tab_2, "listSongs" );
	listSongs->addColumn( tr( "Song Name" ) );
	listSongs->addColumn( tr( "Tracks" ) );
	listSongs->addColumn( tr( "Lenght h:m:s,fr" ), 110 );
	listSongs->addColumn( tr( "Rate" ), 65 );
	listSongs->addColumn( tr( "Bit Depth" ) );
	listSongs->addColumn( tr( "Status" ), 85 );
	listSongs->addColumn( tr( "Space Alloc" ) );
	listSongs->setColumnAlignment( 1, Qt::AlignHCenter );
	listSongs->setColumnAlignment( 2, Qt::AlignHCenter );
	listSongs->setColumnAlignment( 3, Qt::AlignHCenter );
	listSongs->setColumnAlignment( 4, Qt::AlignHCenter );
	listSongs->setColumnAlignment( 5, Qt::AlignHCenter );
	listSongs->setColumnAlignment( 6, Qt::AlignHCenter );
	QListViewItem * item = new QListViewItem( listSongs, 0 );
	item->setText( 0, tr( "Untitled Song" ) );
	listSongs->setGeometry( QRect( 10, 10, 710, 310 ) );
	QWidget* privateLayoutWidget_91 = new QWidget( tab_2 );
	privateLayoutWidget_91->setGeometry( QRect( 10, 350, 710, 30 ) );
	QHBoxLayout* hlayout10 = new QHBoxLayout( privateLayoutWidget_91, 0, 6);

	btnNewSong = new QPushButton( privateLayoutWidget_91);
	btnNewSong->setMinimumSize( QSize( 80, 25 ) );
	btnNewSong->setMaximumSize( QSize( 120, 25 ) );
	btnNewSong->setText( tr( "NEW" ) );
	//QToolTip::add( btnNewSong, tr( "Create a New Song" ) );
	hlayout10->addWidget( btnNewSong );

	btnRemoveSongFromProject = new QPushButton( privateLayoutWidget_91 );
	btnRemoveSongFromProject->setMinimumSize( QSize( 80, 0 ) );
	btnRemoveSongFromProject->setText( tr( "Remove Song from Project" ) );
	//QToolTip::add( btnSaveSong, tr( "Save this song project" ) );
	hlayout10->addWidget( btnRemoveSongFromProject );

	btnDeleteSong = new QPushButton( privateLayoutWidget_91 );
	btnDeleteSong->setMinimumSize( QSize( 80, 0 ) );
	btnDeleteSong->setText( tr( "Delete Song" ) );
	//QToolTip::add( btnSaveSong, tr( "Save this song project" ) );
	hlayout10->addWidget( btnDeleteSong );

	cmbSongStatus = new QComboBox( FALSE, privateLayoutWidget_91 );
	cmbSongStatus->insertItem( tr( "Planning" ) );
	cmbSongStatus->insertItem( tr( "Rehearsals" ) );
	cmbSongStatus->insertItem( tr( "Recording" ) );
	cmbSongStatus->insertItem( tr( "Finishing" ) );
	cmbSongStatus->insertItem( tr( "Tournee" ) );
	cmbSongStatus->setMaximumSize( QSize( 140, 28 ) );
	hlayout10->addWidget( cmbSongStatus );
	mainTabPane->insertTab( tab_2, tr( "Songs" ) );

	connect(listSongs, SIGNAL(selectionChanged()), this, SLOT(listSongs_songSelected()));
	connect(btnSaveSong, SIGNAL(clicked()), this, SLOT(btnSaveSong_clicked()));
	connect( btnSaveSongAs, SIGNAL(clicked()), this, SLOT(btnSaveSongAs_clicked()));
	connect( btn, SIGNAL(clicked()), this, SLOT(btnSaveSongCopyAs_clicked()));
	connect( btnNewSong, SIGNAL(clicked()), this, SLOT(btnNewSong_clicked()));
	connect(btnRemoveSongFromProject, SIGNAL(clicked()), this, SLOT(btnRemoveSongFromProject_clicked()));
	connect(btnDeleteSong, SIGNAL(clicked()), this, SLOT(btnDeleteSong_clicked()));

	// Audio Sources tab -----------------------------------------------
	tab_3 = new QWidget( mainTabPane );
	QVBoxLayout* audioSourcesViewerLayout = new QVBoxLayout( tab_3, 11, 6, "audioSourcesViewerLayout");

	audioListView = new QListView( tab_3 );
	audioSourcesViewerLayout->addWidget( audioListView );
	audioListView->addColumn( tr( "Name" ) );
	audioListView->addColumn( tr( "Type" ) );
	audioListView->addColumn( tr( "Lenght" ) );
	audioListView->addColumn( tr( "Size" ) );
	audioListView->addColumn( tr( "Clips" ) );
	audioListView->addColumn( tr( "Filters" ) );
	audioListView->addColumn( tr( "Rate" ) );
	audioListView->addColumn( tr( "Channels" ) );
	audioListView->addColumn( tr( "Bit Depth" ) );
	audioListView->addColumn( tr( "Project" ) );
	audioListView->addColumn( tr( "Group" ) );
	audioListView->addColumn( tr( "Hardware" ) );
	audioListView->addColumn( tr( "Description" ) );

	QWidget* audioSourcesButtonsContainer = new QWidget( tab_3 ); //        audioSourcesButtonsContainer->setGeometry( QRect( 10, 310, 700, 24 ) );
	audioSourcesViewerLayout->addWidget( audioSourcesButtonsContainer );
	QHBoxLayout* hlayout11 = new QHBoxLayout( audioSourcesButtonsContainer, 0, 6);

	btnRemoveUnusedSources = new QPushButton( audioSourcesButtonsContainer);
	btnRemoveUnusedSources->setText( tr( "Remove Unused Audio Sources" ) );
	hlayout11->addWidget( btnRemoveUnusedSources );

	btnDeleteSource = new QPushButton( audioSourcesButtonsContainer);
	btnDeleteSource->setText( tr( "Remove Audio Source" ) );
	hlayout11->addWidget( btnDeleteSource );

	btnRemoveAllClips = new QPushButton( audioSourcesButtonsContainer);
	btnRemoveAllClips->setText( tr( "Remove All Audio Sources" ) );
	hlayout11->addWidget( btnRemoveAllClips );

	btnExportSource = new QPushButton( audioSourcesButtonsContainer);
	btnExportSource->setText( tr( "Export ..." ) );
	hlayout11->addWidget( btnExportSource );

	grpPreListenSource = new QGroupBox( tab_3);
	audioSourcesViewerLayout->addWidget( grpPreListenSource );
	grpPreListenSource->setGeometry( QRect( 10, 340, 700, 110 ) );
	grpPreListenSource->setTitle( tr( "Listen" ) );

	lblDSPVol1 = new QLabel( grpPreListenSource );
	lblDSPVol1->setText( tr( "Volume" ) );
	lblDSPVol1->setGeometry( QRect( 150, 15, 80, 15) );

	sliderSource = new QSlider( grpPreListenSource );
	sliderSource->setRange( -72 , 72 );
	sliderSource->setValue( 0 );
	sliderSource->setMaximumSize( QSize( 180, 14 ) );
	sliderSource->setOrientation( QSlider::Horizontal );
	sliderSource->setGeometry( QRect( 85, 35, 190, 14 ) );

	volumeSource = new QLabel( grpPreListenSource );
	volumeSource->setGeometry( QRect( 87, 60, 180, 15 ) );
	volumeSource->setMaximumSize ( QSize( 180 , 15 ) );
	volumeSource->setText( tr( "-24                0               +24" ) );

	btnSourcePlay = new QPushButton( grpPreListenSource);
	btnSourcePlay->setGeometry( QRect( 10, 28, 65, 27 ) );
	btnSourcePlay->setText( tr( "Play" ) );


	cmbPreListenBus = new QComboBox( FALSE, grpPreListenSource);
	cmbPreListenBus->setGeometry( QRect( 270, 20, 100, 30 ) );
	// TODO , fill with actual detected buses
	cmbPreListenBus->insertItem( tr( "AUTO" ) );
	cmbPreListenBus->insertItem( tr( "hw:0,0" ) );
	cmbPreListenBus->insertItem( tr( "hw:1,0" ) );
	cmbPreListenBus->insertItem( tr( "hw:2,0" ) );
	cmbPreListenBus->insertItem( tr( "hw:3,0" ) );
	cmbPreListenBus->insertItem( tr( "hw:4,0" ) );

	fieldAudioSourceInfo = new QLineEdit( grpPreListenSource );
	fieldAudioSourceInfo->setGeometry( QRect( 400, 20, 150, 70 ) );
	lblSourceSpectro = new QLabel( grpPreListenSource );
	lblSourceSpectro->setGeometry( QRect( 600, 40, 80, 15 ) );
	lblSourceSpectro->setMaximumSize( QSize( 300, 15 ) );
	lblSourceSpectro->setText( tr( "FFT & Peak" ) );

	QWidget* privateLayoutWidget_12 = new QWidget( grpPreListenSource );
	privateLayoutWidget_12->setGeometry( QRect( 10, 80, 260, 18 ) );
	QHBoxLayout* hlayout13 = new QHBoxLayout( privateLayoutWidget_12, 0, 6);

	checkSourceInfo = new QCheckBox( privateLayoutWidget_12 );
	checkSourceInfo->setEnabled( TRUE );
	checkSourceInfo->setText( tr( "Auto Info" ) );
	hlayout13->addWidget( checkSourceInfo );

	checkSourceSpectro = new QCheckBox( privateLayoutWidget_12 );
	checkSourceSpectro->setText( tr( "AutoSpectro" ) );
	hlayout13->addWidget( checkSourceSpectro );


	// signals and slots connections
	connect( audioListView, SIGNAL( itemRenamed(QListViewItem*,int) ), this, SLOT( audioListView_itemRenamed(QListViewItem*,int) ) );
	connect( audioListView, SIGNAL( spacePressed(QListViewItem*) ), this, SLOT( audioListView_spacePressed(QListViewItem*) ) );
	connect( audioListView, SIGNAL( returnPressed(QListViewItem*) ), this, SLOT( audioListView_returnPressed(QListViewItem*) ) );
	connect( audioListView, SIGNAL( onItem(QListViewItem*) ), this, SLOT( audioListView_onItem(QListViewItem*) ) );
	connect( audioListView, SIGNAL( collapsed(QListViewItem*) ), this, SLOT( audioListView_collapsed(QListViewItem*) ) );
	connect( audioListView, SIGNAL( expanded(QListViewItem*) ), this, SLOT( audioListView_expanded(QListViewItem*) ) );
	connect( btnDeleteSource, SIGNAL(clicked()), this, SLOT(delete_source()));
	connect( btnRemoveUnusedSources, SIGNAL(clicked()), this, SLOT(btnRemoveUnusedSources_clicked()));
	connect( btnRemoveAllClips, SIGNAL(clicked()), this, SLOT(btnRemoveAllClips_clicked()));
	connect( btnExportSource, SIGNAL(clicked()), this, SLOT(btnExportSource_clicked()));
	connect( sliderSource, SIGNAL(sliderMoved(int)), this, SLOT(sliderSource_moved()));
	connect( btnSourcePlay, SIGNAL(clicked()), this, SLOT(btnSourcePlay_clicked()));

	mainTabPane->insertTab( tab_3, tr( "Audio Sources" ) );


	// File Browser ------------------------------------------------------
	tab_4 = new QWidget( mainTabPane);
	fileBrowser = new FileBrowser(tab_4);
	fileBrowser->setMinimumSize(QSize(710,260));
	grpPreListenFile = new QGroupBox( tab_4 );
	grpPreListenFile->setGeometry( QRect( 10, 340, 700, 110 ) );
	grpPreListenFile->setTitle( tr( "Pre-listen" ) );
	//QWidget* privateLayoutWidget_13 = new QWidget( grpPreListenFile);
	//privateLayoutWidget_13->setGeometry( QRect( 70, 20, 190, 30 ) );
	//QHBoxLayout* hlayout14 = new QHBoxLayout( privateLayoutWidget_13, 1, 0);
	lblDSPVol2 = new QLabel( grpPreListenFile );
	lblDSPVol2->setText( tr( "Volume" ) );
	lblDSPVol2->setGeometry( QRect( 150, 15, 80, 15) );
	//hlayout14->addWidget( lblDSPVol2 );
	sliderFile = new QSlider( grpPreListenFile );
	sliderFile->setRange( -72 , 72 );
	sliderFile->setValue( 0 );
	sliderFile->setMaximumSize( QSize( 180, 14 ) );
	sliderFile->setOrientation( QSlider::Horizontal );
	sliderFile->setGeometry( QRect( 85, 35, 190, 14 ) );
	//hlayout14->addWidget( sliderFile );
	volumeFile = new QLabel( grpPreListenFile );
	volumeFile->setGeometry( QRect( 87, 53, 180, 15 ) );
	volumeFile->setMaximumSize ( QSize( 180 , 15 ) );
	volumeFile->setText( tr( "-24                0               +24" ) );

	QWidget* privateLayoutWidget_14 = new QWidget( grpPreListenFile );
	privateLayoutWidget_14->setGeometry( QRect( 10, 80, 260, 18 ) );
	QHBoxLayout* hlayout15 = new QHBoxLayout( privateLayoutWidget_14, 0, 6);

	checkFileInfo = new QCheckBox( privateLayoutWidget_14 );
	checkFileInfo->setEnabled( TRUE );
	checkFileInfo->setText( tr( "Auto Info" ) );
	hlayout15->addWidget( checkFileInfo );

	checkFileSpectro = new QCheckBox( privateLayoutWidget_14 );
	checkFileSpectro->setText( tr( "AutoSpectro" ) );
	hlayout15->addWidget( checkFileSpectro );

	lblFileSpectro = new QLabel( grpPreListenFile );
	lblFileSpectro->setGeometry( QRect( 570, 60, 80, 15 ) );
	lblFileSpectro->setMaximumSize( QSize( 300, 15 ) );
	lblFileSpectro->setText( tr( "FFT & Peak" ) );

	fieldFileInfo = new QLineEdit( grpPreListenFile );
	fieldFileInfo->setGeometry( QRect( 320, 20, 220, 70 ) );

	btnFilePlay = new QPushButton( grpPreListenFile );
	btnFilePlay->setGeometry( QRect( 10, 28, 65, 27 ) );
	btnFilePlay->setText( tr( "Play" ) );

	grpImportFile = new QGroupBox( tab_4 );
	grpImportFile->setGeometry( QRect( 10, 260, 700, 80 ) );
	grpImportFile->setTitle( tr( "Import" ) );

	QWidget* privateLayoutWidget_15 = new QWidget( grpImportFile );
	privateLayoutWidget_15->setGeometry( QRect( 20, 20, 670, 24 ) );
	QHBoxLayout* hlayout16 = new QHBoxLayout( privateLayoutWidget_15, 0, 6);

	lblFileDestSong = new QLabel( privateLayoutWidget_15 );
	lblFileDestSong->setMaximumSize( QSize( 80, 30 ) );
	lblFileDestSong->setText( tr( "Dest: Song" ) );
	hlayout16->addWidget( lblFileDestSong );

	fieldFileDestSong = new QLineEdit( privateLayoutWidget_15 );
	fieldFileDestSong->setMaximumSize( QSize( 25, 30 ) );
	fieldFileDestSong->setText( tr( "1" ) );
	hlayout16->addWidget( fieldFileDestSong );

	lblFileDestTrack = new QLabel( privateLayoutWidget_15 );
	lblFileDestTrack->setMaximumSize( QSize( 36, 30 ) );
	lblFileDestTrack->setText( tr( "Track" ) );
	hlayout16->addWidget( lblFileDestTrack );

	fieldFileDestTrack = new QLineEdit( privateLayoutWidget_15 );
	fieldFileDestTrack->setMaximumSize( QSize( 25, 30 ) );
	fieldFileDestTrack->setText( tr( "1" ) );
	hlayout16->addWidget( fieldFileDestTrack );

	lblFileDestPos = new QLabel( privateLayoutWidget_15 );
	lblFileDestPos->setMaximumSize( QSize( 25, 30 ) );
	lblFileDestPos->setText( tr( "Pos" ) );
	hlayout16->addWidget( lblFileDestPos );

	fieldFileDestPos = new QLineEdit( privateLayoutWidget_15 );
	fieldFileDestPos->setMinimumSize( QSize( 77, 0 ) );
	fieldFileDestPos->setMaximumSize( QSize( 55, 30 ) );
	fieldFileDestPos->setText( tr( "0:02:30,14" ) );
	hlayout16->addWidget( fieldFileDestPos );

	btnAddFileAsSource = new QPushButton( privateLayoutWidget_15 );
	btnAddFileAsSource->setText( tr( "Add as Audio Source" ) );
	//QToolTip::add( btnAddFileAsSource, tr( "Add this file/cd-track to song project as an audio source" ) );
	hlayout16->addWidget( btnAddFileAsSource );

	btnImportFileNow = new QPushButton( privateLayoutWidget_15 );
	btnImportFileNow->setText( tr( "Import now!" ) );
	//QToolTip::add( btnImportFileNow, tr( "Add this file/cd-track to song project as an audio source" ) );
	hlayout16->addWidget( btnImportFileNow );

	QWidget* privateLayoutWidget_16 = new QWidget( grpImportFile );
	privateLayoutWidget_16->setGeometry( QRect( 20, 50, 670, 22 ) );
	QHBoxLayout* hlayout17 = new QHBoxLayout( privateLayoutWidget_16, 0, 6);

	pbarFileImport = new QProgressBar( privateLayoutWidget_16, "pbarFileImport" );
	pbarFileImport->setProgress( 0 );
	hlayout17->addWidget( pbarFileImport );

	checkFileFakeImport = new QCheckBox( privateLayoutWidget_16 );
	checkFileFakeImport->setText( tr( "Fake Import" ) );
	//QToolTip::add( checkFileFakeImport, tr( "Just import the audio source, spending the same amout of time, but does not put it into the song as audio source" ) );
	hlayout17->addWidget( checkFileFakeImport );
	mainTabPane->insertTab( tab_4, "<FB> File Broswer" );

	// signals and slots connections
	connect( sliderFile, SIGNAL(sliderMoved(int)), this, SLOT(sliderFile_moved()));
	connect( btnFilePlay, SIGNAL(clicked()), this, SLOT(btnFilePlay_clicked()));
	connect( btnAddFileAsSource, SIGNAL(clicked()), this, SLOT(btnAddFileAsSource_clicked()));
	connect( btnImportFileNow, SIGNAL(clicked()), this, SLOT(btnImportNow_clicked()));


	// CD BROWSER ---------------------------------------------------------
	tab_5 = new QWidget( mainTabPane, "tab_5" );
	lblCDBrowser_2 = new QLabel( tab_5 );
	lblCDBrowser_2->setGeometry( QRect( 290, 160, 120, 20 ) );
	lblCDBrowser_2->setText( tr( "Audio CD Browser" ) );
	grpImportCDA = new QGroupBox( tab_5 );
	grpImportCDA->setGeometry( QRect( 10, 260, 700, 80 ) );
	grpImportCDA->setTitle( tr( "Import" ) );
	QWidget* privateLayoutWidget_17 = new QWidget( grpImportCDA );
	privateLayoutWidget_17->setGeometry( QRect( 20, 20, 670, 24 ) );
	QHBoxLayout* hlayout18 = new QHBoxLayout( privateLayoutWidget_17, 0, 6);
	lblDestSongCDA = new QLabel( privateLayoutWidget_17 );
	lblDestSongCDA->setMaximumSize( QSize( 80, 30 ) );
	lblDestSongCDA->setText( tr( "Dest: Song" ) );
	hlayout18->addWidget( lblDestSongCDA );
	fieldDestSongCDA = new QLineEdit( privateLayoutWidget_17 );
	fieldDestSongCDA->setMaximumSize( QSize( 25, 30 ) );
	fieldDestSongCDA->setText( tr( "1" ) );
	hlayout18->addWidget( fieldDestSongCDA );
	lblDestTrackCDA = new QLabel( privateLayoutWidget_17 );
	lblDestTrackCDA->setMaximumSize( QSize( 36, 30 ) );
	lblDestTrackCDA->setText( tr( "Track" ) );
	hlayout18->addWidget( lblDestTrackCDA );
	fieldDestTrackCDA = new QLineEdit( privateLayoutWidget_17 );
	fieldDestTrackCDA->setMaximumSize( QSize( 25, 30 ) );
	fieldDestTrackCDA->setText( tr( "1" ) );
	hlayout18->addWidget( fieldDestTrackCDA );
	lblDestPosCDA = new QLabel( privateLayoutWidget_17 );
	lblDestPosCDA->setMaximumSize( QSize( 25, 30 ) );
	lblDestPosCDA->setText( tr( "Pos" ) );
	hlayout18->addWidget( lblDestPosCDA );
	fieldDestPosCDA = new QLineEdit( privateLayoutWidget_17 );
	fieldDestPosCDA->setMinimumSize( QSize( 77, 0 ) );
	fieldDestPosCDA->setMaximumSize( QSize( 55, 30 ) );
	fieldDestPosCDA->setText( "0:02:30,14" );
	hlayout18->addWidget( fieldDestPosCDA );
	btnAddSourceCDA = new QPushButton( privateLayoutWidget_17 );
	connect( btnAddSourceCDA, SIGNAL(clicked()), this, SLOT(btnAddSourceCDA_clicked()));
	btnAddSourceCDA->setText( tr( "Add as Audio Source" ) );
	//QToolTip::add( btnAddSourceCDA, tr( "Add this file/cd-track to song project as an audio source" ) );
	hlayout18->addWidget( btnAddSourceCDA );
	btnImportNowCDA = new QPushButton( privateLayoutWidget_17 );
	connect( btnImportNowCDA, SIGNAL(clicked()), this, SLOT(btnImportNowCDA_clicked()));
	btnImportNowCDA->setText( tr( "Import now!" ) );
	//QToolTip::add( btnImportNowCDA, tr( "Add this file/cd-track to song project as an audio source" ) );
	hlayout18->addWidget( btnImportNowCDA );
	QWidget* privateLayoutWidget_18 = new QWidget( grpImportCDA);
	privateLayoutWidget_18->setGeometry( QRect( 20, 50, 670, 22 ) );
	QHBoxLayout* hlayout19 = new QHBoxLayout( privateLayoutWidget_18, 0, 6);
	pbarImportCDA = new QProgressBar( privateLayoutWidget_18 );
	pbarImportCDA->setProgress( 0 );
	hlayout19->addWidget( pbarImportCDA );
	checkFakeImportCDA = new QCheckBox( privateLayoutWidget_18 );
	checkFakeImportCDA->setText( tr( "Fake Import" ) );
	//QToolTip::add( checkFakeImportCDA, tr( "Just import the audio source, spending the same amout of time, but does not put it into the song as audio source" ) );
	hlayout19->addWidget( checkFakeImportCDA );
	grpPreListenCDA = new QGroupBox( tab_5 );
	grpPreListenCDA->setGeometry( QRect( 10, 340, 700, 110 ) );
	grpPreListenCDA->setTitle( tr( "Pre-listen" ) );
	QWidget* privateLayoutWidget_19 = new QWidget( grpPreListenCDA);
	privateLayoutWidget_19->setGeometry( QRect( 10, 50, 260, 16 ) );
	QHBoxLayout* hlayout20 = new QHBoxLayout( privateLayoutWidget_19, 0, 6);
	lblCDAVol = new QLabel( privateLayoutWidget_19 );
	lblCDAVol->setText( tr( "Vol" ) );
	hlayout20->addWidget( lblCDAVol );
	sliderCDA = new QSlider( privateLayoutWidget_19 );
	sliderCDA->setMaximumSize( QSize( 300, 14 ) );
	sliderCDA->setOrientation( QSlider::Horizontal );
	hlayout20->addWidget( sliderCDA );
	btnCDAPlay = new QPushButton( grpPreListenCDA );
	connect( btnCDAPlay, SIGNAL(clicked()), this, SLOT(btnCDAPlay_clicked()));
	btnCDAPlay->setGeometry( QRect( 10, 20, 260, 22 ) );
	btnCDAPlay->setText( tr( "Play" ) );
	mainTabPane->insertTab( tab_5, tr( "Audio CD Broswer" ) );


	mainTabPane->setCurrentPage ( 0 );
	currentProject = (Project*) 0;
	isPlaying = false;
	PEXITCONS;
	}

ProjectManager::~ProjectManager()
	{
	PENTERDES;
	PEXITDES;
	}

void ProjectManager::show()
	{
	QWidget::show();
	update();
	}

void ProjectManager::update()
	{
	PENTER;
	update_audio_sources_list();
	update_projects_list();
	update_song_list();
	PEXIT;
	}

void ProjectManager::update_audio_sources_list()
	{
	PENTER;
	if (!currentProject) {
		PEXIT;
		return;
		}
	Song* s = currentProject->get_current_song();
	if (!s) {
		PEXIT;
		return;
		}
	AudioSourcesList* asl= s->get_asl();
	int tot = asl->get_total_sources();
	audioListView->clear();
	for (int i=0; i<tot; i++)
		{
		Audio* a = asl->get_source_for_index(i);
		if (!a) continue;
		QString sFile = a->file->filename;

		int ln = sFile.length();
		QString pTitle = currentProject->get_title();
		int skip = sFile.find(pTitle);
		ln -= (skip + pTitle.length());
		sFile = sFile.right(ln);

		QString sType;
		if (sFile.find("wav")>=0)
			sType = "WAV";
		else
			sType = "PRAF";
		QString sRate; sRate.setNum(a->file->rate);
		QString sBitDepth; sBitDepth.setNum(a->file->bitDepth);
		QString sChannels = (a->file->channels==1?"MONO":"STEREO");
		QString sLenght; sLenght.setNum((double)a->file->totalBlocks,'f',0);
		QString sSize; sSize.setNum((double)a->file->fileSize,'f',0);
		QString sClips; sClips.setNum(a->get_clips_count());

		QListViewItem* it = new QListViewItem(audioListView);
		it->setText(0,sFile);
		it->setText(1,sType);
		it->setText(2,sLenght);
		it->setText(3,sSize);
		it->setText(4,sClips);
		it->setText(5,"0");
		it->setText(6,sRate);
		it->setText(7,sChannels);
		it->setText(8,sBitDepth);
		}
	s->recreate();
	PEXIT;
	}


void ProjectManager::update_projects_list()
	{
	listAlbumProjects->clear();

	QString projsDir(getenv("HOME"));
	projsDir = projsDir + "/protuxprojects/";

	QDir pDir( projsDir );

	const QFileInfoList* files = pDir.entryInfoList();
	if ( files )
		{
		QFileInfoListIterator it( *files );
		QFileInfo* fi;
		while( (fi=it.current()) != 0 )
			{
			++it;
			if ( (fi->fileName() != ".") && (fi->fileName() != "..") )
				{
				QString sNumSongs;
				QString engineer;
				QString title = "Hey! Me here?";
				QFile f(projsDir+fi->fileName()+"/project.properties");
				if ( f.open(IO_ReadOnly) )
					{
					QTextStream stream( &f );
					while (!stream.eof())
						{
						QString line = stream.readLine();
						line = line.stripWhiteSpace();
						if (line.find("title=")>=0)
							title=line.mid(line.find("title=")+6);
						if (line.find("numSongs=")>=0)
							sNumSongs=line.mid(line.find("numSongs=")+9);
						if (line.find("engineer=")>=0)
							engineer=line.mid(line.find("engineer=")+9);
						}
					}
				QString status;
				if (currentProject && (currentProject->title == title))
					status = currentProject->has_changed()?"Unsaved":"Saved";
				else
					status="Saved";

				f.close();
				QListViewItem* it = new QListViewItem( listAlbumProjects);
				it->setText(0,title);
				it->setText(1,sNumSongs);
				it->setText(3,status);
				}
			}
		}
	}



void ProjectManager::update_song_list()
	{
	PENTER;
	if (!currentProject) {
		PEXIT;
		return;
		}

	int numSongs = currentProject->get_number_of_songs();

	listSongs->clear();
	for (int i=0; i<numSongs; i++)
		{
		Song* s = currentProject->get_song(i);
		if (!s) continue;
		QString songNr; songNr.setNum( (i + 1) );
		if( (i + 1 ) < 10)
			songNr.prepend("0");
		QString songName = "Song " + songNr + " - ";
		songName.append(s->get_title());
		QString numberOfTracks; numberOfTracks.setNum(s->get_total_tracks());

		int rate = s->get_rate();
		long long bpos = s->get_last_block();
		if (bpos<0) bpos=0;
		float f =  rate / 30;
		float flo = (float) bpos /  f;
		long long framepos = ( long long ) flo;
		int pfr = (int) framepos % 30;
		int pho = (int) (framepos/108000) % 60;
		int pmi = (int) (framepos/1800) % 60;
		int pse = (int) (framepos/30) % 60;
		if (pho<0) pho=0;
		if (pmi<0) pmi=0;
		if (pse<0) pse=0;
		if (pfr<0) pfr=0;
		QString spho; spho.setNum(pho);
		QString spmi; spmi.setNum(pmi); if (pmi<10) spmi.prepend("0");
		QString spse; spse.setNum(pse); if (pse<10) spse.prepend("0");
		QString spfr; spfr.setNum(pfr); if (pfr<10) spfr.prepend("0");
		QString songLength = spho + ":" + spmi + ":" + spse + "," + spfr;

		//QString songLength; songLength.setNum( spos );
		QString songRate; songRate.setNum(rate);
		QString songBitDepth; songBitDepth.setNum(s->get_bit_depth());
		QString songStatus = s->changed?"UnSaved":"Saved";
		QString songSpaceAllocated = "Unknown";
		/* for later:
		QString sLenght; sLenght.setNum((double)a->file->totalBlocks,'f',0);
		QString sSize; sSize.setNum((double)a->file->fileSize,'f',0);
		*/

		QListViewItem* it = new QListViewItem(listSongs);
		it->setText(0,songName);
		it->setText(1,numberOfTracks);
		it->setText(2,songLength);
		it->setText(3,songRate);
		it->setText(4,songBitDepth);
		it->setText(5,songStatus);
		it->setText(6,songSpaceAllocated);
		}
	PEXIT;
	}

// delete file/dir pName after prepending $HOME/protuxprojects/ to it
//
// if it is a directory, calls itself recursively  on any file/dir in the directory
// before removing the directory
int ProjectManager::remove_recursively(QString pName)
	{
	QString name(getenv("HOME"));
	name = name + "/protuxprojects/" + pName;

	// check if we are removing the currentProject, and delete it before removing its files
	if (project_is_current(pName))
		{
		if (currentProject->get_current_song()->get_status() != Mixer::STOPPED)
			{
			PERROR("current song is not stopped\n");
			PEXIT;
			return -1;
			}
		PMESG("removing current project\n");
		delete currentProject;
		currentProject = NULL;
		}

	QFileInfo fileInfo(name);

	if (!fileInfo.exists())
	return -1;
	if (!fileInfo.isWritable())
		{
		PERROR("failed to remove %s: you don't have write access to it\n", name.ascii());
		return -1;
		}

	if(fileInfo.isFile())
		{
		QFile file(name);
		if (!file.remove())
			{
			PERROR("failed to remove file %s\n", name.ascii());
			return -1;
			}
		return 0;
		}
	else if(fileInfo.isDir())
		{
		QDir dir(name);
		const QFileInfoList *files = dir.entryInfoList();
		if (files)
			{
			QFileInfoListIterator it( *files);
			QFileInfo *fi;
			while ((fi = it.current()) != 0)
				{
				QString filename(fi->fileName());
				if ((filename != ".") && (filename != ".."))
					{
					remove_recursively(pName + "/" + filename);
					}
				++it;
				}
			}
		if (!dir.rmdir(name))
			{
			PERROR("failed to remove directory %s\n", name.ascii());
			return -1;
			}

		return 0;
		}

	return -1;
	}


int ProjectManager::copy_recursively(QString pNameFrom, QString pNameTo)
	{
	if (currentProject && (currentProject->get_current_song()->get_status() != Mixer::STOPPED))
		{
		PERROR("current song is not stopped\n");
		PEXIT;
		return -1;
		}

	QString nameFrom(getenv("HOME"));
	QString nameTo(nameFrom);
	nameFrom = nameFrom + "/protuxprojects/" + pNameFrom;
	nameTo = nameTo + "/protuxprojects/" + pNameTo;

	QFileInfo fileFromInfo(nameFrom);
	QFileInfo fileToInfo(nameTo);

	if (!fileFromInfo.exists())
		{
		PERROR("File or directory %s doesn't exist\n", pNameFrom.ascii());
		return -1;
		}
	if (fileToInfo.exists())
		{
		PERROR("File or directory %s already exists", pNameTo.ascii());
		return -1;
		}

	if(fileFromInfo.isFile())
		{
		QFile fileFrom(nameFrom);
		if (!fileFrom.open(IO_ReadOnly))
			{
			PERROR("failed to open file %s for reading\n", nameFrom.ascii());
			return -1;
			}

		QFile fileTo(nameTo);
		if (!fileTo.open(IO_WriteOnly))
			{
			fileFrom.close();
			PERROR("failed to open file for writting%s\n", nameFrom.ascii());
			return -1;
			}

		// the real copy part should perhaps be implemented using QDataStream
		// but .handle() will still be needed to get the optimal block-size
		//
		//! \todo does not keep file mode yet
		int bufferSize = 4096;
		int fileDescFrom = fileFrom.handle();
		int fileDescTo = fileTo.handle();
		struct stat fileStat;
		if (fstat(fileDescFrom, &fileStat) == 0)
			bufferSize = (int)fileStat.st_blksize;

		void *buffer = malloc(sizeof(char) * bufferSize);
		// QMemArray<char> buffer(bufferSize);

		for (;;)
			{
			int nRead = read(fileDescFrom, buffer, bufferSize);
			if (nRead < 0)
				{
				fileFrom.close();
				fileTo.close();
				PERROR("Error while reading file %s\n", nameFrom.ascii());
				return -1;
				}
			if (nRead == 0)
				break;
			if (write(fileDescTo, buffer, nRead) < 0)
				{
				fileFrom.close();
				fileTo.close();
				PERROR("Error while writing file %s\n", nameTo.ascii());
				return -1;
				}
			}
		free(buffer);

		fileFrom.close();
		fileTo.close();

		return 0;
		}
	else if(fileFromInfo.isDir())
		{
		QDir dirFrom(nameFrom);
		QDir dirTo(nameTo);
		if (!dirTo.mkdir(nameTo))
			{
			PERROR("failed to create directory %s\n", nameTo.ascii());
			return -1;
			}

		const QFileInfoList *files = dirFrom.entryInfoList();
		if (files)
			{
			QFileInfoListIterator it( *files);
			QFileInfo *fi;
			while ((fi = it.current()) != 0)
				{
				QString filename(fi->fileName());
				if ((filename != ".") && (filename != ".."))
					{
					copy_recursively(pNameFrom + "/" + filename, pNameTo + "/" + filename);
					}
				++it;
				}
			}
		return 0;
		}
	return -1;
	}



void ProjectManager::set_current_project(Project* pProject)
	{
	currentProject = pProject;
	if (currentProject)
		{
		btnMaster->setEnabled( true );
		btnSaveProject->setEnabled( true );
		fieldAlbumTitle->setText( currentProject->get_title() );
		QString sNOfSongs;
		sNOfSongs.setNum(currentProject->get_number_of_songs());
		//fieldNOfSongs->setText(sNOfSongs);
		fieldEngineer->setText(currentProject->get_engineer());
		}
	else
		{
		btnMaster->setEnabled( FALSE );
		btnSaveProject->setEnabled( FALSE );
		}
	}


int ProjectManager::create_new_project(QString projectName, int numSong)
	{
	PENTER;

	if (currentProject && (currentProject->get_current_song()->get_status() != Mixer::STOPPED))
		{
		PERROR("current song is not stopped\n");
		PEXIT;
		return -1;
		}

	if(projectExists(projectName))
		{
		PERROR("project %s already exists\n", projectName.ascii());
		PEXIT;
		return -1;
		}

	Project *newProject = new Project(projectName, parentInterface);
	if (newProject->create(numSong) < 0)
		{
		delete newProject;
		PERROR("couldn't create new project %s", projectName.ascii());
		PEXIT;
		return -1;
		}

	if (currentProject)
		{
		currentProject->save();
		delete currentProject;
		}

	set_current_project(newProject);
	currentProject->save();
	currentProject->recreate();
	PEXIT;
	return 0;
	}


int ProjectManager::load_project(QString projectName)
	{
	PENTER;
	if (currentProject && (currentProject && (currentProject->get_current_song()->get_status() != Mixer::STOPPED)))
		{
		PERROR("current song is not stopped\n");
		PEXIT;
		return -1;
		}

	if(!projectExists(projectName))
		{
		PERROR("project %s doesn't exist\n", projectName.ascii());
		PEXIT;
		return -1;
		}

	Project *newProject = new Project(projectName, parentInterface);
	if (!newProject)
		return -1;

	if (newProject->load() > 0)
		{
		if (currentProject)
			delete currentProject;
		set_current_project(newProject);
		if(newProject->load_songs() == 0)
			{
			PERROR("couldn't load one or more songs for %s",projectName.ascii());
			}
		newProject->recreate();
		}
	else
		{
		delete newProject;
		PERROR("couldn't load project %s", projectName.ascii());
		PEXIT;
		return -1;
		}

	PEXIT;
	return 0;
	}


bool ProjectManager::project_is_current(QString title)
	{
	QString path(getenv("HOME"));
	path = path + "/protuxprojects/" + title;

	if (currentProject && (currentProject->get_root_dir() == path))
		return true;
	return false;
	}




bool ProjectManager::projectExists(QString title)
	{
	QString path(getenv("HOME"));
	QFileInfo fileInfo(path + "/protuxprojects/" + title);
	if (fileInfo.exists())
		return true;
	return false;
	}

// LG : there could be a utility on libmustux to easy and transparently
// perform audio previews (audio plays)....
// RS : The filosofy behind the "if (pm->isPlaying)" is that this method needs to continue
// until it reaches the end. We can of course use a "goto", but thats bad programming
// Then everything is reset, and the thread is exited.
// RS : This thread handles a lot it shouldnt do. But for now we have to live with it. When
// it is moved as a leightweight player to libmuxtux we can make it more professional, and update
// the methods which are creating this thread
void* ProjectManager::th_preListen(void *arg)
	{
	ProjectManager* pm = (ProjectManager*) arg;
	MustuxAudioFileFormat* inputFile = (MustuxAudioFileFormat*) 0;
	QString fileName = "";
	if (pm->whichTabActive == 2)
		{
		QListViewItem* it = pm->audioListView->selectedItem();
		if (it)
			{
			fileName = it->text(0);
			fileName.prepend(pm->currentProject->get_root_dir());
			}
		}
	else if (pm->whichTabActive == 3)
		{
		fileName = pm->fileBrowser->pathCombo->currentText();
		}
	if (fileName.find("praf",0,false)>=0)	//FIXME There should be a method in MADM like get_file_type(). If it is implemented, use that instead of this crap!!!
		{
		inputFile = new PrafFile();
		}
	else if (fileName.find("wav", 0,false)>=0)
		{
		inputFile = new WavFile();
		}
#ifdef OGG_VORBIS_SUPPORT
	else if (fileName.find("ogg", 0,false)>=0)
		{
		inputFile = new OggFile();
		}
#endif
	else
		{
		pm->isPlaying = false;
		}

	if (pm->isPlaying)
		{
		if (inputFile->open(fileName) != 1)
			{
			PERROR("Could not open selected file!");
			pm->isPlaying = false;
			delete inputFile;
			}
		}

	if (pm->isPlaying)
		{
		if (inputFile->read_header() != 1)
			{
			PMESG("This is not a PRAF, WAV or OGG file");
			pm->isPlaying = false;
			delete inputFile;
			}
		}

	int busId, rate, bitDepth, channels;
	if (pm->isPlaying)
		{
		inputFile->get_info();
		rate = inputFile->rate;
		bitDepth = inputFile->bitDepth;
		channels = inputFile->channels;
		QString desc = pm->cmbPreListenBus->currentText();
		if (desc=="AUTO")
			{
			busId = MustuxAudioDeviceMapper::find_bus_out(rate,bitDepth,channels);
			if (busId<0)
				{
				PERROR("Cannot find a bus to prelisten the audio");
				pm->isPlaying = false;
				delete inputFile;
				}
			}
		else
			{
			QString s = GlobalProperties::get("USE_PLUGHW");
			if (s=="YES")
				{
				MustuxAudioDeviceMapper::set_plughw_use(true);
				desc.prepend("plug");
				}
  			else
				MustuxAudioDeviceMapper::set_plughw_use(false);
			busId=MustuxAudioDeviceMapper::get_bus_out_index_for_descriptor(desc);
			}
		if (pm->isPlaying)
			{
			if (MustuxAudioDeviceMapper::open_bus_out(busId,rate,bitDepth,channels==2?MustuxAudioDeviceMapper::STEREO:MustuxAudioDeviceMapper::MONO) < 0)
				{
				PERROR("Cannot open bus out");
				pm->isPlaying = false;
				delete inputFile;
				}
			}
		}

	if (pm->isPlaying)
		{
		pm->btnSourcePlay->setText("Stop");
		pm->btnFilePlay->setText("Stop");
		char *buffer = MustuxAudioDeviceMapper::get_bus_out_transfer_buffer(busId, inputFile->channels);
		int bufferSize = MustuxAudioDeviceMapper::get_bus_out_transfer_size(busId);
		while ( pm->isPlaying && inputFile->read_fragment(buffer, bufferSize))
			{
			char* p = buffer;
			char* lastp = p + bufferSize;
			float gain = pm->gain;
			float gainFactor = (gain>=0.0f)? 1.0f + gain/6.0f : 1.0f / pow(2.0f,-1.0f*gain/6.0f);
			while (p<lastp)
				{
				for (int k=0; k<channels; k++)
					{
					short* psho = (short*)p;
					short sample = *(psho);
					float fsample = (float) sample * gainFactor;
					CHECK_SAMPLE_OVERLOAD(fsample);
					*(psho)= (short) fsample;
					p+=sizeof(short);
					}
				}
			MustuxAudioDeviceMapper::bus_out_transfer(busId);
			}

		MustuxAudioDeviceMapper::clean_bus_out(busId);
		MustuxAudioDeviceMapper::close_bus_out(busId);
		inputFile->close_file();
		}

	pm->btnSourcePlay->setText("Play");
	pm->btnFilePlay->setText("Play");
	pm->isPlaying = false;
	pthread_exit((void*)0);
	}



// ------------------------------- xxx_clicked()

void ProjectManager::tab_changed()
	{
	if (mainTabPane->currentPageIndex()==2)
		{
		update_audio_sources_list();
		}
	if (mainTabPane->currentPageIndex()==1)
		{
		update_song_list();
		}
	}


void ProjectManager::listAlbumProject_projectSelected()
	{
	QListViewItem *item = NULL;

	if ((item = listAlbumProjects->currentItem()) == 0)
		return;

	fieldSaveProjAs->setText(item->text(0));
	}


void ProjectManager::btnCreateProject_clicked()
	{
	// is current song stopped ?
	if (currentProject && (currentProject->get_current_song()->get_status() != Mixer::STOPPED))
		{
		QMessageBox::warning(this,
				"Protux - Warning",
				"This operation is not permitted while a song is playing,\nrecording, or some tracks are armed",
				"Ok");
		return;
		}

	// do we have the name of the project to create ?
	QString title;
	title = fieldSaveProjAs->text();
	if (title.length() == 0)
		{
		QMessageBox::warning(this,
				"Protux - Warning",
				"You must supply a name for the project",
				"Ok");
		return;
		}

	// ask if the current project should first be saved (only when status is "UnSave")
	if (currentProject && currentProject->has_changed())
		switch (QMessageBox::information(this,
						"Protux - Question",
						"Should the current project first be saved ?",
						"Yes", "No", QString::null, 0, -1))
			{
			case -1:
				return;
				break;
			case 0:
				currentProject->save();
				break;
			default:
				break;
			}


	// first test if project exists already
	if (projectExists(title))
		{
		switch (QMessageBox::information(this,
						"Protux - Question",
						"The project \"" + title + "\" already exists, do you want to remove it and replace it with a new one ?",
						"Yes", "No", QString::null, 1, -1))
			{
			case 0:
				remove_recursively(title);
				break;
			default:
				return;
				break;
			}
		}
	QString sNumSongs = fieldNOfSongs->text();
	bool ok;
	int numSongs = sNumSongs.toInt(&ok, 10);
	//When input is not a decimal number, set it to a sane default value
	if (!ok) numSongs = 2;

	if( create_new_project(title, numSongs) < 0)
		QMessageBox::warning(this,
				"Protux - Error", "Couldn't create project \"" + title +"\"",
				"Ok");
	else
		{
		parentInterface->set_current_project(currentProject);
		update_projects_list();
		}
	}


void ProjectManager::btnImportNow_clicked()
	{
	if (btnImportFileNow->text() == "Import now!")
		{
		if (currentProject)
			{
			btnImportFileNow->setText("Cancel import!");
			qApp->processEvents();	//please remove me, but I'm used when importing mp3 files, which I (btnImportFileNow) can't abort :-(
			if(currentProject->import_audio(fileBrowser->pathCombo->currentText(),pbarFileImport))
				{
				//some error message?
				}
			btnImportFileNow->setText("Import now!");
			}
		}
	else
		{
		currentProject->get_current_song()->cancel_import_audio();
		btnImportFileNow->setText("Import now!");
		pbarFileImport->reset();
		}
	}


void ProjectManager::btnSaveProject_clicked()
	{
	// is current song stopped ?
	if (currentProject && (currentProject->get_current_song()->get_status() != Mixer::STOPPED))
		{
		QMessageBox::warning(this,
				"Protux - Warning",
				"This operation is not permitted while a song is playing,\nrecording, or some tracks are armed",
				"Ok");
		return;
		}

	if (currentProject)
		{
		currentProject->set_title(fieldAlbumTitle->text());
		currentProject->set_engineer(fieldEngineer->text());
		currentProject->save();
		QMessageBox::warning( 0, "Project Saved", "Project \"" + currentProject->title + "\" saved.");
		}
	update_projects_list();
	}


void ProjectManager::btnLoadProject_clicked()
	{
	// is current song stopped ?
	if (currentProject && (currentProject->get_current_song()->get_status() != Mixer::STOPPED))
		{
		QMessageBox::warning(this,
				"Protux - Warning",
				"This operation is not permitted while a song is playing,\nrecording, or some tracks are armed",
				"Ok");
		return;
		}

	// do we have the name of the project to load ?
	QString title;
	title = fieldSaveProjAs->text();
	if (title.length() == 0)
		{
		QMessageBox::warning(this,
				"Protux - Warning",
				"You must supply a name for the project",
				"Ok");
		return;
		}

	// ask if the current project should first be saved
	if (currentProject && currentProject->has_changed())
		switch (QMessageBox::information(this,
						"Protux - Question",
						"Should the current project first be saved ?",
						"Yes", "No", QString::null, 0, -1))
			{
			case -1:
				return;
				break;
			case 0:
				currentProject->save();
				break;
			default:
				break;
			}

	// first test if project exists
	if (!projectExists(title))
		{
		QMessageBox::warning(this,
				"Protux - Warning",
				"The project \"" + title + "\" doesn't exist",
				"Ok");
		update_projects_list();
		return;
		}
	if (load_project(title)<0)
		{
		PERROR("Could not load project %s", title.ascii());
		}
	else
		parentInterface->set_current_project(currentProject);
	}


void ProjectManager::btnCopyToProject_clicked()
	{
	// is current song stopped ?
	if (currentProject && (currentProject->get_current_song()->get_status() != Mixer::STOPPED))
		{
		QMessageBox::warning(this,
				"Protux - Warning",
				"This operation is not permitted while a song is playing,\nrecording, or some tracks are armed",
				"Ok");
		return;
		}

	// do we have the name of the project to create ?
	if (!currentProject)
		{
		QMessageBox::information(this,
				"Protux - Information",
				"There is no Project to save!",
				"Ok");
		return;
		}

	QString projectTo;
	projectTo = fieldSaveProjAs->text();
	if (projectTo.length() == 0)
		{
		QMessageBox::warning(this,
				"Protux - Warning",
				"You must supply a name for the project",
				"Ok");
		return;
		}

	// the current project should first be saved
	//! \todo only ask if project has not been saved
	if (currentProject)
		switch (QMessageBox::information(this,
						"Protux - Question",
						"The current project has first to be saved before you can copy it !\nFuture versions will hopefully get rid of this constraint.",
						"Ok", "Cancel", QString::null, 0, -1))
			{
			case 0:
				currentProject->save();
				break;
			default:
				return;
				break;
			}

	// first test if project exists already
	if (projectExists(projectTo))
		{
		switch (QMessageBox::information(this,
						"Protux - Question",
						"The project \"" + projectTo + "\" already exists, do you want to remove it and replace it with the current one ?",
						"Yes", "No", QString::null, 1, -1))
			{
			case 0:
				remove_recursively(projectTo);
				break;
			default:
				return;
				break;
			}
		}

	QString projectFrom(currentProject->get_root_dir());
	// projectFrom = projectFrom.section("/", -1); // was not compatible with Qt 2.3, replaced with following 2 lines
	int index = projectFrom.findRev("/", -1);
	projectFrom = projectFrom.mid(index + 1);
	copy_recursively(projectFrom, projectTo);
	update_projects_list();
	load_project(projectTo);
	}


void ProjectManager::btnDeleteProject_clicked()
	{
	// do we have the name of the project to delete ?
	QString title;
	title = fieldSaveProjAs->text();
	if (title.length() == 0)
		{
		QMessageBox::warning(this,
				"Protux - Warning",
				"You must supply a name for the project",
				"Ok");
		return;
		}

	// if deleting current project, is current song stopped ?
	if ((project_is_current(title)) && currentProject && (currentProject->get_current_song()->get_status() != Mixer::STOPPED))
		{
		QMessageBox::warning(this,
				"Protux - Warning",
				"This operation is not permitted while a song is playing,\nrecording, or some tracks are armed",
				"Ok");
		return;
		}

	// first test if project exists
	if (!projectExists(title))
		{
		QMessageBox::warning(this,
				"Protux - Error",
				"The project \"" + title + "\" doesn't exist",
				"Ok");
		update_projects_list();
		return;
		}

	switch (QMessageBox::information(this,
					"Protux - Question",
					"Are you sure that you want to remove the project \"" + title + "\" ? It's not possible to undo it !",
					"Yes", "No", QString::null, 1, -1))
		{
		case 0:
			remove_recursively(title);
			update_projects_list();
			break;
		default:
			return;
			break;
		}
	return;
	}

void ProjectManager::listSongs_songSelected()
	{
	QListViewItem *item = NULL;

	if ((item = listSongs->currentItem()) == 0)
		return;

	Song* s;
	QString title;
	QString artists;
	title = item->text(0);
	int length = title.length();
	length -= 10;

	//find the selected Song
	QString t = title.mid(5,2);
	bool b;
	int nr = t.toInt(&b, 10);
	s = currentProject->get_song( (nr - 1) );

	artists = s->get_artists();
	title = title.right(length);
	fieldCopyTo->setText(title);
	fieldArtists->setText(artists);
	}


void ProjectManager::btnSaveSong_clicked()
	{
	PENTER;
	if(!currentProject)
		{
		PEXIT;
		return;
		}
	QListViewItem* it = listSongs->selectedItem();
	Song* s;
	QString name;
	if (it)
		{
		name = it->text(0);
		//assuming that song name is always: "Song nr title" so nr is always on position 5-7
		QString t = name.mid(5,2);
		bool b;
		int nr = t.toInt(&b, 10);
		s = currentProject->get_song( (nr - 1) );
		}
	else
		{
		s = currentProject->get_current_song();
		name.setNum(s->get_mynumber());
		name.prepend("Song ");
		}

	if(!(s->save()))
		{
		QMessageBox::information(this,
				"Protux - Information",
				"Failed to save " + name,
				"Ok");
		}
	update_song_list();
	PEXIT;
	}

void ProjectManager::btnSaveSongAs_clicked()
	{
	PENTER;
	if(!currentProject)
		{
		PEXIT;
		return;
		}
	QListViewItem* it = listSongs->selectedItem();
	Song* s;
	QString name;
	QString title;
	QString artists;
	if (it)
		{
		name = it->text(0);
		//assuming that song name is always: "Song nr title" so nr is always on position 5-7
		QString t = name.mid(5,2);
		bool b;
		int nr = t.toInt(&b, 10);
		s = currentProject->get_song( (nr - 1) );
		title = fieldCopyTo->text();
		artists = fieldArtists->text();
		if (title.length() != 0)
			{
			s->set_title(title);
			update_song_list();
			}
		if (artists.length() != 0)
			{
			s->set_artists(artists);
			}
		if(!(s->save()))
			{
			QMessageBox::information(this,
					"Protux - Information",
					"Failed to rename and save " + name,
					"Ok");
			}
		}
	PEXIT;
	}

void ProjectManager::btnSaveSongCopyAs_clicked()
	{
	DevelGuardian::warn_not_implemented("COPY SONG COPY AS","No tip for this lack.. sorry :-}");
	}

void ProjectManager::btnNewSong_clicked()
	{
	PENTER;
	if(!currentProject)
		{
		PEXIT;
		return;
		}
	Song* s = currentProject->add_song();
	if(!s)	PMESG("Failed to add song to current project");
	update_song_list();
	PEXIT;
	}

void ProjectManager::btnRemoveSongFromProject_clicked()
	{
	DevelGuardian::warn_not_implemented("Remove Song From Project","No tip for this lack.. sorry :-}");
	}

void ProjectManager::btnDeleteSong_clicked()
	{
	DevelGuardian::warn_not_implemented("Delete Song","No tip for this lack.. sorry :-}");
	}

void ProjectManager::delete_source()
	{
	PENTER;
	//Remove current selected audio source in audioListView;
	QListViewItem* it = audioListView->selectedItem();
	if (!it)
		{
		PEXIT;
		return;
		}
	QString s = it->text(0);
	s.prepend(currentProject->get_root_dir());
	Song* cs = currentProject->get_current_song();
	Audio* a = cs->get_asl()->get_audio_for_source(s);
	cs->remove_all_clips_for_audio(a);
	cs->get_asl()->remove(a);
	delete a;
	QFile file(s);
	if (!file.remove())
		{
		PERROR("failed to remove file %s\n", s.ascii());
		}
	QString peakFile = s + ".peak" ;
	file.setName(peakFile);
	if (!file.remove())
		{
		PERROR("failed to remove file %s\n", peakFile.ascii());
		}
	update_audio_sources_list();
	cs->recreate();
	PEXIT;
	}

void ProjectManager::btnRemoveUnusedSources_clicked()
	{
	PENTER;
	if(!currentProject)
		{
		PEXIT;
		return;
		}
	int numberRemovedSources = 0;
	int numberOfSongs = currentProject->get_number_of_songs();
	for (int i=0; i<numberOfSongs; i++)
		{
		Song* s = currentProject->get_song(i);
		AudioSourcesList* asl= s->get_asl();
		int tot = asl->get_total_sources();
		for (int i=tot-1; i>=0; --i)
			{
			Audio* a = asl->get_source_for_index(i);
			if (!a)		//No audio source for this index, strange....
				{
				PWARN("No audioSource for this index in AudioSourcesList!");
				continue;
				}
			QString aName = a->file->filename;
			if (s->get_clips_count_for_audio(a) == 0)
				{
				if (asl->remove(a) < 0)
					PWARN("Could not remove audioSource from AudioSourcesList!");
				QFile file(aName);
				if (!file.remove())
					{
					PERROR("failed to remove file %s\n", aName.ascii());
					}
				QString peakFile = aName + ".peak" ;
				file.setName(peakFile);
				if (!file.remove())
					{
					PERROR("failed to remove file %s\n", peakFile.ascii());
					}
				delete a;
				numberRemovedSources++;
				}
			}
		}
	if (numberRemovedSources == 0)
		{
		PEXIT;
		return;
		}
	update_audio_sources_list();
	QString nrs; nrs.setNum(numberRemovedSources);
	QMessageBox::information(this,
			"Protux - Information",
			"Removed " +  nrs + " unused Sources!",
			"Ok");

	}

void ProjectManager::btnRemoveAllClips_clicked()
	{
	PENTER;
	audioListView->selectAll(true);		//Hmm, maybe not the best way, but it works ;-)
	if(!audioListView->selectedItem())	//nothing to remove at all...
		{
		PEXIT;
		return;
		}
	switch( QMessageBox::warning( this, "Protux - Warning!",
						"This action irreversibly deletes ALL current Song \n"
						"          Audio Sources from your hard disk. \n"
						"                      Are you sure ?",
						"&Yes", "&No", 0, 1 ) )
		{
		case 0: //Yes clicked or Alt+Y pressed or Enter pressed.
			//Continue executing
			break;
		case 1: // No clicked or Alt+N pressed
			//Pfewh, that was a close one :-)
			PEXIT;
			return;
		}
	while(audioListView->selectedItem())
		{
		delete_source();
		audioListView->selectAll(true);
		}
	PEXIT;
	}


void ProjectManager::btnExportSource_clicked()
	{
	DevelGuardian::warn_not_implemented("EXPORT SOURCE","Try to convert the audio source manually with genpraf, ogg123, mpg123, notlame, or some other console tool");
	}


void ProjectManager::btnSourcePlay_clicked()
	{
	if (currentProject && (currentProject->get_current_song()->get_status() != Mixer::STOPPED))
		{
		PERROR("Current song is not stopped\n");
		return;
		}
	if(btnSourcePlay->text() == "Play")
		{
		if(!isPlaying)
			{
			isPlaying = true;
			pthread_t playThread;
			whichTabActive = 2;
			pthread_create (&playThread, 0, th_preListen, this);

			// LG : I Dont know if this is really necessary for a simple preview ...
			// RS : Why not? dropouts in prelisten are also annoying! But the worning can be removed
			struct sched_param rtparam;
			rtparam.sched_priority = 50;
			int x;
			if ((x = pthread_setschedparam (playThread, SCHED_FIFO, &rtparam)) != 0)
				{
				//Do nothing, this warning is allready handled in MADM
				//PWARN("Cannot set playFromSourceThread to real-time priority: %d", rtparam.sched_priority);
				}
			}
		}
	else
		{
		isPlaying = false;
		}
	}

void ProjectManager::sliderFile_moved()
	{
	gain = (float) sliderFile->value() / 3.0;
	sliderSource->setValue( sliderFile->value() );
	}

void ProjectManager::sliderSource_moved()
	{
	gain = (float) sliderSource->value() / 3.0;
	sliderFile->setValue( sliderSource->value() );
	}


void ProjectManager::btnFilePlay_clicked()
	{
	if (currentProject && (currentProject->get_current_song()->get_status() != Mixer::STOPPED))
		{
		PERROR("Current song is not stopped\n");
		return;
		}
	if(btnFilePlay->text() == "Play")
		{
		if(!isPlaying)
			{
			isPlaying = true;
			pthread_t playThread;
			whichTabActive = 3;
			pthread_create (&playThread, 0, th_preListen, this);

			// LG : I Dont know if this is really necessary for a simple preview ...
			// RS : Why not? dropouts in prelisten are also annoying! But the worning can be removed
			struct sched_param rtparam;
			rtparam.sched_priority = 50;
			int x;
			if ((x = pthread_setschedparam (playThread, SCHED_FIFO, &rtparam)) != 0)
				{
				//Do nothing, this warning is allready handled in MADM
				//PWARN("Cannot set playFromFileThread to real-time priority: %d", rtparam.sched_priority);
				}
			}
		}
	else
		{
		isPlaying = false;
		}
	}


void ProjectManager::btnAddFileAsSource_clicked()
	{
	DevelGuardian::warn_not_implemented("ADD AS AUDIO SOURCE","Import it normally then it will become an audio source");
	}

void ProjectManager::btnAddSourceCDA_clicked()
	{
	DevelGuardian::warn_not_implemented("Import CDA track as audio source","Use \"cdda2wav -D/dev/cdrom -tX\" in\na console to import it (Note: X is the number of the track)");
	}

void ProjectManager::btnImportNowCDA_clicked()
	{
	DevelGuardian::warn_not_implemented("Import CDA track as audio source","Use \"cdda2wav -D/dev/cdrom -tX\" in\na console to import it (Note: X is the number of the track)");
	}

void ProjectManager::btnCDAPlay_clicked()
	{
	DevelGuardian::warn_not_implemented("CD PLAYER","Use the KDE or the Gnome CD player");
	}

void ProjectManager::btnRender_clicked()
	{
	DevelGuardian::warn_not_implemented("FULL PROJECT RENDER","You can render each song a time by <<ENTER>> in them");
	}

void ProjectManager::btnMaster_clicked()
	{
	DevelGuardian::warn_not_implemented("MASTER PROJECT","This is feature for a very distant future release.");
	}

void ProjectManager::btnBurnCD_clicked()
	{
	DevelGuardian::warn_not_implemented("BURN CD ","We suggest xcdroast to burning CDs");
	}


void ProjectManager::audioListView_itemRenamed( QListViewItem *, int )
	{

	}

void ProjectManager::audioListView_spacePressed( QListViewItem * )
	{

	}


void ProjectManager::audioListView_returnPressed( QListViewItem * )
	{

	}


void ProjectManager::audioListView_onItem( QListViewItem * )
	{

	}


void ProjectManager::audioListView_collapsed( QListViewItem * )
	{

	}


void ProjectManager::audioListView_expanded( QListViewItem * )
	{

	}

// eof
