/**************************************************************************************

	PROTUX - THE FREE PROFESSIONAL AUDIO TOOLS FOR LINUX
	AUTHOR : See AUTHORS file for details

	This software is distributed under the terms of the GNU General Public License
	as specified in the COPYING file.

***************************************************************************************/

#include <mustux.h>
#include "Track.hh"
#include "ColorManager.hh"

// TODO : ALL lists stuff should be a single super class.. for example
// clips, regions, selections, and so on...

Track::Track(Song* pParentSong, int pID, QString pName, int pBaseY, int pHeight )
	: MustuxDrawableRegion( pParentSong->get_tracks_area() )
	{
	PENTERCONS;
	parentSong = pParentSong;
	mixer = parentSong->get_mixer();
	name = pName;
	ID = pID;
	baseY = pBaseY;
	height = pHeight;
	clipList = (AudioClip*) 0;
	filterChain = new FilterChain(this);
	isActive = false;
	isMuted = false;
	isSolo = false;
	isArmed=false;
	busIn = 0;
	busOut = 0;
	pan=0;
	gain=0;
	recordingBufferSize=0;	//added by Remon (fixes segfault in update_recording_clip)
	lastRecordedClip = (AudioClip*) 0;
	whichChannels = MustuxAudioDeviceMapper::STEREO; // could be ONLY_LEFT_CHANNEL or ONLY_RIGHT_CHANNEL
	int b = 0;
	while (b<MustuxAudioDeviceMapper::get_total_buses_in())
		{
		if (mixer->valid_capture_bus(b))
			{busIn=b; break;}
		b++;
		}
	b=0;
	while (b<MustuxAudioDeviceMapper::get_total_buses_out())
		{
		if (mixer->valid_playback_bus(b))
			{busOut=b; break;}
		b++;
		}
	if (busIn<0)
		{
		PWARN("No Valid capture buses found for this Song settings");
		}
	if (busOut<0)
		{
		PWARN("No Valid playback buses found for this Song settings");
		}

	updateTimer = new QTimer(this);
	connect( updateTimer , SIGNAL(timeout()), this, SLOT(update_recording_clip()));

	PEXITCONS;
	}


Track::~Track()
	{
	PENTERDES;
	delete filterChain;
	PEXITDES;
	}


void Track::set_assoc_track_panel(TrackPanel* pAssocTrackPanel)
	{
	assocTrackPanel = pAssocTrackPanel;
	}


void Track::recreate()
	{
	PENTER3;
	if (visible())
		{
		clear();
		draw();
		update();
		assocTrackPanel->recreate();
		}
	PEXIT3;
	}

// LG: should be named "recrate_only_in_mta"
void Track::recreate_mta()
	{
	PENTER3;
	if (visible())
		{
		clear();
		draw();
		update();
		}
	PEXIT3;
	}


void Track::update()
	{
	drawArea->update(0,real_baseY(),drawArea->width(),height);
	}



void Track::clear()
	{
	PENTER3;
	QPainter* p = drawArea->painter;
	if (isActive)
		p->fillRect(0, real_baseY(), drawArea->width() , height, CM_COLOR(TRACK_BG_ACTIVE));
	else
		p->fillRect(0, real_baseY(), drawArea->width() , height, CM_COLOR(TRACK_BG));

        // TRACK SEPARATOR (IMPROVE-ME (USE COLOMANAGER))
	if (parentSong->editingMode==Song::EDITING_MODE_TRACK_CURVES)
		{
		p->setPen(QColor(50,50,50));// This is intentionally fixed color
		p->drawLine(0, real_baseY() + height, drawArea->width(), real_baseY() + height);
		p->setPen(QColor(150,150,160));// This is intentionally fixed color.
		p->drawLine(0, real_baseY() + height -1 , drawArea->width(), real_baseY() + height -1);
		}
	else
		{
		p->setPen(QColor(100,100,100));// This is intentionally fixed color
		p->drawLine(0, real_baseY() + height, drawArea->width(), real_baseY() + height);
		p->setPen(QColor(240,240,240));// This is intentionally fixed color.
		p->drawLine(0, real_baseY() + height -1 , drawArea->width(), real_baseY() + height -1);
		}

	// Draw region markers
	MtaRegion* m = parentSong->regionList;
	while (m)
		{
		int xrs = parentSong->block_to_xpos(m->beginBlock);
		int xre = parentSong->block_to_xpos(m->endBlock);
		int xs,xe=0;
		int w=drawArea->width();
		if ((xrs>=0) && (xrs<w))
			xs=xrs;
		else if (xrs>=0)
			xs=w;
		else
			xs=0;

		if ((xre>=0) && (xre<w))
			xe=xre;
		else if (xre>=0)
			xe=w;
		else
			xe=0;
		p->fillRect(xs,real_baseY(),xe-xs,3,QColor(155,100,250));
		drawArea->update(xs,real_baseY(),xe-xs,3);
		m=m->next;
		}
	PEXIT3;
	}




int Track::draw()
	{
	PENTER3;
	AudioClip* c=clipList;
	while (c)
		{
		c->draw();
		c=c->next;
		}
	if (parentSong->editingMode==Song::EDITING_MODE_TRACK_CURVES)
		filterChain->draw();
	PEXIT3;
	return 1;
	}



bool Track::is_pointed()
	{
	drawArea = parentSong->tracksArea;
	int x = drawArea->mapFromGlobal(QCursor::pos()).x();
	int y = drawArea->mapFromGlobal(QCursor::pos()).y();
	bool b = ( (x >= 0)  &&
		(x <= drawArea->width() ) &&
		(y >= real_baseY() ) &&
		(y <= real_baseY() + height )
		);
	return b ;
	}



int Track::split_clip(long long splitPoint)
	{
	PENTER2;
	PMESG("Trying to find clip under %d",splitPoint);
	AudioClip* c = get_clip_under(splitPoint);
	if (c!=0)
		{
		long long leftLenght = splitPoint - (c->trackFirstBlock);
		long long clipLenght = c->sourceLastBlock - c->sourceFirstBlock;
		long long rightLenght = clipLenght - leftLenght;
		long long rightSourceFirstBlock = c->sourceFirstBlock + leftLenght;
		c->sourceLastBlock = c->sourceFirstBlock + leftLenght;
		c->trackLastBlock = c->trackFirstBlock + leftLenght;
		add_clip(c->get_audio_source(), splitPoint, rightSourceFirstBlock, rightLenght, c->isTake);
		}
	else
		{
		PMESG("I am upon no clip :-(");
		}
	recreate();
	parentSong->parentProject->get_parent_interface()->get_project_manager()->update();
	PEXIT2;
	return 0;
	}




AudioClip* Track::get_clip_under(long long blockPos)
	{
	PENTER4;
	AudioClip* theClip = (AudioClip*) 0;
	AudioClip* c=clipList;
	while (c)
		{
		long long fb = c->trackFirstBlock;
		long long lb = c->trackLastBlock;
		if ((fb<=blockPos) && (lb>=blockPos))
			theClip = c;
		if (fb>blockPos) break;
		c=c->next;
		}
	PEXIT4;
	return theClip;
	}



AudioClip* Track::get_clip_after(long long blockPos)
	{
	AudioClip* c=clipList;
	while (c)
		{
		if (c->trackFirstBlock > blockPos)
			return c;
		c=c->next;
		}
	return (AudioClip*) 0;
	}



AudioClip* Track::get_clip_before(long long blockPos)
	{
	AudioClip* c  = clipList;
	while (c)
		{
		if (c->trackFirstBlock < blockPos)
			return c;
		c=c->next;
		}
	return (AudioClip*) 0;
	}




int Track::delete_clip(AudioClip* clip, bool permanently)
	{
	PENTER;
	if (!clip)
		{
		PERROR("Trying to delete invalid Clip");
		PEXIT;
		return -1;
		}
	if(permanently && (parentSong->get_clips_count_for_audio(clip->audioSource) == 1))//FIXME this clip is the only one with this audioSource,
		clip->audioSource->peak->unbuild();					//so we have to unbuild the peak RAMBuffers... or not? Waste of memory usage if we keep it there...
	if (!clip->prev) // if it is the first clip...
		{
		clipList=clip->next;
		if (clipList)
			clipList->prev=0;
		delete clip;
		PEXIT;
		return 1;
		}
	if (clip->next==0) // if it is the last clip...
		{
		if (!clip->prev) // oops it is also the first ! (it is a single clip in the track)
			{
			delete clip;
			clipList=0;
			}
		else
			{
			clip->prev->next=0;
			delete clip;
			}
		PEXIT;
		return 1;
		}
	clip->prev->next = clip->next;
	clip->next->prev = clip->prev;
	delete clip;

	// JUST FOR DEBUG
	PMESG3("Entire clip list for track now is ");
	AudioClip* c=clipList;
	int i=0;
	while (c)
		{
		PMESG3("Clip # %d ",i);
		i++;
		c=c->next;
		}

	PEXIT;
	return 1;
	}


AudioClip* Track::add_clip(AudioClip* clip, long long newInsertBlock)
	{
	PENTER;
	AudioClip* c = add_clip(clip->audioSource, newInsertBlock, clip->sourceFirstBlock, clip->get_lenght(), clip->isTake);
	c->set_fade_in(clip->fadeInBlocks);
	c->set_fade_out(clip->fadeOutBlocks);
	c->set_gain(clip->gain);
	return c;
	PEXIT;
	}


AudioClip* Track::add_clip(Audio* audioSource, long long trackInsertBlock, bool isTake)
	{
	PENTER;
	PMESG("Creating clip starting in %d" ,trackInsertBlock);
	AudioClip* c = add_clip(audioSource, trackInsertBlock, 0, audioSource->file->totalBlocks,isTake);
	recreate();
	PEXIT;
	return c;
	}



AudioClip* Track::add_clip(Audio* audioSource, long long trackInsertPos,long long sourceFirstBlock, long long lenght, bool isTake)
	{
	PENTER;
	AudioClip *clip = new AudioClip(this,
					audioSource,
					sourceFirstBlock,
					lenght,
					trackInsertPos);
	clip->isTake = isTake;
	AudioClip* c;
	AudioClip* c2;

	if (!clipList)
		{
		long long clipLenght = clip->sourceLastBlock - clip->sourceFirstBlock;
		/*PMESG("Adding first clip :  [ TFB = " << clip->trackFirstBlock <<
					" , TLB = " << clip->trackLastBlock <<
					" , SFB = " << clip->sourceFirstBlock <<
					" , SLB = " << clip->sourceLastBlock <<
					" , LEN = " << clipLenght << " ]");
		*/
		clipList = clip;
		clip->prev=0;
		clip->next=0;
		parentSong->update_last_block();
		}
	else
		{
		c=clipList;
		c2=clipList;
		while (c &&  (c->trackFirstBlock < trackInsertPos))
			{
			c2=c;
			c=c->next;
			}
		if (!c)
			{
			PMESG("Adding at the end of list");
			c2->next=clip;
			clip->prev=c2;
			clip->next=0;
			parentSong->update_last_block();
			}
		else
			{
			if (c==clipList)
				{
				PMESG("Adding at the begin of list");
				clip->next=clipList;
				clipList->prev=clip;
				clip->prev=0;
				clipList=clip;
				}
			else
				{
				PMESG("Adding in the middle of list");
				clip->next=c2->next;
				clip->prev=c2;
				c2->next->prev=clip;
				c2->next=clip;
				}
			}
		}
	PEXIT;
	return clip;
	}

void Track::activate()
	{
	if (!isActive)
		{
		isActive=true;
		if (parentSong->get_editing_mode()==Song::EDITING_MODE_TRACK_CURVES)
			{
			int cfc  = filterChain->currentFilterController;
			if ((cfc>=0) && (filterChain->filterController[cfc]))
				filterChain->filterController[cfc]->start_blinking();
			}
		}

	}


void Track::deactivate()
	{
	if (isActive)
		{
		for (int k=0; k<MAX_FILTER_CONTROLLERS; k++)
			if (filterChain->filterController[k])
				filterChain->filterController[k]->deactivate();
		isActive=false;
		}
	}




void Track::set_blur(bool stat)
	{
	ColorManager::set_blur(ColorManager::TRACK_BG , stat);
	ColorManager::set_blur(ColorManager::TRACK_BG_ACTIVE, stat);
	AudioClip* c=clipList;
	while (c)
		{
		c->set_blur(stat);
		c=c->next;
		}
	}


int Track::get_baseX() // the implementation for track is simply return 0
	{
	return 0;
	}

int Track::get_baseY()
	{
	return baseY;
	}

int Track::get_width() // the implementation for tracks is simply return the drawArea widths
	{
	return drawArea->width();
	}


int Track::get_height()
	{
	return height;
	}


QString Track::get_name()
	{
	return name;
	}



int Track::arm()
	{
	PENTER;
	if ((mixer->get_status()==Mixer::STOPPED) ||
	(mixer->get_status()==Mixer::ARMED))
		{
		QDate d = QDate::currentDate();
		QTime t = QTime::currentTime();
		QString s1;s1.setNum(d.year());
		QString s2;s2.setNum(d.month());
		QString s3;s3.setNum(d.day());
		QString s4;s4.setNum(t.hour());
		QString s5;s5.setNum(t.minute());
		QString s6;s6.setNum(t.second());
		QString tns = "take-"+s1+s2+s3+"-"+s4+s5+s6+".praf";
		recordingAudioFilename = tns;
		if (mixer->arm(busIn,whichChannels)<0)
			{
			PERROR("Cannot arm track");//improve me
			PEXIT;
			return -1;
			}
		start_recording_clip();
		isArmed=true;
		recreate();
		}
	PEXIT;
	return 1;
	}




int Track::disarm()
	{
	PENTER;
	if ((mixer->get_status()==Mixer::STOPPED) || (mixer->get_status()==Mixer::ARMED)) // this avoid the disarm/arm during going state
		{
		isArmed=false;
		if (mixer->disarm(busIn)<0)
			{
			PERROR("Cannot disarm track");
			PEXIT;
			return -1;
			}
		abort_recording_clip();
		recreate();
		}
	PEXIT;
	return 1;
	}


int Track::cleanup_buffers() // This is called everytime Mixer stops.
	{
	filterChain->cleanup();
	int r=1;
	if (isArmed)
		{
		r=finish_recording_clip();
		QDate d = QDate::currentDate();
		QTime t = QTime::currentTime();
		QString s1;s1.setNum(d.year());
		QString s2;s2.setNum(d.month());
		QString s3;s3.setNum(d.day());
		QString s4;s4.setNum(t.hour());
		QString s5;s5.setNum(t.minute());
		QString s6;s6.setNum(t.second());
		QString tns = "take-"+s1+s2+s3+"-"+s4+s5+s6+".praf";
		recordingAudioFilename = tns;
		start_recording_clip();
		}
	return r;
	}


Song* Track::get_parent_song()
	{
	return parentSong;
	}


int Track::get_bus_in()
	{
	return busIn;
	}

int Track::get_bus_out()
	{
	return busOut;
	}

int Track::get_which_channels()
	{
	return whichChannels;
	}

void Track::set_which_channels(int wc)
	{
	whichChannels = wc; // TODO : validate it..
	}

void Track::set_bus_in(int b)
	{
	bool wasArmed=isArmed;
	if (isArmed)
		disarm();
	busIn=b;
	if (wasArmed)
		arm();
	}

void Track::set_bus_out(int b)
	{
	busOut=b;
	}


bool Track::is_muted()
	{
	return isMuted;
	}

bool Track::is_solo()
	{
	return isSolo;
	}


bool Track::armed()
	{
	return isArmed;
	}

bool Track::is_active()
	{
	return isActive;
	}

void Track::toggle_active()
	{
	isActive=!isActive;
	}

QString Track::get_schema()
	{
	PENTER;
	QString sint;
	QString output="";
	AudioClip* cli;
	sint.setNum(ID);
	output = output + "<track id=\"" + sint + "\"";
	output = output + " name=\"" + name + "\"";
	QString sgain; sgain.setNum(gain,'g',3);
	QString span; span.setNum(pan,'g',3);
	output = output + " gain=" + sgain + " pan="+ span;
	output = output + " mute=" + (isMuted?"1":"0") + " solo=" + (isSolo?"1":"0");
	QString sBusIn; sBusIn.setNum(busIn);
	QString sBusOut; sBusOut.setNum(busOut);
	output = output + " busIn="+ sBusIn + " busOut=" + sBusOut;
	output = output +  ">\n";

	cli = clipList;
	QString tag;
	while (cli)
		{
		if (cli->isTake)
			tag="take";
		else
			tag="source";
		output = output +  "       <clip " + tag + "=\"" + cli->audioSource->get_pure_filename() + "\"";

		sint.setNum((double)cli->trackFirstBlock);
		output = output +  " TFB=" + sint;
		sint.setNum((double)cli->sourceFirstBlock);
		output = output +  " SFB=" + sint;
		sint.setNum((double)cli->get_lenght());
		output = output +  " LEN=" + sint;

		QString sx;
		sx.setNum((float)cli->gain);
		output = output +  " gain=" + sx;
		sx.setNum((double)cli->fadeInBlocks);
		output = output +  " fadeIn=" + sx;
		sx.setNum((double)cli->fadeOutBlocks);
		output = output +  " fadeOut=" + sx  + " >\n";

		cli=cli->next;
		}
	output = output +  filterChain->get_schema();
	output = output +  "</track>\n";
	PEXIT;
	return output;
	}


void Track::set_schema(QString schema)
	{
	PENTER;
	// TO IMPROVE : The parsing is too weak. If must consider all separators (not only SPACE)
	// and must work in any order.
	QTextStream ts( &schema, IO_ReadOnly );
	while (!ts.atEnd())
		{
		QString line = ts.readLine();

		int a=line.find("name=\"")+6;
		int b=line.find("\"",a);
		name=line.mid(a,b-a);

		a=line.find("mute=")+5;
		b=line.find(" ",a+1);
		QString sMute=line.mid(a,b-a);
		isMuted=(sMute=="1");

		a=line.find("solo=")+5;
		b=line.find(" ",a+1);
		QString sSolo=line.mid(a,b-a);
		isSolo=(sSolo=="1");

		a=line.find("gain=")+5;
		b=line.find(" ",a+1);
		QString sGain=line.mid(a,b-a);
		set_gain(sGain.toFloat());

		a=line.find("pan=")+4;
		b=line.find(" ",a+1);
		QString sPan=line.mid(a,b-a);
		set_pan(sPan.toFloat());

		a=line.find("busIn=")+6;
		b=line.find(" ",a+1);
		QString sBusIn=line.mid(a,b-a);
		set_bus_in(sBusIn.toInt());

		a=line.find("busOut=")+7;
		b=line.find(">",a+1);
		QString sBusOut=line.mid(a,b-a);
		set_bus_out(sBusOut.toInt());

		while ((line.find("</track")<0) && (!ts.atEnd()))
			{
			line = ts.readLine();
			line = line.stripWhiteSpace();
			if (line.find("<clip")>=0)
				{
				bool isTake;
				int x1 = line.find("source=");
				if (x1<0)
					{
					x1 = line.find("take=")+6;
					isTake=true;
					}
				else
					{
					x1+=8;
					isTake=false;
					}
				int x2 = line.find("\"", x1+1);
				QString sSource=line.mid(x1,x2-x1);
				int x3 = line.find("TFB=")+4;
				int x4 = line.find(" ", x3+1);
				QString sTFB=line.mid(x3,x4-x3);
				int x5 = line.find("SFB=")+4;
				int x6 = line.find(" ", x5+1);
				QString sSFB=line.mid(x5,x6-x5);
				int x7 = line.find("LEN=")+4;
				int x8 = line.find(" ", x7+1);
				QString sLEN=line.mid(x7,x8-x7);
				long long tfb = (long long) sTFB.toDouble();
				long long sfb = (long long) sSFB.toDouble();
				long long len = (long long) sLEN.toDouble();
				int k1 = line.find("gain=")+5;
				int k2 = line.find(" ", k1+1);
				QString sGain=line.mid(k1,k2-k1);
				int k3 = line.find("fadeIn=")+7;
				int k4 = line.find(" ", k3+1);
				QString sFadeIn=line.mid(k3,k4-k3);
				int k5 = line.find("fadeOut=")+8;
				int k6 = line.find(" ", k5+1);
				QString sFadeOut=line.mid(k5,k6-k5);

				AudioClip* clip;
				PMESG("Creating clip from [%s] TFB=%d SFB=%d LEN=%d",(const char*) sSource.latin1(),tfb,sfb,len );
				Audio* audioSource = parentSong->audioSourcesList->get_audio_for_source(sSource);
				if (!audioSource)
					{
					audioSource = new Audio(parentSong);
					int r = audioSource->build(parentSong->rootDir+(isTake?"/capture":"/audiosources"),sSource);
					if (r>0)
						{
						parentSong->audioSourcesList->add(audioSource);
						clip = add_clip(audioSource, tfb, sfb, len, isTake);
						}
					}
				else
					clip = add_clip(audioSource, tfb, sfb, len, isTake);
				clip->set_gain(sGain.toFloat());
				clip->set_fade_in((long long)sFadeIn.toDouble());
				clip->set_fade_out((long long)sFadeOut.toDouble());
				}
			else if (line.find("<filtercontroller")>=0)
				{
				int x1=line.find("type=")+6;
				int x2=line.find("\"",x1);
				QString fctype=line.mid(x1,x2-x1);
				FilterController* fc = filterChain->add_filter_controller(fctype);
				if (fc)
					{
					while ( fc && (line.find("</filtercontroller")<0) && (!ts.atEnd()))
						{
						line = ts.readLine();
						line = line.stripWhiteSpace();
						if (line.find("<curve")>=0)
							{
							int x1=line.find("type=")+6;
							int x2=line.find("\"",x1);
							QString cutype=line.mid(x1,x2-x1);
							PMESG("Adding Curve %s",(const char*)cutype.latin1());
							Curve* cu = fc->add_curve(cutype);
							if (!cu) // curve already exists. No need to re-add it.
								cu = fc->get_curve(cutype); // use the already existent curve
							while (cu && (line.find("</curve")<0) && (!ts.atEnd()))
								{
								line = ts.readLine();
								line = line.stripWhiteSpace();
								if (line.find("<node")>=0)
									{
									int x1=line.find("pos=")+4;
									int x2=line.find(" ",x1);
									QString spos=line.mid(x1,x2-x1);
									int x3=line.find("value=")+6;
									int x4=line.find(" ",x3);
									QString sval=line.mid(x3,x4-x3);
									long long pos= (long long) spos.toDouble();
									if (pos==0)
										{
										PMESG("Setting root node [%s@%s]",
											(const char*)sval.ascii(),
											(const char*)spos.ascii());
										}
									else
										{
										PMESG("Adding node : [%s@%s]",
											(const char*)sval.latin1(),
											(const char*)spos.latin1());
										}
									float val= sval.toFloat();
									CurveNode* no=cu->add_node(pos,val);
									}
								}
							}
						}
					}
				}
			}
		}
	PEXIT;
	}


int Track::start_recording_clip()
	{
	PENTER;
	int recordingChannels;
	if(whichChannels==MustuxAudioDeviceMapper::ONLY_LEFT_CHANNEL || whichChannels==MustuxAudioDeviceMapper::ONLY_RIGHT_CHANNEL || whichChannels==MustuxAudioDeviceMapper::MONO)
		recordingChannels = 1;
	else
		recordingChannels = 2;
	recordingAudio = new Audio(parentSong);
	recordingAudio->create(parentSong->captureDir, recordingAudioFilename, recordingChannels, parentSong->get_rate(), parentSong->get_bit_depth(),"Untitled", "UndefGroup", "UndefSoundCard","Real Time Take");
	recordingAudio->peak->init(recordingAudio);
	// IMPROVE LINE BELOW. File is already isolated within Audio
	recordingAudioFile = recordingAudio->file;
	recordingAudioFile->rewind_audio();
	recordingAudioStartPos = parentSong->get_working_block();
	recordingAudioPos = recordingAudioStartPos;
	updateTimer->start(50);
	PEXIT;
	return 1; // improve me
	}


int Track::abort_recording_clip()
	{
	PENTER;
	QString name = parentSong->captureDir + "/" + recordingAudioFilename;
	QFile file(name);
	if (!file.remove())				//file is not used, remove it from hard disk
		{
		PERROR("failed to remove file %s\n", name.latin1());
		}
	delete recordingAudio;
	updateTimer->stop();
	PEXIT;
	return 0;
	}



int Track::feed_recording_clip(char* buffer, int bufferSize)
	{
	PENTER4;
	recordingBufferSize = bufferSize;
	recordingBuffer = buffer;
	if ( recordingAudioFile->write_fragment( recordingBuffer, recordingBufferSize) < 0)
		{
		PERROR("Cannot write to output file.");
		PEXIT4;
		return -1;
		}
	recordingAudio->peak->process_samples_from_buffer(buffer, bufferSize);
	recordingAudioPos += recordingBufferSize / recordingAudioFile->blockSize;
	PEXIT4;
	return 1;
	}


int Track::finish_recording_clip()
	{
	PENTER;
	updateTimer->stop();
	int r=accept_recorded_clip();
	PEXIT;
	return r;
	}


void Track::update_recording_clip()
	{
	PENTER3;
	if (mixer->get_status()==Mixer::GOING)
		{
		PMESG2("Updating recording Audio %p",recordingAudio);
		int recordingChannels;
		if(whichChannels==MustuxAudioDeviceMapper::ONLY_LEFT_CHANNEL || whichChannels==MustuxAudioDeviceMapper::ONLY_RIGHT_CHANNEL || whichChannels==MustuxAudioDeviceMapper::MONO)
			recordingChannels = 1;
		else
			recordingChannels = 2;
		int rx = parentSong->block_to_xpos(recordingAudioPos);
		int totalSamples = recordingBufferSize/parentSong->get_block_size();
		int numSamplesToDraw = totalSamples/Peak::zoomStep[parentSong->get_hzoom()];
		QPainter* painter = drawArea->painter;
		if (( rx >= 0 ) && (rx <= drawArea->width()))
			{
			if ( rx + numSamplesToDraw > drawArea->width() )
				numSamplesToDraw = drawArea->width() - rx;
			int recordingBufferStep = parentSong->get_zoom_step()*parentSong->get_block_size();
			int sho,maxsho;
			int chhalf = (height-12) / recordingChannels;
			int x = rx - 2; // this 2 pixels is to prevent that the recording buffer being drawed is blurred by playing cursors, that is being drawed in the same position during record
			int sampleCounter=0;
			if (numSamplesToDraw==0)
				{
				recordingBufferStep = recordingBufferSize;
				numSamplesToDraw=1;
				}
			char* posBuffer = recordingBuffer;
			while ( sampleCounter < numSamplesToDraw )
				{
				for (int k=0; k < recordingChannels; k++)
					{
					int yc = real_baseY() + chhalf*(k+1) + 4 ;
					painter->setPen(Qt::white);
					painter->drawLine(x,yc,x,yc-chhalf+1);
					char* p = posBuffer + k*sizeof(short);
					maxsho=0;
					while ( p < (posBuffer + recordingBufferStep))
						{
						sho = (int)*(short*)p;
						if ( sho > maxsho ) maxsho=sho;
						p += sizeof(short)*recordingChannels;
						}
					if ( maxsho >= 32767 )
						{
						maxsho = 32767;
						painter->setPen(Qt::red);
						inputLevel[k]=maxsho;
						}
					else
						painter->setPen(QColor(150,40,40));
					float f = (float) maxsho / 32768;
					int y = (int) (f*chhalf);
					painter->drawLine(x,yc,x,yc-y);
					}
				posBuffer+=recordingBufferStep;
				x++;
				sampleCounter++;
				}
			}
		drawArea->update((rx-2),real_baseY(),(rx + numSamplesToDraw -2),height);
		}
	PEXIT3;
	}



int Track::accept_recorded_clip()
	{
	PENTER;
	recordingAudioFile->fix_audio_size();
	recordingAudio->build_peaks();	//WAS: rebuild_peaks(); As this is recorded audio, it's impossible that a peakfile allready exist, so just build it, not rebuild (rebuild removes the existing Peak Object, which make's it impossible to generate a peakfile on the fly. By Remon
	AudioClip* clip = add_clip(recordingAudio, recordingAudioStartPos, true); //WAS parentSong->create_clip(...) but nothing is done in Song::create_clip, only Track::add_clip, so why not calling it just from here? By Remon
	if (!clip)
		{
		PERROR("After recording, could not create clip with recorded audio! Continuing anyway ...");
		PEXIT;
		return -1;
		}
	lastRecordedClip = clip;
	parentSong->set_work_at(lastRecordedClip->trackLastBlock);
	parentSong->audioSourcesList->add(recordingAudio);
	PEXIT;
	return 1;
	}


void Track::discard_recorded_clip() //WAS int Track::dis...() return value was 1 By Remon
	{
	delete recordingAudio;
	}


float Track::get_gain()
	{
	return gain;
	}


float Track::get_pan()
	{
	return pan;
	}


void Track::set_gain(float pGain)
	{
	if ( pGain < -24.0 )
		gain=-24.0;
	else
	if ( pGain > 24.0 )
		gain=24.0;
	else
		gain=pGain;
	}


void Track::set_pan(float pPan)
	{
	if ( pPan < -1.0 )
		pan=-1.0;
	else
	if ( pPan > 1.0 )
		pan=1.0;
	else
		pan=pPan;
	}


void Track::vzoom_in(int newBaseY)
	{
	baseY = newBaseY;
	if (1.1*height<500.0)
		height = (int) (1.10*height);
	}


void Track::vzoom_out(int newBaseY)
	{
	baseY = newBaseY;
	if ( 0.9 * height > MINIMUM_HEIGHT )
		height = (int) (0.90 * height);
	}


int Track::set_baseY(int b)
	{
	baseY = b;
	}


int Track::set_height(int h)
	{
	height = h;
	}


int Track::get_total_clips()
	{
	int c=0;
	AudioClip* cli = clipList;
	while (cli) { cli=cli->next; c++; }
	return c;
	}


bool Track::visible()
	{
	if((baseY + height + parentSong->get_mta_baseY()) < 0 )
		return false;
	if ( (baseY + parentSong->get_mta_baseY()) < drawArea->height() )
		return true;
	return false;
	}


int Track::real_baseY() // return baseY with vertical offset applied
	{
	return baseY + parentSong->get_mta_baseY();
	}

int Track::get_mouse_x()
	{
	return parentSong->tracksArea->get_mouse_x();
	}

int Track::get_mouse_y()
	{
	return parentSong->tracksArea->get_mouse_y();
	}
// eof


