/*
  Copyright (C) 2011 Stefan Stäglich

  This file is part of pung.

  Pung is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  Pung is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with pung.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <gtest/gtest.h>
#include "./Paddle.h"
#include "./Pong.h"

// _____________________________________________________________________________
TEST(BallTest, constructor)
{
  Pong pong;
  Ball* ball = new Ball(3, &pong);
  ASSERT_EQ(3, ball->_id);
  ASSERT_EQ(&pong, ball->_pong);

  // reset()
  ASSERT_EQ(12, ball->_xSize);
  ASSERT_EQ(21, ball->_ySize);
  ASSERT_EQ(314, ball->_xPos);
  ASSERT_EQ(229.5, ball->_yPos);

  delete ball;
}

// _____________________________________________________________________________
TEST(BallTest, move)
{
  Pong pong;
  float startBPosX = pong._balls[0]->_xPos;
  float startBPosY = pong._balls[0]->_yPos;

  pong._speed = 7;
  pong._balls[0]->_xSize = 2;
  pong._balls[0]->_ySize = 2;
  pong._balls[0]->_xDiff = 7;
  pong._balls[0]->_yDiff = 7;
  pong._rightPaddle->_xPos = pong._wres - pong._leftPaddle->_xSize;

  // Case 1: Ball in field
  pong._balls[0]->_xPos = 20;
  pong._balls[0]->_yPos = 20;
  pong._balls[0]->move();
  ASSERT_EQ(27, pong._balls[0]->_xPos);
  ASSERT_EQ(27, pong._balls[0]->_yPos);
  ASSERT_EQ(7, pong._balls[0]->_xDiff);
  ASSERT_EQ(7, pong._balls[0]->_yDiff);
  ASSERT_EQ(2, pong._balls[0]->_xSize);
  ASSERT_EQ(2, pong._balls[0]->_ySize);

  // Case 2: Ball touchedBarrier
  pong._balls[0]->_yPos = 0;
  pong._balls[0]->_yDiff = -3;
  pong._balls[0]->move();
  ASSERT_EQ(34, pong._balls[0]->_xPos);
  ASSERT_EQ(-3, pong._balls[0]->_yPos);
  ASSERT_EQ(7, pong._balls[0]->_xDiff);
  ASSERT_EQ(3, pong._balls[0]->_yDiff);

  // Case 3: Ball touched left paddle
  pong._balls[0]->_xPos = pong._leftPaddle->_xSize;
  pong._balls[0]->_yPos = pong._leftPaddle->_yPos
    + pong._leftPaddle->_ySize / 2;
  pong._balls[0]->_xDiff = -3;
  pong._balls[0]->move();
  ASSERT_EQ(pong._leftPaddle->_xSize - 3, pong._balls[0]->_xPos);
  ASSERT_EQ(pong._leftPaddle->_yPos + pong._leftPaddle->_ySize / 2
      + 3, pong._balls[0]->_yPos);
  ASSERT_EQ(3, pong._balls[0]->_xDiff);
  ASSERT_TRUE(pong._speed >= pong._balls[0]->_yDiff);

  // Case 4: Ball touched right paddle
  pong._balls[0]->_xPos = pong._rightPaddle->_xPos;
  pong._balls[0]->_yPos = pong._rightPaddle->_yPos;
  pong._rightPaddle->_yDiff = 4;
  pong._balls[0]->_xDiff = 2;
  pong._balls[0]->_yDiff = 4;
  pong._balls[0]->move();
  ASSERT_EQ(pong._rightPaddle->_xPos + 2, pong._balls[0]->_xPos);
  ASSERT_EQ(pong._rightPaddle->_yPos + 4,
      pong._balls[0]->_yPos);
  ASSERT_EQ(-2, pong._balls[0]->_xDiff);
  ASSERT_TRUE(pong._speed >= pong._balls[0]->_yDiff);

  // Case 5: Ball out of field (left)
  pong._balls[0]->_xPos = -100;
  pong._balls[0]->_yPos = 4;
  pong._balls[0]->move();
  ASSERT_EQ(0, pong._leftPaddle->_score);
  ASSERT_EQ(1, pong._rightPaddle->_score);
  ASSERT_EQ(startBPosX, pong._balls[0]->_xPos);
  ASSERT_EQ(startBPosY, pong._balls[0]->_yPos);


  // Case 6: Ball out of field (right)
  pong._balls[0]->_xPos = 800;
  pong._balls[0]->_yPos = 4;
  pong._balls[0]->move();
  ASSERT_EQ(1, pong._leftPaddle->_score);
  ASSERT_EQ(1, pong._rightPaddle->_score);
  ASSERT_EQ(startBPosX, pong._balls[0]->_xPos);
  ASSERT_EQ(startBPosY, pong._balls[0]->_yPos);
}

// _____________________________________________________________________________
TEST(BallTest, inpulse)
{
  Pong pong;
  Ball* ball = new Ball(3, &pong);

  ball->_xDiff = 3;
  ball->_yDiff = 7;

  ball->inpulse(0);
  ASSERT_EQ(-3, ball->_xDiff);
  ASSERT_EQ(pong._speed, pong._balls[0]->_yDiff);

  ball->inpulse(2);
  ASSERT_EQ(3, ball->_xDiff);
  ASSERT_TRUE(pong._speed >= pong._balls[0]->_yDiff);

  ball->inpulse(5);
  ASSERT_EQ(-3, ball->_xDiff);
  ASSERT_TRUE(pong._speed >= pong._balls[0]->_yDiff);

  ball->inpulse(-8);
  ASSERT_EQ(3, ball->_xDiff);
  ASSERT_TRUE(pong._speed >= pong._balls[0]->_yDiff);

  delete ball;
}

// _____________________________________________________________________________
TEST(BallTest, reset)
{
  Pong pong;
  pong._hres = 80;
  pong._wres = 44;

  Ball* ball1 = new Ball(3, &pong);
  ball1->_xSize = 0;
  ball1->_ySize = 0;
  ball1->_xSize = 0;
  ball1->_ySize = 0;
  ball1->_xDiff = 0;
  ball1->_yDiff = 0;

  ball1->reset();
  ASSERT_EQ(2, ball1->_xSize);
  ASSERT_EQ(3, ball1->_ySize);
  ASSERT_EQ(21, ball1->_xPos);
  ASSERT_EQ(38.5, ball1->_yPos);

  delete ball1;
}
