/*
  Copyright (C) 2011 Stefan Stäglich

  This file is part of pung.

  Pung is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  Pung is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with pung.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <getopt.h>
#include <GL/glut.h>
#include <GL/gl.h>
#include <SDL/SDL.h>
#include <iostream>
#include "./Pong.h"

// _____________________________________________________________________________
Pong::Pong()
{
  _wres = 640;
  _hres = 480;
  _speed = 0;
  _maxScore = 3;

  // Create the other pong objects
  _balls.push_back(new Ball(0, this));
  _leftPaddle = new Paddle(this, SDLK_w, SDLK_s);
  _rightPaddle = new Paddle(this, SDLK_UP, SDLK_DOWN);
}

// _____________________________________________________________________________
Pong::~Pong()
{
  // Delete the balls
  size_t numOfBalls = _balls.size();
  for (size_t i = 0; i < numOfBalls; i++)
  {
    delete _balls[i];
    _balls[i] = NULL;
  }

  // Delete the paddles
  delete _leftPaddle;
  _leftPaddle = NULL;
  delete _rightPaddle;
  _rightPaddle = NULL;

  // Quit SDL
  SDL_Quit();
}

// _____________________________________________________________________________
bool Pong::parseCommandLineArguments(int argc, char** argv)
{
  struct option options[] =
  {
    {"width"          , 1, NULL, 'w'},
    {"height"         , 1, NULL, 'h'},
    {"paddle-size"    , 1, NULL, 'p'},
    {"speed"          , 1, NULL, 's'},
    {"max-score"      , 1, NULL, 'm'},
    {"number-of-balls", 1, NULL, 'b'},
    {NULL             , 0, NULL,  0 }
  };

  optind = 1;

  bool allFine = true;
  size_t numOfBalls = 1;

  while (true)
  {
    int c = getopt_long(argc, argv, "w:h:p:s:m:b:", options, NULL);
    if (c == -1) break;
    switch (c)
    {
      case 'w':
        _wres = atoi(optarg);
        break;
      case 'h':
        _hres = atoi(optarg);
        break;
      case 'p':
        _leftPaddle->_ySize = atoi(optarg);
        _rightPaddle->_ySize = atoi(optarg);
        break;
      case 's':
        _speed = atof(optarg);
        break;
      case 'm':
        _maxScore = atoi(optarg);
        break;
      case 'b':
        numOfBalls = atoi(optarg);
        for (size_t i = 1; i < numOfBalls; i++)
          _balls.push_back(new Ball(i, this));
        break;
      default:
        // For all other options return false
        allFine = false;
    }
  }

  // No argument should be left
  if (optind != argc)
  {
      std::cerr << "Invalid argument: " << argv[optind] << std::endl;
      return false;
  }
  return allFine;
}

// _____________________________________________________________________________
void Pong::printUsage()
{
  std::cerr << "Pung 0.2. Game speed and paddle size depends on the "
    << "resolution."
    << std::endl
    << "Warning: Standard speed might be too fast on some computers!"
    << std::endl
    << std::endl

    << "Controls:"
    << std::endl
    << "           left player      right player"
    << std::endl
    << "up         a                up"
    << std::endl
    << "down       d                down"
    << std::endl
    << std::endl

    << "Options:"
    << std::endl
    << "   -w, --width=WIDTH                 the width            default: 640"
    << std::endl
    << "   -h, --height=HEIGHT               the height           default: 480"
    << std::endl
    << "   -p, --paddle-size=PADDELSIZE      the paddle size"
    << std::endl
    << "   -s, --speed=SPEED                 the game speed"
    << std::endl
    << "   -m, --max-score=MAXSCORE          the maximal score    default:   3"
    << std::endl
    << "   -b, --number-of-balls=NUMBALLS    the number of balls  default:   1"
    << std::endl
    << std::endl

    << "To quit the game press escape."
    << std::endl;
}

// _____________________________________________________________________________
bool Pong::initializeScreen()
{
  // Initialize SDL video system
  if (SDL_Init(SDL_INIT_VIDEO) != 0)
  {
    std::cerr << "Unable to initialize SDL: " << SDL_GetError() << std::endl;
    return false;
  }

  // Enable OpenGL
  SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);

  // Set the color depth
  SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 5);
  SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 6);
  SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 5);

  // Set video mode
  if (SDL_SetVideoMode(_wres, _hres, 16, SDL_OPENGL) == NULL)
  {
    std::cerr << "Unable to set SDL video mode: "
      << SDL_GetError() << std::endl;
    return false;
  }

  // Set window title
  SDL_WM_SetCaption("Pung", "Pung");

  // Set OpenGL coordinate system
  // x-coordinate: 0 (left) .. _wres (right)
  // y-coordinate: 0 (top) .. _hres (bottom)
  gluOrtho2D(0, _wres, _hres, 0);

  // All ok
  return true;
}

// _____________________________________________________________________________
void Pong::draw()
{
  // Background colour (Uni blue)
  glClearColor(0, 0.3, 0.5, 0);
  glClear(GL_COLOR_BUFFER_BIT);

  // draw balls
  size_t numOfBalls = _balls.size();
  for (size_t i = 0; i < numOfBalls; i++)
    _balls[i]->draw();

  // draw paddles
  _leftPaddle->draw();
  _rightPaddle->draw();
  glFlush();
  SDL_GL_SwapBuffers();
}

// _____________________________________________________________________________
void Pong::playGame()
{
  reset();

  SDL_Event event;

  // Quit the game if a player has reached the _maxScore
  while (_leftPaddle->_score < _maxScore && _rightPaddle->_score < _maxScore)
  {
    // SDL input
    if (SDL_PollEvent(&event))
    {
      if (event.key.keysym.sym == SDLK_ESCAPE)
      {
        std::cout << "Good bye." << std::endl;
        break;
      }
      else
      {
        _leftPaddle->keyEvent(event);
        _rightPaddle->keyEvent(event);
      }
    }

    // Move balls
    size_t numOfBalls = _balls.size();
    for (size_t i = 0; i < numOfBalls; i++)
      _balls[i]->move();

    // Move paddles
    _leftPaddle->move();
    _rightPaddle->move();
    draw();
  }

  SDL_Delay(1000);

  // Regulary exit message
  if (_leftPaddle->_score != _rightPaddle->_score)
  {
    std::cout << "The winner is the ";
    if (_leftPaddle->_score > _rightPaddle->_score)
      std::cout << "left";
    else
      std::cout << "right";
    std::cout << " player. Restart for playing a new game." << std::endl;
  }
}

// _____________________________________________________________________________
void Pong::reset()
{
  if (_speed == 0)
    _speed = _wres / 100;

  _rightPaddle->_xPos = _wres - _leftPaddle->_xSize;

  // reset balls
  size_t numOfBalls = _balls.size();
  for (size_t i = 0; i < numOfBalls; i++)
    _balls[i]->reset();

  // reset paddles
  _leftPaddle->reset();
  _rightPaddle->reset();
}
