#    PyDBC: Contracts for Python 2.2+
#    Copyright (C) 2002 Daniel Arbuckle <djarb@highenergymagic.org>
#
#This library is free software; you can redistribute it and/or
#modify it under the terms of the GNU Lesser General Public
#License as published by the Free Software Foundation; either
#version 2.1 of the License, or (at your option) any later version.
#
#This library is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#Lesser General Public License for more details.
#
#You should have received a copy of the GNU Lesser General Public
#License along with this library; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

""" This module implements Contracts in Python
Specifically, preconditions and postconditions for methods,
along with class invariants, are supported.

To use this functionality, set the metaclass of your object
to dbc.DBC as follows:
    
    import dbc

    class foo:
	__metaclass__ = dbc.DBC

that's all there is to it.

You can set a default metaclass for a module by assigning
the global __metaclass__ variable.

If os.environ['PY_DBC'] is unset or set to an empty string, 
DBC functionality will be disabled.
"""

import os

def make_wrapped_call(fun, pre = None, post = None, invar = None):
    'Don\'t use this'
    
    if pre and post and invar:
	def wrapped(*args, **kw):
	    apply(pre, args, kw)
	    ret = apply(fun, args, kw)
	    post(args[0], ret)
	    invar(args[0])
	    return ret
	return wrapped
    if pre and post:
	def wrapped(*args, **kw):
	    apply(pre, args, kw)
	    ret = apply(fun, args, kw)
	    post(args[0], ret)
	    return ret
	return wrapped
    if pre and invar:
	def wrapped(*args, **kw):
	    apply(pre, args, kw)
	    ret = apply(fun, args, kw)
	    invar(args[0])
	    return ret
	return wrapped
    if post and invar:
	def wrapped(*args, **kw):
	    ret = apply(fun, args, kw)
	    post(args[0], ret)
	    invar(args[0])
	    return ret
	return wrapped
    if pre:
	def wrapped(*args, **kw):
	    apply(pre, args, kw)
	    return apply(fun, args, kw)
	return wrapped
    if post:
	def wrapped(*args, **kw):
	    ret = apply(fun, args, kw)
	    post(args[0], ret)
	    return ret
	return wrapped
    if invar:
	def wrapped(*args, **kw):
	    ret = apply(fun, args, kw)
	    invar(args[0])
	    return ret
	return wrapped
    return fun
    
class DBC(type):
    'Metaclass providing Contract behavior. See module docs for usage.'
    
    def __init__(cls, clsname, bases, dict):
	super(DBC, cls).__init__(clsname, bases, dict)
	try:
	    if not os.environ['PY_DBC']: return
	except KeyError:
	    return
	
	try:
	    invar = dict['_%s__invar' % clsname]
	except KeyError:
	    invar = None
	
	for name in dict.keys():
	    feature = dict[name]
	    if not callable(feature): continue
	    
	    try:
                if name[:2] == '__':
                    pre = dict['_%s%s__pre' % (clsname, name)]
                else:
                    pre = dict['%s__pre' % name]
	    except KeyError:
		pre = None
		
	    try:
                if name[:2] == '__':
                    post = dict['_%s%s__post' % (clsname, name)]
                else:
                    post = dict['%s__post' % name]
	    except KeyError:
		post = None
	    
	    setattr(cls, name, make_wrapped_call(feature, pre, post, invar))
	
	if invar:
	    irish_setter = getattr(cls, '__setattr__')
	    def setter(self, attr, val):
		irish_setter(self, attr, val)
		invar(self)
		
	    setattr(cls, '__setattr__', setter)
