#
##
##  This file is part of pyFormex 1.0.5  (Sat Feb 16 10:40:32 CET 2019)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: http://pyformex.org
##  Project page:  http://savannah.nongnu.org/projects/pyformex/
##  Copyright 2004-2018 (C) Benedict Verhegghe (benedict.verhegghe@ugent.be)
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##
"""pyFormex GUI module initialization.

This module is intended to form a single access point to the Python
wrappers for the QT libraries, which form the base of the pyFormex GUI.
By using a single access point, we can better deal with API changes
in the wrappers.

All pyFormex modules accessing QT libraries should do this by importing
from this module.

This module also detects the underlying windowing system.
Currently, the pyFormex GUI is only guaranteed on X11.
For other systems, a warning will be printed that some things may not work.
"""
from __future__ import absolute_import, division, print_function


import pyformex as pf
from pyformex import utils

BINDINGS = ('pyside', 'pyside2', 'pyqt4', 'pyqt5')
bindings = pf.cfg.get('gui/bindings','any').lower()

if bindings in BINDINGS:
    utils.requireModule(bindings)

else:
    # Try using any existing bindings
    bindings = ''
    for b in BINDINGS:
        if utils.hasModule(b):
            bindings = b
            break

#print(bindings)

if bindings == 'pyside':
    from PySide import QtCore, QtGui, QtOpenGL
    import PySide.QtGui as QtWidgets
    from PySide.QtCore import Signal, Slot

elif bindings == 'pyside2':
    from PySide2 import QtCore, QtGui, QtOpenGL, QtWidgets
    from PySide2.QtCore import Signal, Slot

elif bindings == 'pyqt4':
    import sip
    try:
        sip.setapi('QDate', 2)
        sip.setapi('QDateTime', 2)
        sip.setapi('QString', 2)
        sip.setapi('QTextStream', 2)
        sip.setapi('QTime', 2)
        sip.setapi('QUrl', 2)
        sip.setapi('QVariant', 2)
    except ValueError as e:
        raise RuntimeError('Could not set PyQt4 API version (%s)' % e)

    from PyQt4 import QtCore, QtGui, QtOpenGL
    import PyQt4.QtGui as QtWidgets
    from PyQt4.QtCore import pyqtSignal as Signal
    from PyQt4.QtCore import pyqtSlot as Slot

else:
    bindings = ''

if not bindings:
    raise ValueError("\n"+"*"*40+"\nThe pyFormex GUI requires PySide, PySide2 or PyQt4.\nI could not find neither of them, so I can not continue.\nInstall PySide, PySide2 or PyQt4 (including its OpenGL component)\nand then retry."+"\n"+"*"*40)

def loadX11Colors(filename):
    """Load the X11 colors"""
    x11colors = {}
    with open(filename) as f:
        for line in f:
            s = line.strip().split()
            name = ''.join(s[3:]).lower()
            try:
                x11colors[name] = tuple( int(i) for i in s[:3] )
            except:
                pass

    return x11colors



pf.X11colors = loadX11Colors("/etc/X11/rgb.txt")
pf.X11 = len(pf.X11colors) > 0


# if bindings=='pyside2':
#     utils.warn("warn_pyside2")
#     from PySide2.QtX11Extras import QX11Info as x11
#     pf.X11 = x11.isPlatformX11() # DOES NOT WORK!!!
#     pf.X11 = True
#     pf.X11colors = loadX11Colors("/etc/X11/rgb.txt")

# else:

#     try:
#         QtGui.QColor.setAllowX11ColorNames(True)
#         pf.X11 = True
#         pf.X11colors = {}
#     except:
#         print("WARNING: THIS IS NOT AN X11 WINDOW SYSTEM!")
#         print("SOME THINGS MAY NOT WORK PROPERLY!")
#         pf.X11 = False
#         pf.X11colors = {}

# End
