#
##
##  This file is part of pyFormex 1.0.7  (Mon Jun 17 12:20:39 CEST 2019)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: http://pyformex.org
##  Project page:  http://savannah.nongnu.org/projects/pyformex/
##  Copyright 2004-2019 (C) Benedict Verhegghe (benedict.verhegghe@ugent.be)
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##
"""pyFormex apps initialization.

This module contains the functions to detect and load the pyFormex
applications.
"""
from __future__ import absolute_import, division, print_function

import os, sys

import pyformex as pf
if pf.PY3:
    from importlib import reload
from pyformex import utils
from pyformex import Path


# Guess the appdir from the name
# This works for dirs having a matching cfg['NAMEdir'] entry
#
guessDir = lambda n, s: Path(s) if s else pf.cfg['%sdir' % n.lower()]

# global variable used of tracing application load errors
_traceback = ''

class AppDir(object):
    """Application directory

    An AppDir is a directory containing pyFormex applications.
    When creating an AppDir, its path is added to sys.path
    """

    def __init__(self, path, name=None, create=True):
        path = guessDir(name, path)
        self.path = checkAppdir(path)
        if self.path is None:
            raise ValueError("Invalid application path %s" % self.path)

        # Add the parent path to sys.path if it is not there
        parent = self.path.parent
        self.added = parent not in sys.path
        if self.added:
            sys.path.insert(1, str(parent))

        self.pkg = self.path.name
        if name is None:
            self.name = self.pkg.capitalize()
        else:
            self.name = name
        #print("DEBUG: %s %s" % (self.pkg,self.name))
        pf.debug("Created %s" % self, pf.DEBUG.APPS)

    def __repr__(self):
        return "AppDir %s at %s (%s)" % (self.name, self.path, self.pkg)


def setAppDirs():
    """Set the configured application directories"""
    # If this is a reset, first remove sys.path components
    try:
        for p in pf.appdirs:
            if p.added:
                sys.path.remove(str(p.path.parent))
        print('SYSPATH IS NOW:', sys.path)
    except:
        pass

    pf.appdirs = [AppDir(i[1], i[0]) for i in pf.cfg['appdirs']]
    for p in pf.appdirs:
        pf.debug(str(p), pf.DEBUG.CONFIG)


def checkAppdir(d):
    """Check that a directory d can be used as a pyFormex application path.

    If the path does not exist, it is created.
    If no __init__.py exists, it is created.
    If __init__.py exists, it is not checked.

    If successful, returns the path, else None
    """
    d = Path(d)
    if not d.exists():
        os.makedirs(str(d))

    initfile = d / '__init__.py'
    if not initfile.exists():
        try:
            with initfile.open('w') as f:
                f.write("""#
\"\"\"pyFormex application directory.

Do not remove this file. It is used by pyFormex to flag the parent
directory as a pyFormex application path.
\"\"\"
# End
""")
        except:
            pass
    if initfile.exists():
        return d
    else:
        print("Invalid appdir %s" % d)
        return None


def findAppDir(path):
    """Return the AppDir for a given path"""
    for p in pf.appdirs:
        if p.path == path:
            return p


def load(appname, refresh=False, strict=False):
    """Load the named app

    If refresh is True, the module will be reloaded if it was already loaded
    before.
    On succes, returns the loaded module, else returns None.
    In the latter case, if the config variable apptraceback is True, the
    traceback is store in a module variable _traceback.
    """
    global _traceback
    if not appname:
        raise RuntimeError("No appname specified!")
    pf.debug("Loading %s with refresh=%s" % (appname, refresh), pf.DEBUG.APPS)
    print("Loading application %s " % appname)
    try:
        _traceback = ''
        __import__(appname)
        app = sys.modules[appname]
        if refresh:
            reload(app)
        if strict:
            if not hasattr(app, 'run') or not callable(app.run):
                return None
        return app
    except:
        import traceback
        _traceback = traceback.format_exc()
        return None


def findAppSource(app):
    """Find the source file of an application.

    app is either an imported application module (like: pkg.mod)
    or the corresponding application module name(like: 'pkg.mod').
    In the first case the name is extracted from the loaded module.
    In the second case an attempt is made to find the path that the module
    would be loaded from, without actually loading the module. This can
    be used to load the source file when the application can not be loaded.
    """
    from types import ModuleType
    if isinstance(app, (str, ModuleType)):
        path = pf.findModuleSource(app)
        if path:
            return Path(path)
        else:
            return None
    else:
        raise ValueError("app should be a module or a module name")


def unload(appname):
    """Try to unload an application"""
    name = 'apps.'+appname
    if name in sys.modules:
        app = sys.modules[name]
        refcnt = sys.getrefcount(app)
        if refcnt == 4:
            pf.debug("Unloading %s" % name, pf.DEBUG.APPS)
            ## k = globals().keys()
            ## k.sort()
            ## print k
            del globals()[appname]
            del sys.modules[name]
            ## refcnt = sys.getrefcount(app)
            ## print refcnt
            del app
        else:
            print("Can not unload %s" % name)
    else:
        print("Module %s is not loaded" % name)


def listLoaded():
    return sorted([m for m in sys.modules if m.startswith('apps.')])


def detect(appdir):
    """Detect the apps present in the specified appdir.

    Parameters:

    - `appdir`: path to an appdir (i.e. a directory containing a file
      '__init__.py'

    Return a list with all the pyFormex apps in that appdir.

    If a file '.apps.dir' exists in the appdir, the returned list is the
    contents of that file. Otherwise the list contains all '.py' files
    in the directory, without the '.py' extension and sorted.
    """
    pf.debug("Detect apps in %s" % appdir, pf.DEBUG.APPS)
    # Detect, but do not load!!!!
    # because we are using this on import (before some applications can load)
    appdir = Path(appdir)
    appsdirfile = appdir / '.apps.dir'
    if appsdirfile.exists():
        pf.debug("Detect apps: read %s" % appsdirfile, pf.DEBUG.APPS)
        with appsdirfile.open('r') as fil:
            apps = fil.readlines()
            return [a.strip('\n').strip() for a in apps]
    else:
        pf.debug("Detect apps: scan %s" % appdir, pf.DEBUG.APPS)

        files = appdir.listTree(listdirs=False, excludedirs=['.*'], includefiles=[r'.*\.py$'])
        apps = [f.stem for f in files if f.stem[0] not in '._']
        apps = sorted(apps)
        pf.debug("%s" % apps, pf.DEBUG.APPS)
        return apps



# End
