#
##
##  This file is part of pyFormex 2.0  (Mon Sep 14 12:29:05 CEST 2020)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: http://pyformex.org
##  Project page:  http://savannah.nongnu.org/projects/pyformex/
##  Copyright 2004-2020 (C) Benedict Verhegghe (benedict.verhegghe@ugent.be)
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##
#
"""Setup script for pyFormex

To install pyFormex: python3 setup.py install --prefix=/usr/local
To uninstall pyFormex: pyformex --remove
"""
import os,sys

# Detect platform
if sys.version_info < (3, 6):
    raise RuntimeError("Python version is %s.%s but pyFormex requires "
                       ">= 3.6." % sys.version_info[:2])

from glob import glob

from distutils.command.build_py import build_py as _build_py
from distutils.command.sdist import sdist as _sdist
from distutils.core import setup, Extension
from distutils import filelist
from distutils.util import get_platform
from distutils.util import change_root, convert_path


# define the things to include
import manifest

# pyFormex release
__RELEASE__ = '2.0.1'

# The acceleration libraries
LIB_MODULES = [ 'misc_', 'nurbs_' ]

ext_modules = [Extension('pyformex/lib/%s'%m,
                         sources = ['pyformex/lib/%s.c'%m],
                         # optional=True,
                         ) for m in LIB_MODULES ]


class BuildFailed(Exception):

    def __init__(self):
        self.cause = sys.exc_info()[1] # work around py 2/3 different syntax

def status_msgs(*msgs):
    """Print status messages"""
    print('*' * 75)
    for msg in msgs:
        print(msg)
    print('*' * 75)


class build_py(_build_py):
    description = "Custom build_py preserving file mode in bin subdirectory. Also provides the install_type user option."

    def build_package_data (self):
        """Copy data files into build directory

        The default Python distutils do not preserve the file mode
        when copying the Python package.
        This version will preserve the file mode for files in the
        packages `bin` subdirectory. Thus executable scripts there
        will remain executable.
        """
        lastdir = None
        for package, src_dir, build_dir, filenames in self.data_files:
            for filename in filenames:
                target = os.path.join(build_dir, filename)
                self.mkpath(os.path.dirname(target))
                self.copy_file(
                    os.path.join(src_dir, filename), target,
                    preserve_mode = (
                        filename.startswith('bin/') or
                        filename.endswith('.sh')
                        )
                    )
            # if package == 'pyformex':
            #     # Create some additional symlinks
            #     for d in ['ref','_modules']:
            #         src = '../_static'
            #         tgt = os.path.join(build_dir,'doc','html',d,'_static')
            #         print("Creating symlink %s <- %s"% (src,tgt))
            #         os.symlink(src,tgt)

        print("Package data built!")
        #print(self.build_temp)


    # Override to remove the non-maching compat module
    def find_package_modules(self, package, package_dir):
        self.check_package(package, package_dir)
        module_files = glob(os.path.join(package_dir, "*.py"))
        modules = []
        setup_script = os.path.abspath(self.distribution.script_name)

        for f in module_files:
            abs_f = os.path.abspath(f)
            if abs_f != setup_script:
                module = os.path.splitext(os.path.basename(f))[0]
                modules.append((package, module, f))
            else:
                self.debug_print("excluding %s" % setup_script)
        return modules



class sdist(_sdist):

    def get_file_list(self):
        """Create list of files to include in the source distribution

        Create the list of files to include in the source distribution,
        and put it in 'self.filelist'.  This might involve
        reading the manifest template (and writing the manifest), or just
        reading the manifest, or just using the default file set -- it all
        depends on the user's options.
        """
        self.filelist = filelist.FileList()
        self.filelist.files = manifest.DIST_FILES
        self.filelist.sort()
        self.filelist.remove_duplicates()
        self.write_manifest()



def run_setup(with_cext):
    kargs = {}
    if with_cext:
            kargs['ext_modules'] = ext_modules

    # PKG_DATA, relative from pyformex path

    INCLUDE = []

    import numpy
    INCLUDE.append(numpy.get_include()),

    with open('Description') as file:
        long_description = file.read()

    PKG_DATA = [
        'pyformexrc',
        'icons/README',
        'icons/*.xpm',
        'icons/pyformex*.png',
        'icons/64x64/*',
        'fonts/*',
        'examples/apps.cat',
        'bin/*',
        'data/*',
        'glsl/*',
        'extra/Makefile',
        'extra/*/*',
        'scripts/*',
        ]

    PKG_DATA += [ i[9:] for i in manifest.DOC_FILES ]
    setup(cmdclass={
        'build_py': build_py,
        'sdist':sdist
        },
          name='pyformex',
          version=__RELEASE__,
          description='program to create 3D geometry from Python scripts',
          long_description=long_description,
          author='Benedict Verhegghe',
          author_email='benedict.verhegghe@ugent.be',
          url='http://pyformex.org',
          download_url='http://download.savannah.gnu.org/releases/pyformex/pyformex-%s.tar.gz' % __RELEASE__,
          license='GNU General Public License (GPL)',
          packages=[
              'pyformex',
              'pyformex.gui',
              'pyformex.gui.menus',
              'pyformex.lib',
              'pyformex.opengl',
              'pyformex.plugins',
              'pyformex.appsdir',
#              'pyformex.scripts',  # this is not a package!
              'pyformex.examples',
              'pyformex.fe',
              'pyformex.freetype',
              'pyformex.freetype.ft_enums',
              'pyformex.vtk_light',
              'pyformex.vtk_light.util',
          ],
          package_data={ 'pyformex': PKG_DATA },
          scripts=['pyformex/pyformex'],
          data_files=manifest.OTHER_DATA,
          classifiers=[
              'Development Status :: 4 - Beta',
              'Environment :: Console',
              'Environment :: X11 Applications :: Qt',
              'Intended Audience :: End Users/Desktop',
              'Intended Audience :: Science/Research',
              'Intended Audience :: Education',
              'License :: OSI Approved :: GNU General Public License (GPL)',
              'Operating System :: POSIX :: Linux',
              'Operating System :: POSIX',
              'Operating System :: OS Independent',
              'Programming Language :: Python',
              'Programming Language :: C',
              'Topic :: Multimedia :: Graphics :: 3D Modeling',
              'Topic :: Multimedia :: Graphics :: 3D Rendering',
              'Topic :: Scientific/Engineering :: Mathematics',
              'Topic :: Scientific/Engineering :: Visualization',
              'Topic :: Scientific/Engineering :: Physics',
              ],
#          requires=['numpy','OpenGL','PyQt4 | PySide'],
          include_dirs=INCLUDE,
          **kargs
          )


# Detect the --no-accel option
try:
    i = sys.argv.index('--no-accel')
    del(sys.argv[i])
    accel = False
except ValueError:
    accel = True


# Try with compilation
if accel:
    try:
        run_setup(accel)
    except BuildFailed:
        exc = sys.exc_info()[1] # work around py 2/3 different syntax
        status_msgs(
            exc.cause,
            "WARNING: The acceleration library could not be compiled, "
            "I will retry without them.")
        sys.exit()

else:
    # Run without compilation
    run_setup(False)
    status_msgs("WARNING: Building without the acceleration library")

# Final messages

status_msgs("NOTE: You should also run the command 'make -C pyformex/extra build install' to build and install some extra executables used by pyFormex.")

# End
