#
##
##  This file is part of pyFormex 2.4  (Thu Feb 25 13:39:20 CET 2021)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: http://pyformex.org
##  Project page:  http://savannah.nongnu.org/projects/pyformex/
##  Copyright 2004-2020 (C) Benedict Verhegghe (benedict.verhegghe@ugent.be)
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##
"""Specialized dialogs for the pyFormex GUI.

This module provides some dialogs that collect specific input for some
pyFormex GUI function. Most of these dialogs are normally accessed from
the GUI menus. But the user can also use these in his scripts.
"""
import pyformex as pf
from pyformex.path import Path
from pyformex.gui.widgets import Dialog, _I, _C, _G, _T
from pyformex.gui import image



class FileDialog(Dialog):
    """A customizable File Dialog.

    This is a specialized Dialog for the input of file or directory path
    using the standard FileDialog embedded in a pyFormex Dialog.
    This is work in progress and will become the default FileDialog when
    ready. It is already usable though.
    See :func:`plugins.geometry_menu.importGeometry` for an example.

    Parameters
    ----------
    path: :term:`path_like`
        The initial path displayed in the file selector widget.
    filter: list of str
        The filters for the files selectable in the file selecter widget.
    multi: bool
        If True, the dialog will show the multisave option initially
        checked.
    extra: list
        A list of input items to add to the FileDialog

    """
    def __init__(self, path='.', filter='*', exist=False, multi=False,
                 dir=False, compr=False, extra=[], enablers=[], **kargs):
        """Create the dialog."""
        path = Path(path)

        items=[
            _I('filename', path, itemtype='file', filter=filter, exist=False,
               dir=False, text="Filename:", ),
            ] + extra

        super().__init__(items=items, enablers=enablers, modal=False,
                         caption="pyFormex File Dialog")


class SaveImageDialog(Dialog):
    """A dialog for saving an image to a file.

    This is a specialized Dialog for the input of all data required to
    save the current pyFormex rendering to an image file. It is a
    convenient interactive frontend for the :func:`image.saveImage`
    function. The Dialog ask for the target file name and all the
    other parameters accepted by that function.

    Parameters
    ----------
    path: :term:`path_like`
        The initial path displayed in the file selector widget.
        See FileDialog.
    filer: list of str
        The filters for the files selectable in the file selecter widget.
        See FileDialog.
    multi: bool
        If True, the dialog will show the multisave option initially
        checked.

    """
    default_size = None

    def __init__(self, path=None, filter=None, multi=False):
        """Create the dialog."""
        if path is None:
            path = pf.cfg['workdir']
        if filter is None:
            filter = ['img', 'icon', 'all']
        if SaveImageDialog.default_size is None:
            # Late initialization because pf.canvas needed
            SaveImageDialog.default_size = pf.canvas.getSize()

        items=[
            _I('filename', path, itemtype='file', filter=filter, exist=False,
               dir=False, text="Filename:", ),
            _C('',[
                _I('extent', choices=image.extent_choices,
                   func=self.change_extent,
                   text="Extent:",
                   tooltip="The part(s) of the pyFormex window to be saved"),
                _I('tool', choices=image.tool_choices,
                   text="Tool:",
                   tooltip="The tool to be used for saving the image."
                   " The possible extents and formats depend on it."),
                _I('set_size', choices=['No', 'Width', 'Height', 'Both'],
                   text='Set Size:',
                   tooltip="Adjust one or both image dimensions. Beware, "
                   " this may give incorrect results if transparency is used."),
                _I('size', itemtype='ivector',
                   value=SaveImageDialog.default_size,
                   fields=['W', 'H'],
                   text="Size:",
                   tooltip="The size of the save image."),
            ]),
            _C('',[
                _I('format', choices=['From Extension'],
                   text="Format:",
                   tooltip="The image format to be used. Normally derived from"
                   "extension"),
                _I('quality', -1, min=-1, max=100,
                   text="Quality:",
                   tooltip="For compressed image formats (like JPG), specifies"
                   " the compression level (PNG:0..9; JPEG: 1..100"),
                _I('alpha', False, text="Keep Alpha",
                   tooltip="Keep the alpha channel in the result (Experimental!)"),
            ], spacer='l'),
            _C('',[
                _I('multi', False,
                   text="Multisave mode",
                   tooltip="In multisave mode you can save sequences of images,"
                   " by pressing a hotkey and/or automatically on drawing"),
                _I('hotkey', True,
                   text="Activate hotkey",
                   tooltip=f"A new image will be saved every time you hit the hotkey"
                   f" ({pf.cfg['keys/save']}) while the focus is on the canvas"),
                _I('autosave', False,
                   text="Activate ausave mode",
                   tooltip="In autosave mode, a new image will be saved"
                   " at each draw operation"),
            ], spacer='l'),
        ]
        enablers=[
            ('multi', True, 'hotkey', 'autosave'),
            ('set_size', 'Width', 'size'),
            ('set_size', 'Height', 'size'),
            ('set_size', 'Both', 'size'),
        ]

        super().__init__(items=items, enablers=enablers, modal=False,
                       caption="pyFormex Save Image Dialog")
        self.change_extent(self['extent'])


    def change_tool(self, item):
        tool = item.value()
        self['extent'].setChoices(tool_extent_choices[tool])
        formats = ['From Extension'] + image.imageFormats(tool, 'w')
        self['format'].setChoices(formats)


    def change_extent(self, item):
        extent = item.value()
        tools = image.extent_tool_choices(extent)
        self['tool'].setChoices(tools)
        self['tool'].setValue(tools[0])
        formats = ['From Extension'] + image.imageFormats(tools[0], 'w')
        self['format'].setChoices(formats)


    def validate(self):
        if super().validate():
            w, h = SaveImageDialog.default_size = self.results['size']
            resize = self.results['set_size'][0]
            if resize == 'W':
                h = -1
            elif resize == 'H':
                w = -1
            elif resize == 'N':
                w = h = -1    # This would use unscaled offline rendering
            self.results['size'] = (w, h)
            if resize == 'N':
                self.results['size'] = None  # Instead grab from screen buffer
            if self.results['format'] == 'From Extension':
                if self.results['filename'].suffix == '':
                    self.results['filename'] = self.results['filename'].with_suffix('.png')
                self.results['format'] = None
        return self.valid


class RecordSessionDialog(Dialog):
    """A dialog for recording the GUI to a video file.

    This is a specialized Dialog for the input of all data required to
    save the current pyFormex rendering to an image file. It is a
    convenient interactive frontend for the :func:`image.saveImage`
    function. The Dialog ask for the target file name and all the
    other parameters accepted by that function.

    Parameters
    ----------
    path: :term:`path_like`
        The initial path displayed in the file selector widget.
        See FileDialog.
    filer: list of str
        The filters for the files selectable in the file selecter widget.
        See FileDialog.
    multi: bool
        If True, the dialog will show the multisave option initially
        checked.

    """
    def __init__(self, path=None, filter=None, multi=False):
        """Create the dialog."""
        if path is None:
            path = pf.cfg['workdir']
        if filter is None:
            filter = ['video', 'all']

        items=[
            _I('filename', path, itemtype='file', filter=filter, exist=False,
               dir=False, text="Filename:", ),
            _C('',[
                _I('extent', choices=image.record_extents,
                   text="Extent:",
                   tooltip="The part(s) of the pyFormex window to be saved"),
            ]),
            _C('',[
                _I('framerate', 25,
                   text="framerate:",
                   tooltip="The number of frames saved per second."),
            ], spacer='l'),
        ]

        super().__init__(items=items, modal=False,
                       caption="pyFormex Record Session Dialog")
