#
##
##  This file is part of pyFormex 2.4  (Thu Feb 25 13:39:20 CET 2021)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: http://pyformex.org
##  Project page:  http://savannah.nongnu.org/projects/pyformex/
##  Copyright 2004-2020 (C) Benedict Verhegghe (benedict.verhegghe@ugent.be)
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##
"""Basic pyFormex script functions

The :mod:`script` module provides the basic functions available
in all pyFormex scripts. These functions are available in GUI and NONGUI
applications, without the need to explicitely importing the :mod:`script`
module.
"""

import os
import sys
import time
import shutil

import pyformex as pf
from pyformex import Path
from pyformex.filewrite import writePGF as writeGeomFile
from pyformex.fileread import readPGF as readGeomFile
from pyformex import utils
from pyformex.geometry import Geometry

from pyformex.utils import system, command


######################### Exceptions #########################################


class _Exit(Exception):
    """Exception raised to exit from a running script."""
    pass


############################# Globals for scripts ############################


def Globals():
    """Return the globals that are passed to the scripts on execution.

    When running pyformex with the --nogui option, this contains all the
    globals defined in the module formex (which include those from
    coords, arraytools and numpy.

    When running with the GUI, this also includes the globals from gui.draw
    (including those from gui.color).

    Furthermore, the global variable __name__ will be set to either 'draw'
    or 'script' depending on whether the script was executed with the GUI
    or not.
    """
    # :DEV it is not a good idea to put the pf.PF in the globals(),
    # because pf.PF may contain keys that are not valid Python names
    g = {}
    g.update(globals())
    from pyformex import lazy
    g.update(lazy.__dict__)
    if pf.GUI:
        from pyformex.gui import draw
        g.update(draw.__dict__)
    # Set module correct
    if pf.GUI:
        modname = '__draw__'
    else:
        modname = '__script__'
    g['__name__'] = modname
    return g


def export(dic):
    """Export the variables in the given dictionary."""
    pf.PF.update(dic)


def export2(names, values):
    """Export a list of names and values."""
    export(list(zip(names, values)))


def forget(names):
    """Remove the global variables specified in list."""
    g = pf.PF
    for name in names:
        if name in g:
            del g[name]


def forgetAll():
    """Delete all the global variables."""
    pf.PF = {}


def rename(oldnames, newnames):
    """Rename the global variables in oldnames to newnames."""
    g = pf.PF
    for oldname, newname in zip(oldnames, newnames):
        if oldname in g:
            g[newname] = g[oldname]
            del g[oldname]


def listAll(clas=None, like=None, filtr=None, dic=None, sort=False):
    """Return a list of all objects in dictionary that match criteria.

    - `clas`: a class or list of classes: if specified, only instances of
      this/these class(es) will be returned
    - `like`: a string: if given, only object names starting with this string
      will be returned
    - `filtr`: a function taking an object name as parameter and returning True
      or False. If specified, only objects passing the test will be returned.
    - `dic`: a dictionary object with strings as keys, defaults to pyformex.PF.
    - `sort`: bool: if True, the returned list will be sorted.

    The return value is a list of keys from `dic`.
    """
    if dic is None:
        dic = pf.PF

    names = list(dic.keys())
    if clas is not None:
        names = [n for n in names if isinstance(dic[n], clas)]
    if like is not None:
        names = [n for n in names if n.startswith(like)]
    if filtr is not None:
        names = [n for n in names if filtr(n)]
    if sort:
        names = sorted(names)
    # TODO: can/should we return an iterator instead?
    return names


def named(name):
    """Returns the global object named name."""
    if name in pf.PF:
        dic = pf.PF
    else:
        raise NameError(f"Name {name} is not in pyformex.PF")
    return dic[name]


# TODO: can we do this with a returnNone default_factory in pf.refcfg?
def getcfg(name, default=None):
    """Return a value from the configuration or None if nonexistent."""
    try:
        return pf.cfg[name]
    except KeyError:
        return default


#################### Interacting with the user ###############################

def ask(question, choices=None, default=''):
    """Ask a question and present possible answers.

    If no choices are presented, anything will be accepted.
    Else, the question is repeated until one of the choices is selected.
    If a default is given and the value entered is empty, the default is
    substituted.
    Case is not significant, but choices are presented unchanged.
    If no choices are presented, the string typed by the user is returned.
    Else the return value is the lowest matching index of the users answer
    in the choices list. Thus, ask('Do you agree',['Y','n']) will return
    0 on either 'y' or 'Y' and 1 on either 'n' or 'N'.
    """
    if choices:
        question += f" ({', '.join(choices)}) "
        choices = [c.lower() for c in choices]
    while True:
        res = input(question)
        if res == '' and default:
            res = default
        if not choices:
            return res
        try:
            return choices.index(res.lower())
        except ValueError:
            pass


def ack(question):
    """Show a Yes/No question and return True/False depending on answer."""
    return ask(question, ['Y', 'N']) == 0


def error(message):
    """Show an error message and wait for user acknowlegement."""
    print("pyFormex Error: "+message)
    if not ack("Do you want to continue?"):
        exit()


def warning(message):
    print("pyFormex Warning: "+message)
    if not ack("Do you want to continue?"):
        exit()


def showInfo(message):
    print("pyFormex Info: "+message)


########################### PLAYING SCRIPTS ##############################

exitrequested = False
starttime = 0.0
scriptInit = None  # can be set to execute something before each script


def autoExport(g):
    """Autoexport globals from script/app globals.

    g: dict holding the globals dict from a script/app run enviroment.

    This exports some objects from the script/app runtime globals
    to the pf.PF session globals directory.
    The default is to export all instances of class Geometry.

    This can be customized in the script/app by setting the global
    variable `autoglobals`. If set to a value that evaluates to
    False, no autoexport will be done. If set to True, the default
    autoexport will be done: all instances of :class:`geometry`.
    If set to a list of names, only the specified names will be exported.
    Furthermore, a global variable `autoclasses` may be set
    to a list of class names. All global instances of the specified classes
    will be exported.

    Remember that the variables need to be globals in your script/app
    in order to be autoexported, and that autoglobals feature needs to
    be enabled in your configuration.
    """
    ag = g.get('autoglobals', True)
    if ag:
        if ag is True:
            # default autoglobals: all Geometry instances
            ag = [Geometry]
        an = []
        for a in ag:
            if isinstance(a, str) and a in g:
                an.append(a)
            elif isinstance(a, type):
                try:
                    an.extend(listAll(clas=a, dic=g))
                except Exception:
                    pass
        if an:
            an = sorted(list(set(an)))
            print(f"Autoglobals: {', '.join(an)}")
            pf.PF.update([(k, g[k]) for k in an])


def scriptLock(id):
    global _run_mode
    if id == '__auto/script__':
        pf.scriptMode = 'script'
    elif id == '__auto/app__':
        pf.scriptMode = 'app'

    pf.debug(f"Setting script lock {id}", pf.DEBUG.SCRIPT)
    pf.scriptlock |= {id}


def scriptRelease(id):
    pf.debug(f"Releasing script lock {id}", pf.DEBUG.SCRIPT)
    pf.scriptlock -= {id}
    pf.scriptMode = None


def playScript(scr, name=None, filename=None, argv=[], encoding=None):
    """Play a pyformex script scr. scr should be a valid Python text.

    There is a lock to prevent multiple scripts from being executed at the
    same time. This implies that pyFormex scripts can currently not be
    recurrent.
    If name is specified, set the global variable pyformex.scriptName to it
    when the script is started.
    If filename is specified, set the global variable __file__ to it.
    """
    global starttime
    global exitrequested

    # (We only allow one script executing at a time!)
    # and scripts are non-reentrant
    if len(pf.scriptlock) > 0:
        print("!!Not executing because a script lock has been set: "
              f"{pf.scriptlock}")
        return 1

    scriptLock('__auto/script__')
    exitrequested = False

    if pf.GUI:
        pf.GUI.startRun()

    # Read the script, if a file was specified
    if not isinstance(scr, str):
        # scr should be an open file/stream
        if filename is None:
            filename = scr.name
        scr = scr.read() + '\n'

    # Get the globals
    g = Globals()
    if filename:
        g.update({'__file__': filename})
    g.update({'argv': argv})

    # BV: Should we continue support for this?
    if encoding=='pye':
        n = (len(scr)+1) // 2
        scr = utils.mergeme(scr[:n], scr[n:])
    elif encoding=='egg':
        import base64
        scr = base64.b64decode(scr)

    if isinstance(scr, bytes):
        scr = scr.decode('utf-8')

    # Now we can execute the script using these collected globals
    pf.scriptName = name
    exitall = False

    if pf.debugOn(pf.DEBUG.MEM):  # use a test to avoid evaluation of memUsed
        memu = utils.memUsed()
        vmsiz = utils.vmSize()
        pf.debug(f"MemUsed = {memu}; vmSize = {vmsiz}", pf.DEBUG.MEM)

    if filename is None:
        filename = '<string>'

    # Execute the code
    # starttime = time.clock()
    try:
        pf.interpreter.locals.update(g)
        pf.interpreter.runsource(scr, str(filename), 'exec')

    except SystemExit:
        print("EXIT FROM SCRIPT")

    finally:
        # honour the exit function
        if 'atExit' in g:
            atExit = g['atExit']
            try:
                atExit()
            except Exception:
                pf.debug('Error while calling script exit function',
                         pf.DEBUG.SCRIPT)

        if pf.cfg['autoglobals']:
            if pf.console:
                g = pf.console.interpreter.locals
            autoExport(g)
        scriptRelease('__auto/script__')  # release the lock
        if pf.GUI:
            pf.GUI.stopRun()

    if pf.debugOn(pf.DEBUG.MEM):  # use a test to avoid evaluation of memUsed
        pf.debug(f"MemUsed = {utils.memUsed()}; vmSize = {utils.vmSize()}",
                 pf.DEBUG.MEM)
        pf.debug(f"Diff MemUsed = {utils.memUsed()-memu}; "
                 f"diff vmSize = {utils.vmSize()-vmsiz}", pf.DEBUG.MEM)

    if exitall:
        pf.debug("Calling quit() from playscript", pf.DEBUG.SCRIPT)
        quit()

    return 0


def force_finish():
    pf.scriptlock = set()  # release all script locks (in case of an error)


def breakpt(msg=None):
    """Set a breakpoint where the script can be halted on a signal.

    If an argument is specified, it will be written to the message board.

    The exitrequested signal is usually emitted by pressing a button in the GUI.
    """
    global exitrequested
    if exitrequested:
        if msg is not None:
            print(msg)
        exitrequested = False  # reset for next time
        raise SystemExit


def raiseExit():
    pf.debug("RAISING SystemExit", pf.DEBUG.SCRIPT)
    if pf.GUI:
        pf.GUI.drawlock.release()
    raise _Exit("EXIT REQUESTED FROM SCRIPT")


def enableBreak(mode=True):
    if pf.GUI:
        pf.GUI.enableButtons(pf.GUI.actions, ['Stop'], mode)


def stopatbreakpt():
    """Set the exitrequested flag."""
    global exitrequested
    exitrequested = True


def runScript(fn, argv=[]):
    """Play a formex script from file fn.

    fn is the name of a file holding a pyFormex script.
    A list of arguments can be passed. They will be available under the name
    argv. This variable can be changed by the script and the resulting argv
    is returned to the caller.
    """
    fn = Path(fn)
    from pyformex.timer import Timer
    t = Timer()
    msg = f"Running script ({fn})"
    if pf.GUI:
        pf.GUI.scripthistory.add(str(fn))
        pf.GUI.board.write(msg, color='red')
    else:
        print(msg)
    pf.debug(f"  Executing with arguments: {argv}", pf.DEBUG.SCRIPT)
    encoding = None
    if fn.suffix == '.pye':
        encoding = 'pye'

    res = playScript(Path(fn).open('r'), fn, fn, argv, encoding)
    pf.debug(f"  Arguments left after execution: {argv}", pf.DEBUG.SCRIPT)
    msg = f"Finished script {fn} in {t.seconds()} seconds"
    if pf.GUI:
        pf.GUI.board.write(msg, color='red')
    else:
        print(msg)
    return res


def runApp(appname, argv=[], refresh=False, lock=True, check=True, wait=False):
    """Run a pyFormex application.

    A pyFormex application is a Python module that can be loaded in
    pyFormex and that contains a function 'run()'. Running the application
    is equivalent to executing this function.

    Parameters:

    - `appname`: name of the module in Python dot notation. The module should
      live in a path included the the a file holding a pyFormex script.

    - `argv`: list of arguments. This variable can be changed by the app
      and the resulting argv will be returned to the caller.

    Returns the exit value of the run function. A zero value is supposed
    to mean a normal exit.
    """
    pf.debug(f"runApp '{appname}'", pf.DEBUG.APPS)
    global exitrequested
    if check:
        while len(pf.scriptlock) > 0:
            if wait:
                 print(f"!!Waiting for lock {pf.scriptlock} to be released")
                 sleep(5)
            else:
                 print("!!Not executing because a script lock has been set: "
                       f"{pf.scriptlock}")
                 return

    from pyformex import apps
    from pyformex.timer import Timer
    t = Timer()
    print(f"Loading application {appname} with refresh={refresh}")
    app = apps.load(appname, refresh=refresh)
    if app is None:
        errmsg = "An error occurred while loading application {appname}"
        if pf.GUI:
            if apps._traceback and pf.cfg['showapploaderrors']:
                print(apps._traceback)

            from pyformex.gui import draw
            fn = apps.findAppSource(appname)
            if fn.exists():
                errmsg += ("\n\nYou may try executing the application "
                           "as a script,\n  or you can load the source "
                           "file in the editor.")
                res = draw.ask(errmsg, choices=[
                    'Run as script', 'Load in editor', "Don't bother"])
                if res[0] in 'RL':
                    if res[0] == 'L':
                        draw.editFile(fn)
                    elif res[0] == 'R':
                        pf.GUI.setcurfile(fn)
                        draw.runScript(fn)
            else:
                errmsg += "and I can not find the application source file."
                draw.error(errmsg)
        else:
            error(errmsg)

        return

    if hasattr(app, '_opengl2') and not app._opengl2:
        pf.warning("This Example can not yet be run under the pyFormex "
                   "opengl2 engine.\n")
        return

    if hasattr(app, '_status') and app._status == 'unchecked':
        pf.warning(
            "This looks like an Example script that has been automatically "
            "converted to the pyFormex Application model, but has not been "
            "checked yet as to whether it is working correctly in App mode.\n"
            "You can help here by running and rerunning the example, checking "
            "that it works correctly, and where needed fixing it (or reporting "
            "the failure to us). If the example runs well, you can change its "
            "status to 'checked'")

    if lock:
        scriptLock('__auto/app__')
    msg = f"Running application '{appname}' from {app.__file__}"
    pf.scriptName = appname
    if pf.GUI:
        pf.GUI.startRun()
        pf.GUI.apphistory.add(appname)
        pf.GUI.board.write(msg, color='green')
    else:
        print(msg)
    pf.debug(f"  Passing arguments: {argv}", pf.DEBUG.SCRIPT)
    app._args_ = argv
    try:
        try:
            res = app.run()
        except SystemExit:
            print("EXIT FROM APP")
            pass
        except Exception:
            raise
    finally:
        if hasattr(app, 'atExit'):
            app.atExit()
        if pf.cfg['autoglobals']:
            g = app.__dict__
            autoExport(g)
        if lock:
            scriptRelease('__auto/app__')  # release the lock
        if pf.GUI:
            pf.GUI.stopRun()

    pf.debug(f"  Arguments left after execution: {argv}", pf.DEBUG.SCRIPT)
    msg = f"Finished {appname} in {t.seconds()} seconds"
    if pf.GUI:
        pf.GUI.board.write(msg, color='green')
    else:
        print(msg)
    pf.debug(f"Memory: {utils.vmSize()}", pf.DEBUG.MEM)
    return res


def runAny(appname=None, argv=[], step=False, refresh=False, remember=True, wait=False):
    """Run the current pyFormex application or script file.

    Parameters:

    - `appname`: either the name of a pyFormex application (app) or a file
      containing a pyFormex script. An app name is specified in Python
      module syntax (package.subpackage.module) and the path to the package
      should be in the configured app paths.

    This function does nothing if no appname/filename is passed or no current
    script/app was set.
    If arguments are given, they are passed to the script. If `step` is True,
    the script is executed in step mode. The 'refresh' parameter will reload
    the app.
    """
    if appname is None:
        appname = pf.cfg['curfile']
    if not appname:
        return

    if scriptInit:
        scriptInit()

    if pf.GUI and remember:
        pf.GUI.setcurfile(appname)

    if utils.is_script(appname):
        return runScript(appname, argv)
    else:
        return runApp(appname, argv, refresh, wait=wait)


def exit(all=False):
    """Exit from the current script or from pyformex if no script running."""
    if len(pf.scriptlock) > 0:
        if all:
            utils.warn("warn_exit_all")
            pass
        else:
            # This is the only exception we can use in script mode
            # to stop the execution
            raise SystemExit


def quit():
    """Quit the pyFormex program

    This is a hard exit from pyFormex. It is normally not called
    directly, but results from an exit(True) call.
    """
    if pf.app and pf.app_started:  # quit the QT app
        pf.debug("draw.exit called while no script running", pf.DEBUG.SCRIPT)
        pf.app.quit()  # closes the GUI and exits pyformex
    else:  # the QT app didn't even start
        sys.exit(0)  # use Python to exit pyformex


def processArgs(args):
    """Run the application without gui.

    Arguments are interpreted as names of script files, possibly interspersed
    with arguments for the scripts.
    Each running script should pop the required arguments from the list.
    """
    res = 0
    while len(args) > 0:
        fn = args.pop(0)
        if fn == '-c':
            # next arg is a script
            txt = args.pop(0)
            res = playScript(txt, name='__inline__')
        else:
            res = runAny(fn, args, remember=False)
        if res:
            print(f"Error during execution of script/app {fn}")

    return res


def setPrefs(res, save=False):
    """Update the current settings (store) with the values in res.

    res is a dictionary with configuration values.
    The current settings will be update with the values in res.

    If save is True, the changes will be stored to the user's
    configuration file.
    """
    pf.debug(f"Accepted settings:\n{res}", pf.DEBUG.CONFIG)
    for k in res:
        pf.cfg[k] = res[k]
        if save and pf.prefcfg[k] != pf.cfg[k]:
            pf.prefcfg[k] = pf.cfg[k]

    pf.debug(f"New settings:\n{pf.cfg}", pf.DEBUG.CONFIG)
    if save:
        pf.debug(f"New preferences:\n{pf.prefcfg}", pf.DEBUG.CONFIG)


########################## print information ################################


def printConfig():
    print("Reference Configuration: " + str(pf.refcfg))
    print("Preference Configuration: " + str(pf.prefcfg))
    print("User Configuration: " + str(pf.cfg))


def printLoadedApps():
    from pyformex import apps, sys
    loaded = apps.listLoaded()
    refcnt = [sys.getrefcount(sys.modules[k]) for k in loaded]
    print(', '.join([f"{k} ({r})" for k, r in zip(loaded, refcnt)]))


### Utilities


def chdir(path, create=False):
    """Change the current working directory.

    If path exists and it is a directory name, make it the current directory.
    If path exists and it is a file name, make the containing directory the
    current directory.
    If path does not exist and create is True, create the path and make it the
    current directory. If create is False, raise an Error.

    Parameters:

    - `path`: pathname of the directory or file. If it is a file, the name of
      the directory holding the file is used. The path can be an absolute
      or a relative pathname. A '~' character at the start of the pathname will
      be expanded to the user's home directory.

    - `create`: bool. If True and the specified path does not exist, it will
      be created. The default is to do nothing if the specified path does
      not exist.

    The changed to current directory is stored in the user's preferences
    for persistence between pyFormex invocations.

    """
    path = Path(path).expanduser()
    if path.exists():
        if not path.is_dir():
            path = path.resolve().parent
    else:
        if create:
            mkdir(path)
        else:
            raise ValueError(f"The path {path} does not exist")
    try:
        os.chdir(str(path))
        setPrefs({'workdir': path}, save=True)
    except Exception:
        pass
    pwdir()
    if pf.GUI:
        pf.GUI.setcurdir()


def pwdir():
    """Print the current working directory.

    """
    print(f"Current workdir is {Path.cwd()}")


def mkdir(path, clear=False, new=False):
    """Create a directory.

    Create a directory, including any needed parent directories.
    Any part of the path may already exist.

    - `path`: pathname of the directory to create, either an absolute
      or relative path. A '~' character at the start of the pathname will
      be expanded to the user's home directory.
    - `clear`: bool. If True, and the directory already exists, its contents
      will be deleted.
    - `new`: bool. If True, requires the directory to be a new one. An error
      will be raised if the path already exists.

    The following table gives an overview of the actions for different
    combinations of the parameters:

    ======   ====  ====================  =============
    clear    new   path does not exist   path exists
    ======   ====  ====================  =============
    F        F     kept as is            newly created
    T        F     emptied               newly created
    T/F      T     raise                 newly created
    ======   ====  ====================  =============

    If successful, returns the tilde-expanded path of the directory.
    Raises an exception in the following cases:

    - the directory could not be created,
    - `clear` is True, and the existing directory could not be cleared,
    - `new` is False, `clear` is False, and the existing path is not a directory,
    - `new` is True, and the path exists.

    """
    path = Path(path).expanduser()
    if new and path.exists():
        raise RuntimeError(f"Path already exists: {path}")
    if clear and path.exists():
        if path.is_dir():
            shutil.rmtree(str(path))
        else:
            path.unlink()
    if sys.hexversion >= 0x03040100:
        os.makedirs(path, exist_ok=not new)
    else:
        try:
            path.mkdir(parents=True, exist_ok=False)
        except OSError:
            if not path.is_dir():
                raise


def runtime():
    """Return the time elapsed since start of execution of the script."""
    return time.clock() - starttime


def startGui(args=[]):
    """Start the gui"""
    if pf.GUI is None:
        pf.debug("Starting the pyFormex GUI", pf.DEBUG.GUI)
        from pyformex.gui import guimain
        if guimain.startGUI(args) == 0:
            guimain.runGUI()


def create_interpreter():
    import code
    pf.interpreter = code.InteractiveInterpreter(Globals())


# Always create an interpreter
create_interpreter()


#### End
