#
##
##  SPDX-FileCopyrightText: © 2007-2021 Benedict Verhegghe <bverheg@gmail.com>
##  SPDX-License-Identifier: GPL-3.0-or-later
##
##  This file is part of pyFormex 2.6  (Mon Aug 23 15:13:50 CEST 2021)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: https://pyformex.org
##  Project page: https://savannah.nongnu.org/projects/pyformex/
##  Development: https://gitlab.com/bverheg/pyformex
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##

"""Read geometry from files in a number of formats.

This module defines basic routines to read geometrical data
from a file and the specialized importers to read files in a number of
well known standardized formats.
"""

import re

import numpy as np

import pyformex as pf
import pyformex.arraytools as at
from pyformex import utils
from pyformex import Path
from pyformex.connectivity import Connectivity
from pyformex.coords import Coords
from pyformex.mesh import Mesh

__all__ = ['readPGF', 'readOFF', 'readOBJ', 'readPLY', 'readGTS', 'readSTL']


def readPGF(filename, count=-1):
    """Read a pyFormex Geometry File.

    A pyFormex Geometry File can store multiple geometrical objects in a
    native format that can be efficiently read back into pyformex.
    The format is portable over different pyFormex versions and
    even to other software.

    Parameters
    ----------
    filename: :term:`path_like`
        The name of an existing pyFormex Geometry File. If the
        filename ends on '.gz', it is considered to be a gzipped file and
        will be uncompressed transparently during the reading.

    Returns
    -------
    dict
        A dictionary with the geometric objects read from the file.
        If object names were stored in the file, they will be used as the keys.
        Else, default names will be provided.
    """
    from pyformex import geomfile
    filename = Path(filename)
    pf.verbose(1, f"Read PGF file {filename.absolute()}")
    f = geomfile.GeometryFile(filename, 'r')
    objects = f.read(count)
    pf.verbose(2, f"Got {len(objects)} objects")
    return objects


def readOFF(filename):
    """Read a surface mesh from an OFF file.

    Note
    ----
    All the faces in the file should have 3 vertices.
    There should be no comments lines on the file.

    Parameters
    ----------
    filename: :term:`path_like`
        The name of a file in OFF format, commonly having a suffix '.off'.
        If the name ends with '.off.gz' or '.off.bz2', then the file will
        transparently be uncompressed during reading.

    Returns
    -------
    coords: float array (ncoords, 3)
        The coordinates of all vertices.
    elems: int array (nelems,3)
        The element connectivity table.

    Examples
    --------
    >>> from .filewrite import writeOFF
    >>> f = Path('test_filewrite.off')
    >>> M = Mesh(eltype='quad4').convert('tri3-u')
    >>> writeOFF(f, M)
    >>> coords, elems = readOFF(f)[:2]
    >>> print(coords)
    [[ 0.  0.  0.]
     [ 1.  0.  0.]
     [ 1.  1.  0.]
     [ 0.  1.  0.]]
    >>> print(elems)
    [[0 1 2]
     [2 3 0]]
    """
    filename = Path(filename)
    pf.verbose(1, f"Read OFF file {filename.absolute()}")
    with utils.File(filename, 'r') as fil:
        head = fil.readline().strip()
        if head != "OFF":
            raise ValueError(f"{filename} is not an OFF file!")
        nnodes, nelems, nedges = [int(i) for i in fil.readline().split()]
        coords = np.fromfile(file=fil, dtype=at.Float, count=3 * nnodes, sep=' ')
        # elems have number of vertices + 3 vertex numbers
        elems = np.fromfile(file=fil, dtype=np.int32, count=4 * nelems, sep=' ')
        # edges have number of vertices + 2 vertex numbers
        edges = np.fromfile(file=fil, dtype=np.int32, count=3 * nedges, sep=' ')
    pf.verbose(2, f"Got {nnodes} nodes and {nelems} elems")
    coords = coords.reshape(nnodes, 3)
    elems = elems.reshape(nelems, 4)
    edges = edges.reshape(nedges, 3)
    if (elems[:,0] != 3).any() or (edges[:,0] != 2).any():
        raise ValueError("Currently we can only read OFF files consisting"
                         "off pure triangles and files (sorted).")
    return coords, elems[:, 1:], edges[:, 1:]


def readOBJ(filename, mplex=None):
    """Read a mesh from an OBJ file.

    Reads the mesh data from a wavefront .obj file.

    Parameters
    ----------
    filename: :term:`path_like`
        The name of a file in OBJ format, commonly having a suffix '.obj'.
        If the name ends with '.obj.gz' or '.obj.bz2', the file will
        transparently be uncompressed during reading.
    mplex: int, optional
        The maximum plexitude of the output polygons. If provided, polygons
        with a higher plexitude will be split into smaller ones. For example
        with mplex=4, polygons with 5 or more vertices will be split into
        quads and triangles.

    Returns
    -------
    coords: float array (ncoords, 3)
        The coordinates of all vertices.
    faces: dict
        A dict where the keys are plexitudes (<= mplex) and the values are
        the Connectivity tables of all the polygons of the plexitude.

    Notes
    -----
    Currently only handles polygons.
    It does not handle relative indexing, subobjects, groups, lines,
    splines, beziers, materials.
    Normals and texture coordinates are read but not returned.

    Examples
    --------
    >>> from .filewrite import writeOBJ
    >>> f = Path('test_filewrite.obj')
    >>> M = Mesh(eltype='quad4')
    >>> writeOBJ(f, M)
    >>> coords, elems = readOBJ(f)
    >>> print(coords)
    [[ 0.  0.  0.]
     [ 1.  0.  0.]
     [ 1.  1.  0.]
     [ 0.  1.  0.]]
    >>> print(elems)
    {4: array([[0, 1, 2, 3]])}
    """
    from pyformex.geomtools import splitPolygon
    def vertex_data(s):
        t = s.split('/')
        vid = int(t[0])
        tid = int(t[1]) if len(t) > 1 and t[1] else -1
        nid = int(t[2]) if len(t) > 2 else -1
        return vid, tid, nid

    filename = Path(filename)
    pf.verbose(1, f"Read OBJ file {filename.absolute()}")
    coords = { '':[], 'n':[], 't':[] } # storage for coords + normals + texture
    elems = { } # storage for polygon data
    with utils.File(filename, 'r') as fil:
        for line in fil:
            s = line.split()
            if len(s) == 0:
                continue
            typ, *data = s
            # We currently skip texture and normals
            if typ == 'v':
                coords[''].append([float(d) for d in data])
            elif typ in 'lf':
                data = [vertex_data(d)[0] for d in data]
                nplex = len(data)
                if nplex not in elems:
                    elems[nplex] = []
                elems[nplex].append(data)
            # This would be a more general reader
            # if typ[0] == 'v':
            #     if typ[1:2] == '':  # We currently skip texture and normals
            #         coords[typ[1:2]].append([float(d) for d in data])
    try:
        coords = np.array(coords[''], dtype=at.Float)
        faces = dict((k, np.array(elems[k], dtype=at.Int) - 1)
                     for k in elems if len(elems[k]) > 0)
    except:
        raise ValueError("This file is too complex for our current .OBJ reader")
    if not faces:
        raise ValueError("Could not read any faces or lines")
    if mplex and max(faces) > mplex:
        faces = splitPolygon(coords, faces, mplex)

    pf.verbose(2, f"Got {coords.shape[0]} nodes, {[faces[e].shape for e in faces]} faces")
    return coords, faces


def readPLY(filename, mplex=None):
    """Read polygons from a PLY file.

    Reads the polygon data from a stanford .ply file and possibly
    splits the high plexitude polygons into smaller ones.

    Parameters
    ----------
    filename: :term:`path_like`
        The name of a file in PLY format, commonly having a suffix '.ply'.
        Ascii as well as binary formats are supported
        If the name ends with '.ply.gz' or '.ply.bz2', the file will
        transparently be uncompressed during reading.
    mplex: int, optional
        The maximum plexitude of the output polygons. If provided, polygons
        with a higher plexitude will be split into smaller ones. For example
        with mplex=4, polygons with 5 or more vertices will be split into
        quads and triangles.

    Returns
    -------
    coords: float array (ncoords, 3)
        The coordinates of all vertices.
    faces: dict
        A dict where the keys are plexitudes (<= mplex) and the values are
        the Connectivity tables of all the polygons of the plexitude.

    Notes
    -----
    This uses plyfile from https://github.com/dranjan/python-plyfile
    to read the PLY file.

    Examples
    --------
    >>> from .filewrite import writePLY
    >>> f = Path('test_filewrite.ply')
    >>> M = Mesh(eltype='quad4')
    >>> writePLY(f, M.coords, {4:M.elems})
    >>> coords, elems = readPLY(f)
    >>> print(coords)
    [[ 0.  0.  0.]
     [ 1.  0.  0.]
     [ 1.  1.  0.]
     [ 0.  1.  0.]]
    >>> print(elems)
    {4: array([[0, 1, 2, 3]])}
    """
    """PLY to pyformex mesh using plyfile

    :return: list of connectivities
    """
    from pyformex.plugins.plyfile import PlyData
    from pyformex.geomtools import splitPolygon

    filename = Path(filename)
    pf.verbose(1, f"Read PLY file {filename.absolute()}")
    with utils.File(filename, 'rb') as fil:
        ply_data = PlyData.read(fil)
    vertices = ply_data['vertex']
    faces = ply_data['face']
    # Point coordinates
    coords = Coords(np.stack([
        vertices['x'],
        vertices['y'],
        vertices['z'], ], axis=1))
    # Vertex indices
    vertex_indices = faces['vertex_indices']
    # plexitude per face
    nvertices = np.asarray([v_id.size for v_id in vertex_indices])
    # dict with faces of different plexitude
    faces = dict((nplex, np.stack(vertex_indices[nvertices == nplex]))
            for nplex in np.unique(nvertices))
    if mplex and max(faces) > mplex:
        faces = splitPolygon(coords, faces, mplex)
    pf.verbose(2, f"Got {coords.shape[0]} nodes, {[faces[e].shape for e in faces]} faces")
    return coords, faces


def readGTS(filename):
    """Read a surface mesh from a GTS file.

    Parameters
    ----------
    filename: :term:`path_like`
        The name of a file in GTS format, commonly having a suffix '.gts'.
        If the name ends with '.gts.gz' or '.gts.bz2', then the file will
        transparently be uncompressed during reading.

    Returns
    -------
    coords: float array (ncoords, 3)
        The coordinates of all vertices.
    edges: int array (nedges,2)
        The edges to nodes connectivity table.
    faces: int array (nfaces,2)
        The faces to edges connectivity table.

    Examples
    --------
    >>> from .filewrite import writeGTS
    >>> f = Path('test_filewrite.gts')
    >>> M = Mesh(eltype='quad4').convert('tri3-u')
    >>> writeGTS(f, M.toSurface())
    >>> coords, edges, faces = readGTS(f)
    >>> print(coords)
    [[ 0.  0.  0.]
     [ 1.  0.  0.]
     [ 1.  1.  0.]
     [ 0.  1.  0.]]
    >>> print(edges)
    [[0 1]
     [2 0]
     [3 0]
     [1 2]
     [2 3]]
    >>> print(faces)
    [[0 3 1]
     [4 2 1]]
    """
    filename = Path(filename)
    pf.verbose(1, f"Read GTS file {filename.absolute()}")
    with utils.File(filename, 'r') as fil:
        header = fil.readline().split()
        ncoords, nedges, nfaces = [int(i) for i in header[:3]]
        if len(header) >= 7 and header[6].endswith('Binary'):
            sep = ''
            raise RuntimeError(
                "We can not read binary GTS format yet. "
                "See https://savannah.nongnu.org/bugs/index.php?38608. "
                "Maybe you should recompile the extra/gts commands."
            )
        else:
            sep = ' '
        coords = np.fromfile(
            fil, dtype=at.Float, count=3 * ncoords, sep=sep
        ).reshape(-1, 3)
        edges = np.fromfile(
            fil, dtype=np.int32, count=2 * nedges, sep=' '
        ).reshape(-1, 2) - 1
        faces = np.fromfile(
            fil, dtype=np.int32, count=3 * nfaces, sep=' '
        ).reshape(-1, 3) - 1
    pf.verbose(2, f"Got {ncoords} coords, {nedges} edges, {nfaces} faces")
    if coords.shape[0] != ncoords or \
       edges.shape[0] != nedges or \
       faces.shape[0] != nfaces:
        pf.verbose(1, "Error reading GTS file! Result may be incomplete.")
    return coords, edges, faces


def readSTL(filename):
    """Read a surface mesh from an STL file.

    Parameters
    ----------
    filename: :term:`path_like`
        The name of a file in STL format, commonly having a suffix '.stl'.
        If the name ends with '.gz' or '.bz2', then the file will
        transparently be uncompressed during reading.

    Returns
    -------
    coords: float array (ncoords, 3)
        The coordinates of all vertices.
    edges: int array (nedges,2)
        The edges to nodes connectivity table.
    faces: int array (nfaces,2)
        The faces to edges connectivity table.

    Notes
    -----
    STL files come in ascii and binary formats. As there is no simple
    way to detect the format, a binary read is tried first, and if
    unsuccessful, the ascii read is tried next.

    Examples
    --------
    >>> from .filewrite import writeSTL
    >>> f = Path('test_filewrite.stl')
    >>> M = Mesh(eltype='quad4').convert('tri3-u')
    >>> writeSTL(f, M.toFormex().coords, binary=True, color=[255,0,0,128])
    >>> coords, normals, color = readSTL(f)
    >>> print(coords)
    [[[ 0.  0.  0.]
      [ 1.  0.  0.]
      [ 1.  1.  0.]]
    <BLANKLINE>
     [[ 1.  1.  0.]
      [ 0.  1.  0.]
      [ 0.  0.  0.]]]
    >>> print(normals)
    [[ 0.  0.  1.]
     [ 0.  0.  1.]]
    >>> print(color)
    (1.0, 0.0, 0.0)
    >>> writeSTL(f, M.toFormex().coords, binary=False)
    >>> coords, normals, color = readSTL(f)
    >>> print(coords)
    [[[ 0.  0.  0.]
      [ 1.  0.  0.]
      [ 1.  1.  0.]]
    <BLANKLINE>
     [[ 1.  1.  0.]
      [ 0.  1.  0.]
      [ 0.  0.  0.]]]
    >>> print(normals)
    [[ 0.  0.  1.]
     [ 0.  0.  1.]]
    """
    filename = Path(filename)
    pf.verbose(1, f"Read STL file {filename.absolute()}")
    with utils.File(filename, 'rb') as fil:
        head = fil.read(5)
        asc = head[:5] == b'solid'
        fil.seek(0)
        pf.verbose(2, f"Reading {'ascii' if asc else 'binary'} STL file")
        if asc:
            coords, normals = read_stl_asc(fil)
            color = None
        else:
            coords, normals, color = read_stl_bin(fil)
    pf.verbose(2, f"Got {coords.shape[0]} triangles")
    return coords, normals, color


def read_stl_bin(fil):
    """Read a binary STL file.

    Note
    ----
    This is a low level routine for use in readSTL. It is not intended
    to be used directly.

    Parameters
    ----------
    fil: open file
        File opened in binary read mode, holding binary STL data.

    Returns
    -------
    coords: Coords (ntri,3,3)
        A Coords with ntri triangles. Each triangle consists of 3 vertices.
    normals: Coords (ntri,3)
        A Coords with ntri vectors: the outer normals on the triangles.
    color: None | float array (3,)
        If the STL file header contained a color in Materialise (TM) format,
        the RGB color components are returned as OpenGL color components.
        The alpha value is currently not returned.
    """

    def addTriangle(i):
        x[i] = np.fromfile(file=fil, dtype=at.Float, count=12).reshape(4, 3)
        fil.read(2)

    head = fil.read(80)
    i = head.find(b'COLOR=')
    if i >= 0 and i <= 70:
        color = np.frombuffer(head[i + 6:i + 10], dtype=np.uint8, count=4)
    else:
        color = None

    ntri = np.fromfile(fil, dtype=at.Int, count=1)[0]
    x = np.zeros((ntri, 4, 3), dtype=at.Float)
    # nbytes = 12*4 + 2
    [addTriangle(it) for it in range(ntri)]
    normals = Coords(x[:, 0])
    coords = Coords(x[:, 1:])
    if color is not None:
        from pyformex import colors
        color = colors.GLcolor(color[:3])
    return coords, normals, color


def read_stl_asc(fil):
    """Read an ascii STL file.

    Note
    ----
    This is a low level routine for use in readSTL. It is not intended
    to be used directly.

    Parameters
    ----------
    fil: open file
        File opened in binary read mode, holding ascii STL data.

    Returns
    -------
    coords: Coords (ntri,3,3)
        A Coords with ntri triangles. Each triangle consists of 3 vertices.
    normals: Coords (ntri,3)
        A Coords with ntri vectors: the outer normals on the triangles.

    """
    # different line modes and the corresponding re's
    solid, facet, outer, vertex, endloop, endfacet, endsolid = range(7)
    _re_solid = re.compile(b"\\s*solid\\s+.*")
    _re_facet = re.compile(b"\\s*facet\\s+normal\\s+(?P<data>.*)")
    _re_outer = re.compile(b"\\s*outer\\s+loop\\s*")
    _re_vertex = re.compile(b"\\s*vertex\\s+(?P<data>.*)")
    _re_endloop = re.compile(b"\\s*endloop\\s*")
    _re_endfacet = re.compile(b"\\s*endfacet\\s*")
    _re_endsolid = re.compile(b"\\s*endsolid\\s*")
    _re_expect = {
        solid: _re_solid,
        facet: _re_facet,
        outer: _re_outer,
        vertex: _re_vertex,
        endloop: _re_endloop,
        endfacet: _re_endfacet,
    }

    # place to store the results
    normals = []
    coords = []

    def getdata(s):
        """Get 3 floats from a string"""
        data = [float(i) for i in s.split()]
        if len(data) != 3:
            raise ValueError("Expected 3 float values")
        return data

    mode = solid
    facets = 0
    nverts = 0
    count = 0
    for line in fil:
        count += 1
        m = _re_expect[mode].match(line)
        if m is None:
            if mode == facet and _re_endsolid.match(line):
                # We reached the end
                break
            else:
                raise ValueError(f"Invalid input on line {count}:\n{line}")
        if mode == vertex:
            try:
                data = getdata(m.group(1))
            except Exception:
                raise ValueError(f"Data error in line {count}: \n{line}")
            nverts += 1
            coords.append(data)
            if nverts == 3:
                mode = endloop
        elif mode == facet:
            try:
                data = getdata(m.group(1))
            except Exception:
                raise ValueError(f"Data error in line {count}: \n{line}")
            normals.append(data)
            mode = outer
        elif mode == outer:
            nverts = 0
            mode = vertex
        elif mode == endloop:
            mode = endfacet
        elif mode == endfacet:
            facets += 1
            mode = facet
        elif mode == solid:
            if count != 1:
                raise ValueError(f"'solid' found on line {count}")
            mode = facet
        else:
            raise ValueError(f"Invalid input on line {count}")

    coords = Coords(coords).reshape(-1, 3, 3)
    normals = Coords(normals)
    return coords, normals


def read_stl_cvt(filename, intermediate=None):
    """Read a surface from .stl file.

    This is done by first coverting the .stl to .gts or .off format.
    The name of the intermediate file may be specified. If not, it will be
    generated by changing the extension of filename to '.gts' or '.off' depending
    on the setting of the 'surface/stlread' config setting.

    Return a coords,edges,faces or a coords,elems tuple, depending on the
    intermediate format.
    """
    ofn, sta, out = stlConvert(filename, intermediate)
    if sta:
        pf.debug("Error during conversion of file '%s' to '%s'" % (filename, ofn))
        pf.debug(out)
        return ()

    ftype, compr = ofn.ftype_compr()
    if ftype == 'off':
        return readOFF(ofn)
    elif ftype == 'gts':
        return readGTS(ofn)


def stlConvert(stlname, outname=None, binary=False, options='-d'):
    # TODO: this should unzip compressed input files and zip compressed output
    """Convert an .stl file to .off or .gts or binary .stl format.

    Parameters
    ----------
    stlname: :term:`path_like`
        Name of an existing .stl file (either ascii or binary).
    outname: str or Path
        Name or suffix of the output file. The suffix defines the format
        and should be one of '.off', '.gts', '.stl', '.stla', or .stlb'.
        If a suffix only is given (other than '.stl'), then the outname
        will be constructed by changing the suffix of the input ``stlname``.
        If not specified, the suffix of the configuration variable
        'surface/stlread' is used.
    binary: bool
        Only used if the extension of ``outname`` is '.stl'. Defines whether
        the output format is a binary or ascii STL format.
    options: str

    Returns
    -------
    outname: Path
        The name of the output file.
    status: int
        The exit status (0 if successful) of the conversion program.
    stdout: str
        The output of running the conversion program or a
        'file is already up to date' message.

    Notes
    -----
    If the outname file exists and its mtime is more recent than the stlname,
    the outname file is considered up to date and the conversion program will
    not be run.

    The conversion program will be choosen depending on the extension.
    This uses the external commands 'admesh' or 'stl2gts'.

    """
    stlname = Path(stlname)
    if outname is None:
        outname = pf.cfg['surface/stlread']
    outname = Path(outname)
    if outname.suffix == '' and outname.name.startswith('.'):
        # It is considered as a suffix only
        outname = stlname.with_suffix(outname)
        if outname.resolve() == stlname.resolve():
            raise ValueError("stlname and outname resolve to the same file")
    if outname.exists() and stlname.mtime < outname.mtime:
        return outname, 0, "File '%s' seems to be up to date" % outname

    ftype = outname.ftype
    if ftype == 'stl' and binary:
        ftype = 'stlb'

    if ftype == 'off':
        utils.External.has('admesh')
        cmd = "admesh %s --write-off '%s' '%s'" % (options, outname, stlname)
    elif ftype in ['stl', 'stla']:
        utils.External.has('admesh')
        cmd = "admesh %s -a '%s' '%s'" % (options, outname, stlname)
    elif ftype == 'stlb':
        utils.External.has('admesh')
        cmd = "admesh %s -b '%s' '%s'" % (options, outname, stlname)
    elif ftype == 'gts':
        cmd = "stl2gts < '%s' > '%s'" % (stlname, outname)
    else:
        return outname, 1, "Can not convert file '%s' to '%s'" % (stlname, outname)

    P = utils.command(cmd, shell=True)
    return outname, P.returncode, P.stdout


# Global variables used by some functions
filename = None
mode = None
nplex = None
offset = None


def getParams(line):
    """Strip the parameters from a comment line"""
    s = line.split()
    d = {'mode': s.pop(0), 'filename': s.pop(0)}
    d.update(zip(s[::2], s[1::2]))
    return d


def readNodes(fil):
    """Read a set of nodes from an open mesh file"""
    a = np.fromfile(fil, sep=" ").reshape(-1, 3)
    x = Coords(a)
    return x


def readElems(fil, nplex):
    """Read a set of elems of plexitude nplex from an open mesh file"""
    print("Reading elements of plexitude %s" % nplex)
    e = np.fromfile(fil, sep=" ", dtype=at.Int).reshape(-1, nplex)
    e = Connectivity(e)
    return e


def readEsets(fil):
    """Read the eset data of type generate"""
    data = []
    for line in fil:
        s = line.strip('\n').split()
        if len(s) == 4:
            data.append(s[:1] + [int(i) for i in s[1:]])
    return data


def readMeshFile(filename):
    """Read a nodes/elems model from file.

    Returns a dict:

    - 'coords': a Coords with all nodes
    - 'elems': a list of Connectivities
    - 'esets': a list of element sets
    """
    d = {}
    fil = open(filename, 'r')
    for line in fil:
        if line[0] == '#':
            line = line[1:]
        globals().update(getParams(line))
        dfil = open(filename, 'r')
        if mode == 'nodes':
            d['coords'] = readNodes(dfil)
        elif mode == 'elems':
            elems = d.setdefault('elems', [])
            e = readElems(dfil, int(nplex)) - int(offset)
            elems.append(e)
        elif mode == 'esets':
            d['esets'] = readEsets(dfil)
        else:
            print("Skipping unrecognized line: %s" % line)
        dfil.close()

    fil.close()
    return d


def extractMeshes(d):
    """Extract the Meshes read from a .mesh file.

    """
    x = d['coords']
    e = d['elems']
    M = [Mesh(x, ei) for ei in e]
    return M

def convertInp(filename):
    """Convert an Abaqus .inp to a .mesh set of files"""
    filename = Path(filename).resolve()
    converter = pf.cfg['bindir'] / 'read_abq_inp.awk'
    cmd = 'cd %s;awk -f %s %s' % (filename.parent, converter, filename.name)
    print(cmd)
    P = utils.command(cmd, shell=True)
    print(P.stdout)
    print(P.stderr)


######## Abaqus/Calculix INP format (.inp) ##########


def readINP(filename, tempdir=None):
    """Read the geometry from an Abaqus/Calculix .inp file

    This is a replacement for the convertInp/readMeshFile combination.
    It uses the ccxinp plugin to provide a direct import of the Finite
    Element meshes from an Abaqus or Calculix input file.
    Currently still experimental and limited in functionality (aimed
    primarily at Calculix). But also many simple meshes from Abaqus can
    already be read.

    Parameters
    ----------
    filename: :term:`path_like`
        The path of the input file.
    tempdir: :term:`path_like`, optional
        The path of a directory where intermediary results can be stored.
        If provided but not existent, it will be created. If not provided,
        a temporary directory is created and removed after return.

    Returns
    -------
    dict
        A dict where the keys are part names and the values are
        :class:`FeModel` instances.
    """
    from pyformex.plugins import ccxinp
    from pyformex.plugins import fe

    model = ccxinp.readInpFile(filename, tempdir=tempdir)
    print("Number of parts: %s" % len(model.parts))
    fem = {}
    for part in model.parts:
        try:
            coords = Coords(part['coords'])
            nodid = part['nodid']
            nodpos = at.inverseUniqueIndex(nodid)
            print("nnodes = %s" % coords.shape[0])
            print("nodid: %s" % nodid)
            print("nodpos: %s" % nodpos)
            elems = []
            for t, e in part['elems']:
                print("Orig els %s" % e)
                el = nodpos[e]
                print("Trl els %s" % el)
                els = Connectivity(nodpos[e], eltype=t)
                elems.append(els)
            print('ELEM TYPES: %s' % [e.eltype for e in elems])
            fem[part['name']] = fe.Model(coords, elems)
        except Exception as e:
            print("Skipping part %s" % part['name'])
            print(e)
    return fem


######## Gambit neutral file (.neu) ##########

# convert Gambit neutral node order to pyFormex order
neu_pyf_order = {
    4: (0, 1, 3, 2, 4, 5, 7, 6)
    }

def readNEU(filename, return_numbering=False):
    """Read a Mesh from a Gambit neutral file.

    Parameters
    ----------
    filename: :term:`path_like`
        The name of a file in Gambit neutral format, commonly having a suffix
        '.neu'. If the name ends with '.gz' or '.bz2', the file will
        transparently be uncompressed during reading.
    return_numbering: bool
        If True, also returns the original node and element numbers as found
        in the file. The internal pyFormex numbers are always in order of
        appearance in the file.

    Returns
    -------
    Mesh
        The Mesh read from the Gambit neutral file, if reading was successful.
        None if the file could not be read. If the .neu file contains groups,
        the Mesh will have prop values equal to the material type numbers in
        the file.
    nrs: int array
        The node numbers in the .neu file corrsponding to the nodes in
        Mesh.coords. Only returned if return_numbering is True.
    enrs: in array
        The element numbers in the .neu file corresponding to the elements
        in Mesh.elems. Only returned if return_numbering is True.

    Notes
    -----
    Currently this function can only read Gambit files where all cells
    are of the same element type.

    Examples
    --------
    >>> from pyformex.plugins.neu_exp import writeNEU
    >>> f = Path('test_filewrite.neu')
    >>> M = Mesh(eltype='hex8').subdivide(3)
    >>> writeNEU(f, M)
    >>> M = readNEU(f)
    Successfully read test_filewrite.neu
     ncoords=64; nelems=27;  nplex=8; eltype=Hex8
    >>> print(M)
    Mesh: nnodes: 64, nelems: 27, nplex: 8, level: 3, eltype: hex8
      BBox: [ 0.  0.  0.], [ 1.  1.  1.]
      Size: [ 1.  1.  1.]
      Volume: 0.999999...
    """

    from pyformex.plugins.neu_exp import pyf_neu_eltype, neu_pyf_order
    neu_pyf_eltype = utils.inverseDict(pyf_neu_eltype)

    def find_line(fil, pat):
        for line in fil:
            if line.split()[:len(pat)] == pat:
                return True

    def read_coords(fil, ncoords):
        nrs = np.full((ncoords,), -1, dtype=at.Int)
        coords = np.empty((ncoords, 3), dtype=at.Float)
        for i in range(ncoords):
            line = next(fil)
            s = line.split()
            nrs[i] = int(s[0])
            coords[i] = [float(si) for si in s[1:4]]
        return nrs, coords

    def read_elems(fil, nelems):
        enrs = np.full((nelems,), -1, dtype=at.Int)
        for i in range(nelems):
            line = next(fil)
            s = line.split()
            enrs[i] = int(s[0])
            if i == 0:
                eltyp = int(s[1])
                nplex = int(s[2])
                elems = np.empty((nelems, nplex), dtype=at.Int)
            else:
                if eltyp != int(s[1]) or nplex != int(s[2]):
                    raise ValueError(
                        "Currently I can not handle .neu files with more "
                        "than 1 element type")
            s = s[3:]
            while len(s) < nplex:
                s.extend(next(fil).split())
            elems[i] = [int(si) for si in s]
        return eltyp, enrs, elems

    def read_group(fil):
        s = next(fil).split()
        group, nels, matnr, nflags = [int(i) for i in s[1::2]]
        next(fil)  # groupname
        next(fil)  # flags
        values = []
        for i in range((nels+9) // 10):
            values.extend([int(si) for si in next(fil).split()])
        return matnr, np.array(values, dtype=at.Int)

    filename = Path(filename)
    pf.verbose(1, f"Read NEU file {filename.absolute()}")
    with utils.File(filename, 'r') as fil:
        if find_line(fil, ['NUMNP']):
            # read the parameters
            line = next(fil)
            s = line.split()
            ncoords = int(s[0])
            nelems = int(s[1])
            if find_line(fil, ['NODAL', 'COORDINATES']):
                # read the nodal coordinates
                nrs, coords = read_coords(fil, ncoords)
                line = next(fil)
                if find_line(fil, ['ELEMENTS/CELLS']):
                    # read the elements
                    eltyp, enrs, elems = read_elems(fil, nelems)
                    nplex = elems.shape[-1]
                    elname = neu_pyf_eltype.get((eltyp, nplex), None)
                    if elname is None:
                        raise ValueError(
                            f"I can not convert .neu files with elements"
                            f" of type {eltyp} and plexitude {nplex}")
                    # Translate node numbers to in-order coordinates
                    inv = at.inverseUniqueIndex(nrs)
                    elems = inv[elems]
                    if elname in neu_pyf_order:
                        # Reorder the nodes to our element node numbering scheme
                        elems = elems[:, neu_pyf_order[elname]]
                    M = Mesh(coords, elems, eltype=elname)

                    # print Read groups
                    while find_line(fil, ['ELEMENT', 'GROUP']):
                        # read an element group
                        p, els = read_group(fil)
                        if M.prop is None:
                            M.setProp(0)
                        M.prop[els] = p

                    print(f"Successfully read {filename}")
                    print(f" ncoords={ncoords}; nelems={M.nelems()}; "
                          f" nplex={M.nplex()}; eltype={M.elType()}")

                    if return_numbering:
                        return M, nrs, enrs
                    else:
                        return M
        print(f"Error while reading {filename}")

# End
