#
##
##  SPDX-FileCopyrightText: © 2007-2021 Benedict Verhegghe <bverheg@gmail.com>
##  SPDX-License-Identifier: GPL-3.0-or-later
##
##  This file is part of pyFormex 2.6  (Mon Aug 23 15:13:50 CEST 2021)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: https://pyformex.org
##  Project page: https://savannah.nongnu.org/projects/pyformex/
##  Development: https://gitlab.com/bverheg/pyformex
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##
"""Display help

"""

import sys

import pyformex as pf
from pyformex import utils
from pyformex import software
from pyformex.path import Path
from pyformex.gui import draw
from pyformex.gui import qtgl
from .Settings import updateSettings


# Doc build/install schemes
#
#  'G': build/install under git tree
#         build_dir = pyformexdir / 'doc'
#
#  'RD': as user post-install:
#         build_dir = .local/share/pyformex/doc/html
#
#  'R': pyformex installation from .tar.gz, during install:
#         build in pyformexdir / 'doc', install in /usr/local/share
#
#  'D': debian package during creation
#         build in pyformexdir / 'doc'
#         create pyformex-doc package /usr/share/doc/pyformex/html
#

def checkHtmlReqs():
    """Check the requirements for building the html docs"""
    required = {
        'System': {
            'pyFormex_installtype': 'G',
        },
        'Modules': {
            'docutils': '>= 0',
            'sphinx': '>= 1.4',
        },
        'Externals': {
            'sphinx-build': '>= 1.4',
        },
    }
    # CHECK SOFT
    result, s = software.checkSoftware(required, report=True)
    print(s)
    return result


def buildHtmlDocs(builddir=None):
    """Build the html documentation.

    Note: this is only for users building the docs.
    Builds at installation time are done by the Makefile.
    builddir is where the docs are built.
    For 'G' this is pyformex/doc/.
    For 'RD' this is .local/share/pyformex/doc/html.

    """
    # Check requirements
    if not checkHtmlReqs() != 'OK':
        raise ValueError("I could not find some required software")
    # the source should always be here
    sourcedir = pf.cfg['pyformexdir'] / 'sphinx'
    if builddir is None:
        # set default build dir
        if pf.installtype in 'GU':
            builddir = pf.cfg['docdir']
        else:
            builddir = Path.home() / '.local' / 'share' / 'pyformex' / 'doc'
    builddir.mkdir(parents=True, exist_ok=True)
    if not builddir.is_writable_dir():
        draw.showError("TODO: !!!!!!")
    if draw.showInfo(f"""..

Building local documentation
----------------------------
I will now build the local documentation in ::

  {builddir}

As this is a lengthy process (especially if you are building
for the first time), I will run it in the background
and notify you when it's ready.
""", actions=['Cancel', 'OK']) == 'OK':
        cmd = f'make -C {sourcedir} html BUILDDIR={builddir}'
        P = draw.runLongTask(cmd)
        print(f"Return with value {P.returncode}")
        updateSettings({'htmldir': builddir}, save=True)
        menu_enabler(pf.GUI.menu)


def showLocalURL(filename):
    """Show a html document in the local html dir.

    Parameters
    ----------
    filename: str
        Path of the file relative to pf.cfg['htmldir']
    """
    path = pf.cfg['htmldir'] / filename
    draw.showURL(f"file:{path}")


def searchIndex():
    """Search text in pyFormex refman index.

    Asks a pattern from the user and searches for it the index of the
    local pyFormex documentation. Displays the results in the browser.
    """
    from pyformex.gui.draw import _I
    res = draw.askItems([
        _I('text', '', text='String to search'),
        ])

    if res:
        showLocalURL(f"search.html?q={res['text']}&check_keywords=yes&area=default")


def searchText():
    """Search text in pyFormex source files.

    Asks a pattern from the user and searches for it through all
    the pyFormex source files.
    """
    from pyformex.gui.draw import _I
    res = draw.askItems([
        _I('pattern', '', text='String to grep'),
        _I('options', '', text='Options', tooltip="Some cool options: -a (extended search), -i (ignore case), -f (literal string), -e (extended regexp)"),
        ])

    if res:
        out = utils.grepSource(relative=False, **res)
        draw.showText(out, mono=True) #, modal=False)


def catchAndDisplay(expression):
    """Catch stdout from a Python expression and display it in a window."""
    save = sys.stdout
    try:
        with utils.TempFile() as f:
            sys.stdout = f
            eval(expression)
            f.seek(0)
            draw.showText(f.read())
    finally:
        sys.stdout = save


def opengl():
    """Display the OpenGL format description."""
    from pyformex.opengl import gl
    s = utils.formatDict(gl.gl_version()) + '\n'
    s += qtgl.OpenGLFormat(pf.canvas.format())
    draw.showText(s, mono=True)

def detected():
    """Display the detected software components."""
    from pyformex.main import whereami
    draw.showText(whereami() +
                  '\n\n' +
                  software.reportSoftware(header="Detected Software"),
                  mono=True)

def about():
    """Display short information about pyFormex."""
    version = pf.Version()
    draw.showInfo("""..

%s
%s

A tool for generating, manipulating and transforming 3D geometrical models by sequences of mathematical operations.

%s

Distributed under the GNU GPL version 3 or later
""" % (version, '='*len(version), pf.Copyright))

# List of developers/contributors (current and past)
_developers = [
    'Matthieu De Beule',
    'Nic Debusschere',
    'Gianluca De Santis',
    'Bart Desloovere',
    'Wouter Devriendt',
    'Francesco Iannaccone',
    'Peter Mortier',
    'Tim Neels',
    'Tomas Praet',
    'Tran Phuong Toan',
    'Sofie Van Cauter',
    'Benedict Verhegghe',
    'Zhou Wenxuan',
    ]
utils.shuffle(_developers)

def developers():
    """Display the list of developers."""
    utils.shuffle(_developers)
    draw.showInfo("""
The following people have
contributed to pyFormex.
They are listed in random order.

%s

If you feel that your name was left
out in error, please write to
bverheg@gmail.com
""" % '\n'.join(_developers))


_cookies = [
    "Smoking may be hazardous to your health.",
    "Windows is a virus.",
    "Coincidence does not exist. Perfection does.",
    "It's all in the code.",
    "Python is the universal glue.",
    "Intellectual property is a mental illness.",
    "Programmers are tools for converting caffeine into code.",
    "There are 10 types of people in the world: those who understand binary, and those who don't.",
    "Linux: the choice of a GNU generation",
    "Everything should be made as simple as possible, but not simpler. (A. Einstein)",
    "Perfection [in design] is achieved, not when there is nothing more to add, but when there is nothing left to take away. (Antoine de Saint-Exupéry)",
    "Programming today is a race between software engineers striving to build bigger and better idiot-proof programs, and the universe trying to build bigger and better idiots. So far, the universe is winning. (Rick Cook)",
    "In theory, theory and practice are the same. In practice, they're not. (Yoggi Berra)",
    "Most good programmers do programming not because they expect to get paid or get adulation by the public, but because it is fun to program. (Linus Torvalds)",
    "Always code as if the guy who ends up maintaining your code will be a violent psychopath who knows where you live. (Martin Golding)",
    "If Microsoft had developed Internet, we could not ever see the source code of web pages. HTML might be a compiled language then.",
    "What one programmer can do in one month, two programmers can do in two months.",
    "Windows 9x: n. 32 bit extensions and a graphical shell for a 16 bit patch to an 8 bit operating system originally coded for a 4 bit microprocessor, written by a 2 bit company that can't stand 1 bit of competition. (Cygwin FAQ)",
    "You know, when you have a program that does something really cool, and you wrote it from scratch, and it took a significant part of your life, you grow fond of it. When it's finished, it feels like some kind of amorphous sculpture that you've created. It has an abstract shape in your head that's completely independent of its actual purpose. Elegant, simple, beautiful.\nThen, only a year later, after making dozens of pragmatic alterations to suit the people who use it, not only has your Venus-de-Milo lost both arms, she also has a giraffe's head sticking out of her chest and a cherubic penis that squirts colored water into a plastic bucket. The romance has become so painful that each day you struggle with an overwhelming urge to smash the fucking thing to pieces with a hammer. (Nick Foster)",
    "One of my most productive days was throwing away 1000 lines of code. (Ken Thompson)",
    ]
utils.shuffle(_cookies)

def roll(l):
    l.append(l.pop(0))

def cookie():
    draw.showInfo(_cookies[0], ["OK"])
    roll(_cookies)


def _L(text, url):
    """Define a menu_item that is a local html file"""
    return (text, showLocalURL, {'data': str(url)})

def _F(text, url):
    """Define a menu_item that is an text file"""
    return (text, draw.showFile, {'data': str(url)})

def _U(text, url):
    """Define a menu_item that is an URL link"""
    return (text, draw.showURL, {'data': str(url)})

docdir = pf.cfg['docdir']

localdoc = [
    _L('&Local documentation', 'index.html'),
    _L('&Reference Manual', 'refman.html'),
    _L('&Tutorial', 'tutorial.html'),
    _L('&Running pyFormex', 'running.html'),
    _L('&Module Index', 'py-modindex.html'),
    _L('&Index', 'genindex.html'),
]
distdoc = [
    _F('&Readme', docdir / 'README.rst'),
    _F('&ReleaseNotes', docdir / 'ReleaseNotes'),
    _F('&License', docdir / 'LICENSE'),
]

savannah = 'http://savannah.nongnu.org/'
links = [
    _U('pyFormex', 'http://pyformex.org'),
    _U('pyFormex development', 'https://gitlab.com/bverheg/pyformex'),
    _U('pyFormex issues', 'https://gitlab.com/bverheg/pyformex/-/issues'),
    ('pyFormex @Savannah', [
        _U('project page', savannah + 'projects/pyformex/'),
        _U('support', savannah + 'support/?func=additem&group=pyformex'),
        _U('bugs', savannah + 'bugs/?func=additem&group=pyformex'),
    ]),
    _U('Python', 'https://python.org'),
]


def menu_enabler(menu):
    ok = False
    for htmldir in (
        # paths where the html docs might be
            pf.cfg['htmldir'],
            pf.cfg['docdir'] / 'html',
            Path.home() / '.local' / 'share' / 'pyformex' / 'doc' / 'html',
            Path.home() / '.local' / 'lib' / 'pyformex' / 'doc' / 'html',
            ):
        if (htmldir / 'index.html').exists():
            pf.cfg['htmldir'] = htmldir
            ok = True
            break
    m = menu['&Help']
    items = [ i[0] for i in localdoc ] + [ '&Search in index' ]
    for item in items:
        m[item].setEnabled(ok)
    #ok = pf.cfg['sphinxdir'].is_writable_dir() and len(utils.External.has('sphinx-build')) > 0
    #m['&Build local documentation'].setEnabled(ok)

extra_help = []

################### extra software installers ##################

def install_external(pkgdir, prgname):
    extdir = pf.cfg['pyformexdir'] / 'extra' / pkgdir
    cmd = f"cd {extdir} && make && make install"
    P = utils.command(cmd, shell=True)
    if P.returncode:
        draw.showText(P.stdout + P.stderr)
    else:
        if utils.External.has(prgname, check=True):
            info = "Succesfully installed %s" % prgname
        else:
            info ="You should now restart pyFormex!"
        draw.showInfo(info)
    return P.returncode

def install_dxfparser():
    install_external('dxfparser', 'pyformex-dxfparser')

def install_postabq():
    install_external('postabq', 'pyformex-postabq')

def install_gts():
    install_external('gts', 'gtsinside')

def install_instant_meshes():
    install_external('instant', 'instant-meshes')


extra_help = [
    ('---', None),
    ('&Install Externals', [
        ('Instant Meshes', install_instant_meshes),
    ]),
]

if pf.installtype in 'UG':

    extra_help[1][1].extend([
        ('postabq', install_postabq),
        ('gts', install_gts),
        ])

    def DevLinksMenuData():
        """Define extra help items for developer version"""
        pyformexdir = pf.cfg['pyformexdir']
        devdir = pyformexdir / ".."
        devtodo = devdir / "TODO"
        devhowto = devdir / "HOWTO-dev.rst"
        devstyle = devdir / "HOWTO-style.rst"
        devapp = devdir / "scripts-apps.rst"
        devextra = devdir / "install-extra.rst"
        developer = [
            _F('Developer HOWTO', devhowto),
            _F('Scripts versus Apps', devapp),
            _F('pyFormex coding style guide', devstyle),
            _F('pyFormex TODO list', devtodo),
            _F('Installation of extra software', devextra),
            _U('Numpy documentation guidelines', 'https://numpydoc.readthedocs.io/en/latest/format.html#docstring-standard'),
            _U('re-structured text (reST)', 'http://docutils.sourceforge.io/rst.html')
            ]
        return developer

    extra_help.append(('&Build local documentation', buildHtmlDocs))
    extra_help.append(('&Developer Guides', DevLinksMenuData()))

MenuData = ('&Help', [
    _U('&Online documentation', pf.cfg['help/webdoc']),
    ('---', None),
] + localdoc + [
    ('&Search in index', searchIndex),
    ('&Search in source', searchText),
    ('&Current Application', draw.showDoc),
    ('---', None),
] + distdoc + [
    ('&Detected Software', detected),
    ('&OpenGL Format', opengl),
    ('---', None),
    ('&Important Links', links),
    ('&Fortune Cookie', cookie),
    ('&People', developers),
    ('---', None),
    ('&About', about),
] + extra_help
)


# End
