#
##
##  SPDX-FileCopyrightText: © 2007-2021 Benedict Verhegghe <bverheg@gmail.com>
##  SPDX-License-Identifier: GPL-3.0-or-later
##
##  This file is part of pyFormex 3.0  (Mon Nov 22 14:32:59 CET 2021)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: https://pyformex.org
##  Project page: https://savannah.nongnu.org/projects/pyformex/
##  Development: https://gitlab.com/bverheg/pyformex
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##

"""pyFormex core module initialization.

This module initializes the pyFormex global variables and
defines a few essential functions.
It is the very first thing that is loaded when starting pyFormex.
"""

import os
import sys
import time
import datetime
import logging
import logging.config
from getpass import getuser

__prog__ = "pyformex"
__version__ = "3.0"
__revision__ = __version__
__branch__ = "master"
Copyright = 'Copyright (C) 2004-2021 Benedict Verhegghe'
Url = 'http://pyformex.org'
Description = ("pyFormex is a tool for generating, manipulating and "
               "transforming large geometrical models of 3D structures "
               "by sequences of mathematical transformations.")


def Version():
    """Return a string with the pyFormex name and version"""
    return f"pyFormex {__version__}"


def fullVersion():
    """Return a string with the pyFormex name, version and revision"""
    if __revision__ == __version__:
        return Version()
    else:
        return f"{Version()} ({__revision__}) DEVELOPMENT VERSION"


# set start date/time
StartTime = datetime.datetime.now()

startup_messages = ''
startup_warnings = ''
started = False

#########  Check Python version #############

# intended Python version
# We only support 3.7+
minimal_version = 0x03070000
target_version = 0x03090000
future_version = 0x030A0000


def major(v):
    """Return the major component of version"""
    return v >> 24


def minor(v):
    """Return the minor component of version"""
    return (v & 0x00FF0000) >> 16


def human_version(v):
    """Return the human readable string for version"""
    return f"{major(v)}.{minor(v)}"


if sys.hexversion < minimal_version:
    # Older than minimal
    y = human_version(sys.hexversion)
    m = human_version(minimal_version)
    startup_warnings += f"""
#######################################################################
##  Your Python version is {y}, but pyFormex requires Python >= {m}.
##  We advice you to upgrade your Python version.
#######################################################################
"""
    print(startup_warnings)
    sys.exit()

if sys.hexversion & 0xFFFF0000 > target_version:
    # Major,minor newer than target:
    y = human_version(sys.hexversion)
    t = human_version(target_version)
    startup_warnings += f"""
#######################################################################
##  Your Python version is {y}, but pyFormex has only been tested
##  with Python <= {t}. We expect pyFormex to run correctly with
##  your Python version, but if you encounter problems, please
##  contact the developers at http://pyformex.org.
#######################################################################
"""

###################################################################
## Load Path class
##################

from pyformex.path import Path

# Placeholder for excutable, if pyformex: filled in by startup.py
executable = Path(sys.executable)

#### Detect install type ########

# Install type.
# This can have the following values:
#     'G' : unreleased version running from GIT sources: no installation
#           required. This is set if the directory containing the
#           pyformex start script is a git repository.
#     'U' : normal (source) Release, i.e. tarball installed as user.
#           Installation is done with 'python setup.py install'
#           in the unpacked source.
#           The builtin pyFormex removal is activated.
#     'R' : normal (source) Release, i.e. tarball (default). Installation
#           is done with 'python setup.py install' in the unpacked source.
#           The builtin pyFormex removal is activated.
#     'D' : Distribution package of a released version, official or not
#           (e.g. Debian package). The distribution package tools should
#           be used to install/uninstall. The builtin pyFormex removal is
#           deactivated.
#
installtype = 'R'

pyformexdir = Path(__file__).parent
if (pyformexdir.parent / '.git').exists():
    installtype = 'G'
if (installtype == 'R' and getuser() != 'root' and
    (pyformexdir / 'doc').is_writable_dir()):
    installtype = 'U'

########## start the logger ####################
# TODO: this is work in progress
LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'formatters': {
        'default': {
            'format': "%(asctime)s - %(name)s - %(levelname)s - %(message)s"
        },
        'short': {
            'format': "%(asctime)s %(message)s",
        },
        'detail': {
            'format': "%(asctime)s::%(levelname)s::%(name)s::"
            "%(filename)s::%(lineno)d::%(message)s",
        },
    },
    'handlers': {
        'console': {
            'class': 'logging.StreamHandler',
            'formatter': 'detail',
        },
        'memory': {
            'class': 'logging.handlers.MemoryHandler',
            'capacity': 100,
            'formatter': 'default',
        },
        # 'logfile': {
        #     'level':'DEBUG',
        #     'class':'logging.handlers.RotatingFileHandler',
        #     'filename': pf.cfg['logfile'],
        #     'maxBytes': 1024*1024*5, # 5 MB
        #     'backupCount': 100,
        #     'formatter':'standard',
        # },
    },
    'loggers': {
        'pyformex': {
            'level': 'INFO',
            'handlers': ['memory'],
        }
    },
}

logging.config.dictConfig(LOGGING)
logger = logging.getLogger('pyformex')
logger.info(f"{fullVersion()} start on Python{human_version(sys.hexversion)}")

########## acceleration libraries ##############

# Are we importing from Sphinx doc generator?
#sphinx = installtype == 'G' and 'sphinx' in sys.modules
sphinx = 'sphinx' in sys.modules

libraries = ['misc_', 'nurbs_', 'clust_']

########## Special adjustements for source versions ############

if installtype == 'G':
    # We are running from a source tree (Git)

    def run_cmd(cmd):
        """Run command"""
        import subprocess
        P = subprocess.Popen(
            str(cmd), stdout=subprocess.PIPE, shell=True, universal_newlines=True
        )
        out, err = P.communicate()
        if P.returncode:
            print(out)
            print(err)
        return P.returncode, out, err

    # Clean the source tree
    source_clean = pyformexdir / 'source_clean'
    if source_clean.exists():
        try:
            from importlib import machinery
            suffix = machinery.EXTENSION_SUFFIXES[0]
            cmd = f"{source_clean} --keep_modules '*{suffix}'"
            run_cmd(cmd)
        except Exception as e:
            print(str(e))
            print(f"Error while executing {source_clean}\n"
                  "Ignore error and continue")

    # Running from source tree: make sure the compiled libraries are up-to-date
    libdir = pyformexdir / 'lib'

    def checkLibraries():
        # find extension used by compiled library (.so)
        import sysconfig
        ext = sysconfig.get_config_var('EXT_SUFFIX')
        if ext is None:
            ext = '.so'
        msg = ''
        for lib in libraries:
            if lib.startswith('_'):
                src = libdir / lib + '.pyx'
            else:
                src = libdir / lib + '.c'
            obj = libdir / lib + ext
            if not obj.exists() or obj.mtime < src.mtime:
                msg += f"\nThe compiled library '{lib}' is not up to date!"
        return msg

    msg = checkLibraries()
    if msg and not sphinx:
        print(msg)
        print("Rebuilding pyFormex libraries, please wait")
        target = 'lib3'
        cmd = f"cd {pyformexdir.parent}; PYTHON={sys.executable} make {target}"
        run_cmd(cmd)
        msg = checkLibraries()

    if msg:
        # No success
        msg += f"""

I had a problem rebuilding the libraries in {libdir}.
You should probably exit pyFormex, fix the problem and then restart pyFormex.
"""
    startup_warnings += msg

    # Set the proper revision number when running from git sources

    def set_revision():
        global __revision__
        cmd = f"cd {pyformexdir} && git describe --always"
        sta, out, err = run_cmd(cmd)
        if sta == 0:
            __revision__ = out.split('\n')[0].strip()
        else:
            print("Could not set revision")

    def set_branch():
        global __branch__
        cmd = f"cd {pyformexdir} && git rev-parse --abbrev-ref HEAD"
        sta, out, err = run_cmd(cmd)
        if sta == 0:
            __branch__ = out.split('\n')[0]
        else:
            print("Could not set branch name")

    set_revision()
    set_branch()

logger.info(f"This is {fullVersion()}")
logger.info(f"Installation type {installtype} at {pyformexdir}")

################ pyFormex global variables and functions ###############

# The GUI parts
app_started = False
interactive = False
app = None  # the Qapplication
GUI = None  # the GUI QMainWindow
canvas = None  # the OpenGL Drawing widget controlled by the running script
console = None  # alternate Python console

# initialize some global variables used for communication between modules

# the options found on the command line
# this is replaced with the options read from the command line
# but we define it here with an attribute debuglevel, so that we can
# use debug (unintentionally) before the debuglevel is set


class options:
    debuglevel = {}  # Make sure we can use  VAR in pf.options.debuglevel
    verbose = 1     # Make sure we can use
    uselib = False  # initial default to keep sphinx happy


# Important pyFormex globals
PF = {}  # explicitely exported globals
GS = None # only set in gui mode

scriptName = None
scriptlock = set()
scriptMode = None

# define default of warning and error
warning = print
error = print


def _busy(state=True):
    """Flag a busy state to the user"""
    if state:
        print("This may take some time...")


def busy(state=True):
    """Flag a busy state to the user"""
    if GUI:
        GUI.setBusy(state)
    else:
        _busy(state)


######################################################################
###  DEBUG items
################################

from enum import Flag
from operator import or_
from functools import reduce

class DEBUG(Flag):
    """A class with debug items.

    This class holds the defined debug items as attributes.
    Each debug topic is a binary value with a single bit set (a power of 2).
    Debug items can be combined with & (AND), | (OR) and ^ (XOR).
    Two extra values are defined: NONE switches off all debugging, ALL
    activates all debug items.
    """
    NONE = 0
    (WARNING, CONFIG, DETECT, MEM, SCRIPT, GUI, MENU, DRAW,
     CANVAS, OPENGL, LIB, MOUSE, APPS, IMAGE, MISC, ABQ, WIDGET, PROJECT,
     WEBGL, MULTI, LEGACY, PLUGIN, UNICODE, FONT, PGF, VTK, DOCTEST, INFO,
    ) = [2**i for i in range(28)]
    ALL = -1

    @classmethod
    def item(cls, name):
        """Convert a string to a DEBUG item

        The string is case insensitive.

        Raises
        ------
        ValueError
            If the name is not a DEBUG item.
        """
        try:
            return cls[name.upper()]
        except KeyError:
            raise ValueError(f"No such {cls.__name__} item: {name}")

    @classmethod
    def level(cls, names):
        """Return the combined debug level for a list of debug items

        Raises
        ------
        ValueError
            If any of the names is not a DEBUG item.
        """
        return reduce(or_, [cls.item(name) for name in names])


def debug(s, level=DEBUG.ALL):
    """Print a debug message

    Parameters
    ----------
    s: str
        The message to be printed
    level: DEBUG
        The DEBUG level in which to print this message. Messages of a certain
        level are only printed if the global variable pf.options.debuglevel
        has the specified DEBUG flag set.
    """
    if level in options.debuglevel:
        print(f"{level}: {s}")


def debugt(s, level):
    """Print a debug message with timer"""
    debug(f"{time.time()}: {s}")


def verbosity(verbose):
    """Check that verbosity is equal or higher than given value"""
    return options.verbose >= verbose


def verbose(verbose, *args, **kargs):
    """Print a message if the verbosity is high enough"""
    if options.verbose >= verbose:
        print(*args, **kargs)


def run(args=[]):
    from pyformex import main
    main.run(args)


###################################################################
## Parse the options
####################

#
# TODO: this was put here when we had an option to select the Path
# implementation, which had to be done before loading the Path class
# but now it could probably be move elsewhere
#
if len(sys.argv) > 1 and sys.argv[1].endswith('pyformex'):
    from pyformex.options import parseOptions
    if not parseOptions(sys.argv[2:]):
        sys.exit(1)

############################################################################
###  Load default configuration
### (pyFormex does not work if not loaded)
##########################################

from pyformex.config import Config  # noqa: E402

# Create a config instance
cfg = Config()
cfg['pyformexdir'] = pyformexdir
# load the factory defaults
defaults = cfg['pyformexdir'] / 'pyformexrc'
cfg.load(defaults)
# The default user config path (do not change!)
cfg['userprefs'] = cfg['userconfdir'] / 'pyformex.conf'

# Set the current session configurations in pyformex module
prefcfg = None  # the preferenced configuration
refcfg = None  # the reference configuration
preffile = None  # the file where the preferenced configuration will be saved

#print(f"Current path is {Path.cwd()}")
#def except_hook(cls, exception, traceback):
#    sys.__excepthook__(cls, exception, traceback)

# if __name__ == "__main__":

#     import sys
#     sys.excepthook = except_hook

#print(f"EXCEPTHOOK builtin {id(sys.__excepthook__)}")
#print(f"EXCEPTHOOK active  {id(sys.excepthook)}")

# End
