# flake8: noqa
##
##  SPDX-FileCopyrightText: © 2007-2021 Benedict Verhegghe <bverheg@gmail.com>
##  SPDX-License-Identifier: GPL-3.0-or-later
##
##  This file is part of pyFormex 3.0  (Mon Nov 22 14:32:59 CET 2021)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: https://pyformex.org
##  Project page: https://savannah.nongnu.org/projects/pyformex/
##  Development: https://gitlab.com/bverheg/pyformex
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##
"""Default imports in the scripting language.

This module defines which symbols are always imported when
you run a script, and thus constitute the core part
of the pyFormex scripting language.

For an app, you get this definitions if you add::

  from pyformex.script import *

or, to also enable the GUI functionality::

  from pyformex.gui.draw import *

Warning
-------
This is subject to future changes.
"""

import numpy as np

import pyformex as pf
from pyformex import utils
from pyformex import software
from pyformex import arraytools as at
from pyformex import simple
from pyformex import fileread
from pyformex import filewrite
from pyformex import saveload

from pyformex.path import Path
from pyformex.timer import Timer
from pyformex.utils import system, command
from pyformex.colors import *
from pyformex.coords import *
from pyformex.formex import *
from pyformex.mesh import Mesh
from pyformex.olist import List
from pyformex.varray import Varray
from pyformex.connectivity import Connectivity
from pyformex.elements import Elems
from pyformex.trisurface import TriSurface
from pyformex.coordsys import CoordSys
from pyformex.plugins.curve import PolyLine, BezierSpline
from pyformex.project import Project
from pyformex.field import Field

from numpy import pi, sin, cos, tan, arcsin, arccos, arctan, arctan2, sqrt, abs
from numpy.linalg import norm

if pf.cfg['scriptmode'] == 'lazy':
    from pyformex.arraytools import *
    from numpy import *
    from pyformex.geomfile import *
    from pyformex.saveload import savePZF, loadPZF


def readGeometry(filename, filetype=None, target=None, **kargs):
    """Read geometry from a stored file.

    This is a wrapper function over several other functions specialized
    for some file type. Some file types require the existence of more
    than one file, may need to write intermediate files, or may call
    external programs. Some file types contain a single geometric object,
    others can contain collections of objects. Some formats store other
    than geometric information, like colors, normals or field variables.

    Parameters
    ----------
    filename: :term:`path-like`
        The name of the file to read. If the filetype requires the existence
        of more than one file (e.g. tetgen), only a single one should be
        specified and the others are derived from it. Usually, the filename
        has a suffix that hints at the filetype.
        Most of the pyFormex readers transparently support decompression of
        compressed files. This works if the filename ends on '.gz' or '.bz2'.
    filetype: str, optional
        The file type. If not specified, it is derived from the filename
        suffix. Currently the following file types can be handled:

        pzf: pyFormex Zip File, see :func:`saveload.loadPZF`,
        pgf: pyFormex Geometry File, see :func:`fileread.readPGF`,
        obj: Wavefront .obj file, see :func:`fileread.readOBJ`,
        off: Geomview .off file, see :func:`fileread.readOFF`,
        ply: Polygon File Format, see :func:`fileread.readPLY`,
        neu: Gambit Neutral file, see :func:`fileread.readNEU`,
        stl: stereolithography file, see :func:`fileread.readSTL`,
        gts: libgts file format, see :func:`fileread.readGTS`,
        surface: a generic filetype for any of the following formats
            holding a triangulated surface: stl, gts, off, neu,
            see :func:`TriSurface.read`,
        inp: Abaqus and CalculiX input format, see :func:`fileread.readINP`,
        tetgen: a generic filetype for several tetgen files,
            see :func:`plugins.tetgen.readTetgen`
    target: str, optional
        The target object type. Currently, the only accepted value is
        'surface'. In combination with a file of type 'obj', 'off', 'ply'
        or 'neu', it will return TriSurface objects instead of the more general
        Mesh objects.
    mplex: int, optional
        For 'obj', 'off' and 'ply' types, specifies the maximum accepted
        plexitude of the objects. Larger plexitudes are decomposed into
        smaller ones.

    Returns
    -------
    dict
        A dictionary with named geometry objects read from the file.

    """
    filename = Path(filename)
    res = {}
    compr = None
    if filetype is None:
        filetype, compr = filename.ftype_compr()
    else:
        filetype = filetype.lower()

    print(f"Reading file {filename} of type {filetype.upper()}")

    if filetype == 'pzf':
        from pyformex import saveload
        res = saveload.loadPZF(filename)

    elif filetype == 'pgf':
        res = fileread.readPGF(filename)

    elif filetype in ['obj', 'off', 'ply'] and target != 'surface':
        reader = {'obj':fileread.readOBJ,
                  'off':fileread.readOFF,
                  'ply':fileread.readPLY,
                  }[filetype]
        coords, elems = reader(filename, mplex=kargs.get('mplex',None))
        coords = Coords(coords)
        name = next(utils.autoName(Mesh))
        for nplex in elems:
            if nplex <= 4:
                els = Elems(elems[nplex], eltype=None)
                mesh = Mesh(coords, els)
                res[f"{name}-{mesh.elems.eltype}"] = mesh
            else:
                pf.verbose(1, f"Skipping {elems[nplex].shape[0]}"
                           f" elements of plexitude {nplex}")

    elif filetype == 'neu' and target != 'surface':
        mesh = fileread.readNEU(filename)
        res[f"{filename.stem}"] = mesh

    # Beware! this should come after 'obj', otherwise
    # non-triangles in the .obj file will be skipped
    elif filetype in utils.fileTypes('surface') or \
         filetype in utils.fileTypes('tetsurf') or \
         filetype in utils.fileTypes('vtk'):
        S = TriSurface.read(filename)
        name = next(utils.autoName(TriSurface))
        res = {name: S}

    elif filetype == 'inp':
        parts = fileread.readINP(filename)
        color_by_part = len(parts) > 1
        j = 0
        for name in parts:
            for i, mesh in enumerate(parts[name].meshes()):
                p = j if color_by_part else i
                print("Color %s" % p)
                res["%s-%s" % (name, i)] = mesh.setProp(p)
            j += 1

    elif filetype in utils.fileTypes('tetgen'):
        from pyformex.plugins import tetgen
        res = tetgen.readTetgen(filename)

    else:
        raise ValueError(
            f"Can not import {filetype} file {filename}")

    return res


def writeGeometry(filename, obj, ftype=None, compr=False, **kargs):
    """Write geometry items to the specified file.

    This is a wrapper function around the many dedicated file
    exporter functions. It provides a single interface to export
    geometry to one of many file formats:

    - pyFormex native formats: PZF, PGF
    - widely supported geometry formats: OBJ, OFF, PLY, STL, VTP, VTK
    - GTS, NEU(gambit), SMESH(tetgen), INP(Abaqus)
    - many more formats supported through mesh_io



    Parameters
    ----------
    filename:
    obj:
    ftype: str, optional
    compr: str, optional
    writer: str, optional
        If not specified, internal pyFormex functions are used to do
        the export.
        If specified, an external library is used to do the export.
        Currently the only available value is 'meshio'. You need to
        have the meshio library installed to use it.
        See https://github.com/nschloe/meshio.
    mode: str
        'ascii', 'binary' or 'shortascii'


    Returns the number of objects written.
    """
    print(f"Exporting {list(obj.keys())}")
    nobj = 0
    filename = Path(filename)
    writer = kargs.pop('writer', None)
    if writer == 'meshio':
        from pyformex.plugins.mesh_io import writeMesh
        for name in obj:
            writeMesh(filename, obj[name])
            return 1

    if ftype is None:
        ftype, compr = filename.ftype_compr()

    print("Writing file of type %s" % ftype)

    if ftype == 'pzf':
        saveload.savePZF(filename, **obj)
        nobj = len(obj)

    elif ftype == 'pgf':
        mode = kargs.pop('mode', 'ascii')
        sep = '' if mode=='binary' else ' '
        shortlines = mode.startswith('short')
        nobj = filewrite.writePGF(filename, obj, sep=sep, shortlines=shortlines)

    elif ftype in ('obj', 'off', 'ply', 'neu'):
        for name in obj: # There is only one and it's a Mesh
            mesh = obj[name]
            if ftype == 'obj':
                filewrite.writeOBJ(filename, mesh)
            elif ftype == 'off':
                filewrite.writeOFF(filename, mesh)
            elif ftype == 'neu':
                filewrite.writeNEU(filename, mesh)
            elif ftype == 'ply':
                filewrite.writePLY(filename, mesh.coords,
                                   {mesh.nplex():mesh.elems})
            nobj += 1

    elif ftype == 'stl':
        if kargs['binary']:
            from pyformex import colors
            ftype = 'stlb'
            color = colors.RGBAcolor(kargs['color'], kargs['alpha'])
            print(f"Also saving color {color}")
        else:
            color = None
        for name in obj:  # There is only one and it's a TriSurface
            S = obj[name]
            S.write(filename, ftype, color=color)
            nobj += 1

    elif ftype in ('gts', 'neu', 'smesh', 'vtp', 'vtk'):
        for name in obj: # There is only one and it's a TriSurface
            S = obj[name]
            S.write(filename, ftype)
            nobj += 1

    elif ftype == 'inp':
        from pyformex.plugins import fe_abq
        for name in obj: # There is only one and it's a Mesh
            M = obj[name]
            fe_abq.exportMesh(filename, M, eltype=kargs['eltype'])
            nobj += 1

    else:
        error("Don't know how to export in '%s' format" % ftype)

    return nobj


# Retained for compatibility 2021-11-05
# TODO: to be removed in 3.1 or after 2022-06-01
@utils.deprecated("The use of readGeomFile is deprecated. Either use the"
                  " more general readGeometry, or fileread.readPGF")
def readGeomFile(*args, **kargs):
    return fileread.readPGF(*args, **kargs)

@utils.deprecated("The use of writeGeomFile is deprecated. Either use the"
                  " more general writeGeometry, or fileread.writePGF")
def writeGeomFile(*args, **kargs):
    return filewrite.writePGF(*args, **kargs)

# End
