#
##
##  SPDX-FileCopyrightText: © 2007-2022 Benedict Verhegghe <bverheg@gmail.com>
##  SPDX-License-Identifier: GPL-3.0-or-later
##
##  This file is part of pyFormex 3.2  (Wed Nov  9 17:44:14 CET 2022)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: https://pyformex.org
##  Project page: https://savannah.nongnu.org/projects/pyformex/
##  Development: https://gitlab.com/bverheg/pyformex
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##
"""NurbsInterpolation

This example illustrates the creation of NurbsCurves interpolating or
approximating a given point set.
"""
_status = 'checked'
_level = 'advanced'
_topics = ['geometry', 'curve']
_techniques = ['nurbs', 'connect', 'border', 'frenet']
_name = 'NurbsInterpolation'

from pyformex.gui.draw import *
from pyformex.plugins.nurbs import *

np.set_printoptions(precision=3)

point_set = {
#    'pt_9.1': [
#        ( 3.5,),
#        ( 4.9,),
#        ( 7.0,),
#        ( 9.0,),
#        (11.0,),
#    ],
    'pt_9.3': (
        (3.9, -6.4),
        (4.0, -5.2),
        (5.3, -3.5),
        (8.0, -4.2),
        (9.4, -5.6),
        (10.7, -5.3),
        (12.2, -5.5),
    ),
    'ex_9.1': ((0,0), (3,4), (-1,4), (-4,0), (-4,-3)),
    'pt_9.7': (
        (-12.1, -8.6),
        (-12.1, -7.8),
        (-12.1, -4.9),
        (-12.1, -4.0),
        (-10.7, -2.6),
        (- 9.3, -2.6),
        (- 8.6, -3.3),
        (- 8.6, -4.0),
        (- 8.6, -4.8),
        (- 7.9, -5.6),
        (- 7.2, -5.6),
        (- 4.4, -5.6),
        (- 3.5, -5.6),
    ),
    'pt_9.8': (
        (0.0, 2.6),
        (0.4, 4.3),
        (2.8, 5.0),
        (4.2, 3.0),
        (7.1, 2.5),
        (8.5, 6.3),
        (7.0, 6.5),
    ),
    'pt_9.10': (
        (0.0, -7.2),
        (0.5, -5.5),
        (3.7, -4.8),
        (4.5, -6.3),
        (5.2, -7.2),
        (9.0, -5.1),
        (7.5, -3.3),
    ),
    'pt_9.11': (
        (-11.4, 4.4),
        (-10.2, 7.5),
        ( -7.8, 5.7),
        ( -6.5, 3.8),
        ( -4.1, 7.0),
    ),
    'pt_9.20': (
        (-12.2, 2.5),
        (-11.6, 5.5),
        ( -9.5, 7.3),
        ( -6.0, 7.3),
        ( -4.2, 5.5),
        ( -3.9, 3.3),
    ),
    'pt_9.21': (
        ( 4.7, 3.6),
        ( 4.1, 6.4),
        ( 6.2, 9.0),
        ( 9.1, 9.3),
        (11.7, 7.2),
        (11.0, 3.7),
    ),
    'pt_9.24': (
        ( 0.0, 0.0),
        ( 0.0, 1.4),
        ( 0.0, 2.8),
        ( 2.8, 2.8),
        ( 2.8, 1.4),
        ( 2.8, 0.0),
        ( 5.6, 0.0),
        ( 8.4, 0.0),
        ( 8.4, 1.4),
        ( 8.4, 4.4),
    ),
}

point_patterns = [
    '51414336',
    '51i4143I36',
    '2584',
    '25984',
    '184',
    '514',
    '1234',
    '5858585858',
    '12345678',
    '121873',
    '1218973',
    '8585',
    '85985',
    '214121',
    '214412',
    '151783',
    'ABCDABCD',
    ]

point_ids = utils.hsorted(point_set.keys()) + point_patterns


def drawNurbs(points, degree, exp, t0=None, t1=None, Clear=False, show_control=False):
    if Clear or show_control:
        clear()

    print("Selected point set: %s" % points)
    if points in point_set:
        X = Coords(point_set[points])
    else:
        X = pattern(points)
    draw(X, marksize=10, bbox='auto', view='front')
    drawNumbers(X, prefix='Q', trl=[0.02, 0.02, 0.])
    t1 = None
    N, u = globalInterpolationCurve(X, degree=degree, exp=exp, t0=t0, t1=t1, return_param=True)
    print(N)
    print(u)
    draw(N, color=degree)
    if show_control and degree > 1:
        P = N.coords
        draw(PolyLine(P), color=red)
        drawNumbers(P, color=red)

dialog = None


def clean_results(res):
    print(res)
    set_t0_t1 = res.pop('set_t0_t1', False)
    if not set_t0_t1:
        res['t0'] = res['t1'] = None
    return res


def close():
    global dialog
    if dialog:
        dialog.close()
        dialog = None
    # Release script lock
    scriptRelease(__file__)


def show():
    if dialog.validate():
        res = clean_results(dialog.results)
        drawNurbs(**res)

def showAll():
    if dialog.validate():
        res = clean_results(dialog.results)
        for points in point_ids:
            if len(points) <= res['degree']:
                print(f"Skipping point set: {points}: {len(points) + 1}"
                      f" points required for degree {res['degree']}")
                continue
            res['points'] = points
            drawNurbs(**res)
            sleep(1)


def timeOut():
    print(f"TIMEOUT CALLED")
    try:
        showAll()
        wait()
    finally:
        close()


def run():
    global dialog
    clear()
    setDrawOptions({'bbox': None})
    linewidth(2)
    flat()

    ## Closing this dialog should release the script lock
    items=[
        _I('points', text='Point set', choices=point_ids),
        _I('degree', 3),
        _I('exp', 0.7),
        _I('set_t0_t1', False, text = 'Set end tangents'),
        _I('t0', (1.,0.,0.), itemtype='point', text='Start tangent'),
        _I('t1', (1.,0.,0.), itemtype='point', text='End tangent'),
        _I('Clear', True),
        _I('show_control', False, text='Show control points'),
    ]
    dialog = Dialog(
        items=items, caption=_name, store=_name+'_ data',
        enablers=(('set_t0_t1', True, 't0', 't1'),),
        actions=[('Close', close), ('Clear', clear),
                 ('Show All', showAll), ('Show', show)],
        default='Show',
    )
    dialog.show(timeoutfunc=timeOut)

    # Block other scripts
    scriptLock(__file__)


if __name__ == '__draw__':
    run()
# End
