#
# NPI - Calculatrice en Notation Polonaise Inverse. 
# Copyright (C) 2005-2011 MiKael NAVARRO
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""NPI - Reverse Polish Notation Calculator. 
Copyright (C) 2005-2011 MiKael NAVARRO

Arithmetic commands.
"""

# Include directives
from npi_errors import TooFewArguments, InfiniteResult  # NPI errors
from npi_utils import func_name  # decorator: set func_name


#
# +
#
@func_name("+")
def npi_add(stack):
  """Add X to Y
  
  >>> from stack import Stack
  >>> npi_add(Stack([3, 2]))  # 3 2 +
  [5]
  >>> npi_add(Stack([3]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: + Error: Too Few Arguments
  """

  if len(stack) >= 2:
    op1, op2 = stack.pop(), stack.pop()
    res = op1 + op2
    stack.push(res)
  else:
    raise TooFewArguments(npi_add)

  return stack


#
# -
#
@func_name("-")
def npi_subtract(stack):
  """Subtract X from Y or return -X
  
  >>> from stack import Stack
  >>> npi_subtract(Stack([3, 2]))  # 3 2 -
  [1]
  >>> npi_subtract(Stack([3]))  # 3 -
  [-3]
  >>> npi_subtract(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: - Error: Too Few Arguments
  """

  if len(stack) >= 2:
    op1, op2 = stack.pop(), stack.pop()
    res = op2 - op1
    stack.push(res)
  elif len(stack) == 1:
    op1 = stack.pop()
    res = -1 * op1
    stack.push(res)
  else:
    raise TooFewArguments(npi_subtract)

  return stack


#
# *
#
@func_name("*")
def npi_multiply(stack):
  """Multiply X with Y

  >>> from stack import Stack
  >>> npi_multiply(Stack([3, 2]))  # 3 2 *
  [6]
  >>> npi_multiply(Stack([3]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: * Error: Too Few Arguments
  """
  
  if len(stack) >= 2:
    op1, op2 = stack.pop(), stack.pop()
    res = op1 * op2
    stack.push(res)
  else:
    raise TooFewArguments(npi_multiply)

  return stack


#
# /
#
@func_name("/")
def npi_divide(stack):
  """Divide Y by X

  >>> from stack import Stack
  >>> npi_divide(Stack([3, 2]))  # 3 2 /
  [1.5]
  >>> npi_divide(Stack([1]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: / Error: Too Few Arguments
  >>> npi_divide(Stack([1, 0]))
  Traceback (most recent call last):
  ...
  npi_errors.InfiniteResult: / Error: Infinite Result
  """
  
  if len(stack) >= 2:
    op1, op2 = stack.pop(), stack.pop()
    
    if op1 == 0:  # Y/0
      stack.push(op2)
      stack.push(op1)
      raise InfiniteResult(npi_divide)

    res = op2 / op1
    stack.push(res)
  else:
    raise TooFewArguments(npi_divide)

  return stack


#
# +/-
#
@func_name("neg")
def npi_neg(stack):
  """Return -X

  >>> from stack import Stack
  >>> npi_neg(Stack([2]))  # 2 neg
  [-2]
  >>> npi_neg(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: neg Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    op1 = stack.pop()
    res = -1 * op1
    stack.push(res)
  else:
    raise TooFewArguments(npi_neg)

  return stack


#
# Integer division (Y/X).
#
@func_name("div")
def npi_div(stack):
  """Integer division Y/X

  >>> from stack import Stack
  >>> npi_div(Stack([3, 2]))  # 3 2 div
  [1]
  >>> npi_div(Stack([1]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: div Error: Too Few Arguments
  >>> npi_div(Stack([3, 0]))
  Traceback (most recent call last):
  ...
  npi_errors.InfiniteResult: div Error: Infinite Result
  """
  
  if len(stack) >= 2:
    op1, op2 = stack.pop(), stack.pop()

    if op1 == 0:  # Y/0
      stack.push(op2)
      stack.push(op1)
      raise InfiniteResult(npi_div)

    res = int(op2 // op1)
    stack.push(res);
  else:
    raise TooFewArguments(npi_div)

  return stack


#
# Inverse (1/X).
#
@func_name("inv")
def npi_inv(stack):
  """Inverse 1/X

  >>> from stack import Stack
  >>> npi_inv(Stack([2]))  # 2 inv
  [0.5]
  >>> npi_inv(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: inv Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    op1 = stack.pop()
    
    if op1 == 0:  # 1/0
      stack.push(op1)
      raise InfiniteResult(npi_inv)

    res = 1.0 / op1
    stack.push(res)
  else:
    raise TooFewArguments(npi_inv)

  return stack


#
# Sum up the stack.
#
@func_name("sum")
def npi_sum(stack):
  """Sum up the stack

  >>> from stack import Stack
  >>> npi_sum(Stack([3, 2, -1]))  # 3 2 -1 sum
  [4]
  >>> npi_sum(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: sum Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    sum = 0
    while len(stack):
      sum += stack.pop()
    stack.push(sum)
  else:
    raise TooFewArguments(npi_sum)

  return stack


#
# Self-test.
#
def _test():
  import doctest
  print("Testing 'Arithmetic' commands...", end="")
  (failure_count, test_count) = doctest.testmod()
  if not failure_count: print("Okey")
  else: print("Ko!")


#
# External entry point.
#
if __name__ == "__main__":
  _test()
