#
# NPI - Calculatrice en Notation Polonaise Inverse. 
# Copyright (C) 2005-2011 MiKael NAVARRO
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""NPI - Reverse Polish Notation Calculator. 
Copyright (C) 2005-2011 MiKael NAVARRO

Conversion commands.
"""

# Include directives
from npi_errors import TooFewArguments  # NPI errors
from npi_utils import func_name  # decorator: set func_name
from math import degrees, radians


#
# Degrees.
#
@func_name("deg")
def npi_deg(stack):
  """Radian to degree conversion

  >>> from stack import Stack
  >>> npi_deg(Stack([3.14159]))  # doctest:+ELLIPSIS
  [179.99...]
  >>> npi_deg(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: deg Error: Too Few Arguments
  """

  if len(stack) >= 1:
    stack.push(degrees(stack.pop()))
  else:
    raise TooFewArguments(npi_deg)

  return stack


#
# Radians.
#
@func_name("rad")
def npi_rad(stack):
  """Degree to radian conversion

  >>> from stack import Stack
  >>> npi_rad(Stack([180]))  # doctest:+ELLIPSIS
  [3.14...]
  >>> npi_rad(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: rad Error: Too Few Arguments
  """

  if len(stack) >= 1:
    stack.push(radians(stack.pop()))
  else:
    raise TooFewArguments(npi_rad)

  return stack


# Euro -> Frf conversion
_euro_frf = 6.55957

#
# Euros.
#
@func_name("f2e")
def npi_euro(stack):
  """Frf to Euro conversion

  >>> from stack import Stack
  >>> npi_euro(Stack([10]))  # doctest:+ELLIPSIS
  [1.52...]
  >>> npi_euro(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: f2e Error: Too Few Arguments
  """

  if len(stack) >= 1:
    stack.push(stack.pop() / _euro_frf)
  else:
    raise TooFewArguments(npi_euro)

  return stack

  
#
# Francs (francais).
#
@func_name("e2f")
def npi_frf(stack):
  """Euro to Frf conversion

  >>> from stack import Stack
  >>> npi_frf(Stack([1]))
  [6.55957]
  >>> npi_frf(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: e2f Error: Too Few Arguments
  """

  if len(stack) >= 1:
    stack.push(stack.pop() * _euro_frf)
  else:
    raise TooFewArguments(npi_frf)

  return stack


#
# Self-test.
#
def _test():
  import doctest
  print("Testing 'Conversion' commands...", end="")
  (failure_count, test_count) = doctest.testmod()
  if not failure_count: print("Okey")
  else: print("Ko!")


#
# External entry point.
#
if __name__ == "__main__":
  _test()
