#
# NPI - Calculatrice en Notation Polonaise Inverse. 
# Copyright (C) 2005-2011 MiKael NAVARRO
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""NPI - Reverse Polish Notation Calculator. 
Copyright (C) 2005-2011 MiKael NAVARRO

Exponential commands.
"""

# Include directives
from npi_errors import TooFewArguments  # NPI errors
from npi_utils import func_name  # decorator: set func_name
from math import exp, log, log10, pow, sqrt, ldexp, frexp


#
# Exp (e^X).
#
@func_name("exp")
def npi_exp(stack):
  """Exponential e^X

  >>> from stack import Stack
  >>> npi_exp(Stack([3]))  # doctest:+ELLIPSIS
  [20...]
  >>> npi_exp(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: exp Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(exp(stack.pop()))
  else:
    raise TooFewArguments(npi_exp)
  
  return stack


#
# Ln.
#
@func_name("ln")
def npi_ln(stack):
  """Logarithm

  >>> from stack import Stack
  >>> npi_ln(Stack([5]))  # doctest:+ELLIPSIS
  [1.6...]
  >>> npi_ln(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: ln Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(log(stack.pop()))
  else:
    raise TooFewArguments(npi_ln)
  
  return stack


#
# Log.
#
@func_name("log")
def npi_log(stack):
  """Logarithm (base 10)

  >>> from stack import Stack
  >>> npi_log(Stack([5]))  # doctest:+ELLIPSIS
  [0.69...]
  >>> npi_log(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: log Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(log10(stack.pop()))
  else:
    raise TooFewArguments(npi_log)
  
  return stack


#
# Pow (Y^X).
#
@func_name("pow")
def npi_pow(stack):
  """Y to the power of X

  >>> from stack import Stack
  >>> npi_pow(Stack([2, 8]))
  [256.0]
  >>> npi_pow(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: pow Error: Too Few Arguments
  """
  
  if len(stack) >= 2:
    n = stack.pop()
    stack.push(pow(stack.pop(), n))
  else:
    raise TooFewArguments(npi_pow)
  
  return stack


#
# Pow10 (10^X).
#
@func_name("pow10")
def npi_pow10(stack):
  """10 to the power of X

  >>> from stack import Stack
  >>> npi_pow10(Stack([8]))
  [100000000.0]
  >>> npi_pow10(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: pow10 Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(pow(10.0, stack.pop()))
  else:
    raise TooFewArguments(npi_pow10)
  
  return stack


#
# Sqrt.
#
@func_name("sqrt")
def npi_sqrt(stack):
  """Square root

  >>> from stack import Stack
  >>> npi_sqrt(Stack([64]))
  [8.0]
  >>> npi_sqrt(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: sqrt Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(sqrt(stack.pop()))
  else:
    raise TooFewArguments(npi_sqrt)
  
  return stack


#
# Root (= Y^(1/X)).
#
@func_name("root")
def npi_root(stack):
  """X root Y

  >>> from stack import Stack
  >>> npi_root(Stack([27, 3]))
  [3.0]
  >>> npi_root(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: root Error: Too Few Arguments
  """
  
  if len(stack) >= 2:
    n = stack.pop()
    stack.push(pow(stack.pop(), 1.0 / n))
  else:
    raise TooFewArguments(npi_root)
  
  return stack


#
# Square (X^2).
#
@func_name("sq")
def npi_sq(stack):
  """X squared

  >>> from stack import Stack
  >>> npi_sq(Stack([8]))
  [64]
  >>> npi_sq(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: sq Error: Too Few Arguments
  """

  if len(stack) >= 1:
    stack.push(stack.pop()**2)
  else:
    raise TooFewArguments(npi_sq)
  
  return stack


#
# Ldexp (X.2^Y).
#
@func_name("ldexp")
def npi_ldexp(stack):
  """X.2^Y

  >>> from stack import Stack
  >>> npi_ldexp(Stack([2, 8]))
  [32.0]
  >>> npi_ldexp(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: ldexp Error: Too Few Arguments
  """
  
  if len(stack) >= 2:
    op = stack.pop()
    stack.push(ldexp(op, int(stack.pop())))
  else:
    raise TooFewArguments(npi_ldexp)
  
  return stack


#
# Frexp.
#
@func_name("frexp")
def npi_frexp(stack):
  """Factorize X as x.2^y

  >>> from stack import Stack
  >>> npi_frexp(Stack([32]))
  [6, 0.5]
  >>> npi_frexp(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: frexp Error: Too Few Arguments
  """

  if len(stack) >= 1:
    x, exp = frexp(stack.pop())
    stack.push(exp)
    stack.push(x)
  else:
    raise TooFewArguments(npi_frexp)
  
  return stack


#
# Self-test.
#
def _test():
  import doctest
  print("Testing 'Exponential' commands...", end="")
  (failure_count, test_count) = doctest.testmod()
  if not failure_count: print("Okey")
  else: print("Ko!")


#
# External entry point.
#
if __name__ == "__main__":
  _test()
