#
# NPI - Calculatrice en Notation Polonaise Inverse. 
# Copyright (C) 2005-2013 MiKael NAVARRO
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""NPI - Reverse Polish Notation Calculator. 
Copyright (C) 2005-2013 MiKael NAVARRO

Loaning calc.
"""

# Include directives
from npi_errors import TooFewArguments, BadArgumentValue, NotYetImplemented  # NPI errors
from npi_utils import func_name  # decorator: set func_name
from math import pow


#
# Monthly loan.
#
@func_name("monthly_loan")
def npi_monthly_loan(stack):
  """Find the monthly payment
  from: amount borrowed (K), annual interest rate (%), term of the loans (months)
  return: monthly payment, total interest paid
  
  m = K * ( t/12 ) / ( 1 - ( 1 + t/12)^-n )
    = K * t * (1+t)^n / ( (1+t)^n -1 )
  I = m * n - K
    = n * (K * t * (1+t)^n ) / ( (1+t)^n - 1) - K
  
  >>> from stack import Stack
  >>> npi_monthly_loan(Stack([150000, 4.8, 240]))  # doctest:+ELLIPSIS
  [973.4362047682152, 83624.68914437166]
  >>> npi_monthly_loan(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: monthly_loan Error: Too Few Arguments
  """

  if len(stack) >= 3:
    n = float(stack.pop())  # number of months
    t = float(stack.pop())  # annual interest rate (%)
    k = float(stack.pop())  # amount borrowed

    # Calculate monthly payment
    method = 2
    if method == 1:
      stack.push( (k * t/100/12) / (1 - pow(1+t/100/12,-n)) )  # formula 1
    elif method == 2:
      stack.push( (k * t/100/12 * pow(1+t/100/12,n)) / (pow(1+t/100/12, n) -1) )  # formula 2
    else:
      # npi.py -i "150000 4.8 100 / 12 / * 1 4.8 100 / 12 / + 240 pow * 1 4.8 100 / 12 / + 240 pow 1 - / q"
      from cmd_arith import npi_add, npi_subtract, npi_multiply, npi_divide, npi_neg
      from cmd_exp import npi_pow
      stack.push(k)
      stack.push(t)
      stack.push(100)
      npi_divide(stack)
      stack.push(12)
      npi_divide(stack)
      npi_multiply(stack)
      stack.push(1)
      stack.push(t)
      stack.push(100)
      npi_divide(stack)
      stack.push(12)
      npi_divide(stack)
      npi_add(stack)
      stack.push(n)
      npi_pow(stack)
      npi_multiply(stack)
      stack.push(1)
      stack.push(t)
      stack.push(100)
      npi_divide(stack)
      stack.push(12)
      npi_divide(stack)
      npi_add(stack)
      stack.push(n)
      npi_pow(stack)
      stack.push(1)
      npi_subtract(stack)
      npi_divide(stack)

    # Calculate interest paid
    m = stack.pop()
    i = m * n - k
    stack.push(m)
    stack.push(i)
 
  else:
    raise TooFewArguments(npi_monthly_loan)

  return stack


#
# Amount borrowed.
#
@func_name("amt_borrowed")
def npi_amount_borrowed(stack):
  """Find the amount borrowed
  from: monthly payment, interest rate (%), number of months
  return: loan amount

  K = m * ( (1+t)^n - 1 ) / ( t * (1+t)^n ) 
  
  >>> from stack import Stack
  >>> npi_amount_borrowed(Stack([973.44, 4.8, 240]))  # doctest:+ELLIPSIS
  [150000...]
  >>> npi_amount_borrowed(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: amt_borrowed Error: Too Few Arguments
  """

  if len(stack) >= 3:
    n = float(stack.pop())  # number of months
    t = float(stack.pop())  # annual interest rate (%)
    m = float(stack.pop())  # monthly payment

    stack.push(m * ( pow(1+t/100/12, n) -1 ) / ( t/100/12 * pow(1+t/100/12, n) ) )

  else:
    raise TooFewArguments(npi_amount_borrowed)

  return stack


#
# Number of payments.
#
@func_name("num_payments")
def npi_num_payments(stack):
  """Find the number of payments
  from: loan amount, interest rate (%), monthly payment
  return: number of months

  n = ( ln(m) - ln(m - K * t) ) / ln(1+t)
  
  >>> from stack import Stack
  >>> npi_num_payments(Stack([150000, 4.8, 973.44]))  # doctest:+ELLIPSIS
  Traceback (most recent call last):
  ...
  npi_errors.NotYetImplemented: num_payments Error: Not Yet Implemented
  >>> npi_num_payments(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: num_payments Error: Too Few Arguments
  """

  if len(stack) >= 3:
    m = float(stack.pop())  # monthly payment
    t = float(stack.pop())  # annual interest rate (%)
    k = float(stack.pop())  # loan amount

    raise NotYetImplemented(npi_num_payments)

  else:
    raise TooFewArguments(npi_num_payments)

  return stack


#
# Interest rate.
#
@func_name("interest_rate")
def npi_interest_rate(stack):
  """Find the Interest Rate
  from: loan amount, monthly payment, number of months
  return: interest rate (%)

  For given K, m and n, one can only solve
  the following equation for t by numerical means.
    ( t * (1+t)^n ) / ( (1+t)^n -1 ) - m/K = 0
    
  Given the rather smooth behavior of this equation,
  this calculator employs the Newton-Raphson method
  with an educated initial guess:

    tguess = 2 * (n * m - K) / (n * K) 

  >>> from stack import Stack
  >>> npi_interest_rate(Stack([150000, 973.44, 240]))  # doctest:+ELLIPSIS
  Traceback (most recent call last):
  ...
  npi_errors.NotYetImplemented: interest_rate Error: Not Yet Implemented
  >>> npi_interest_rate(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: interest_rate Error: Too Few Arguments
  """

  if len(stack) >= 3:
    n = float(stack.pop())  # number of months
    m = float(stack.pop())  # monthly payment
    k = float(stack.pop())  # loan amount

    i = n * m - k
    t = (2 * i) / (n * k)
    stack.push( t * 100 * 12 )
    raise NotYetImplemented(npi_interest_rate)

  else:
    raise TooFewArguments(npi_interest_rate)

  return stack


#
# Self-test.
#
def _test():
  import doctest
  print("Testing 'loaning' utilities...", end="")
  (failure_count, test_count) = doctest.testmod()
  if not failure_count: print("Okey")
  else: print("Ko!")


#
# External entry point.
#
if __name__ == "__main__":
  _test()
