#
# NPI - Calculatrice en Notation Polonaise Inverse. 
# Copyright (C) 2005-2011 MiKael NAVARRO
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""NPI - Reverse Polish Notation Calculator. 
Copyright (C) 2005-2011 MiKael NAVARRO

Proba commands.
"""

# Include directives
from npi_errors import TooFewArguments, BadArgumentValue  # NPI errors
from npi_utils import func_name  # decorator: set func_name


#
# Factorial.
#
def _factorial(n):
  if n == 0:
    return 1
  else:
    return n * _factorial(n-1)


# 
# ! (permutation).
#
@func_name("!")
def npi_fact(stack):
  """Factorial X

  >>> from stack import Stack
  >>> npi_fact(Stack([32]))
  [263130836933693530167218012160000000]
  >>> npi_fact(Stack([-3]))
  Traceback (most recent call last):
  ...
  npi_errors.BadArgumentValue: ! Error: Bad Argument Value
  >>> npi_fact(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: ! Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    n = int(stack.pop())
    if n >= 0: 
      stack.push(_factorial(n))
    else:
      stack.push(n)
      raise BadArgumentValue(npi_fact)
  else:
    raise TooFewArguments(npi_fact)

  return stack


#
# Combination C(Y,X).
#
@func_name("comb")
def npi_comb(stack):
  """Combination (Y,X)

  >>> from stack import Stack
  >>> npi_comb(Stack([49, 5]))
  [1906884.0]
  >>> npi_comb(Stack([5, 49]))
  Traceback (most recent call last):
  ...
  npi_errors.BadArgumentValue: comb Error: Bad Argument Value
  >>> npi_comb(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: comb Error: Too Few Arguments
  """
  
  if len(stack) >= 2:
    k, n = int(stack.pop()), int(stack.pop())
    if k >= 0 and n >= 0 and k <= n:
      a = _factorial(n)
      b = _factorial(n-k)
      c = _factorial(k)
      stack.push(a/(c*b))
    else:
      stack.push(n)
      stack.push(k)
      raise BadArgumentValue(npi_comb)
  else:
    raise TooFewArguments(npi_comb)

  return stack


#
# Arrangement A(Y,X).
#
@func_name("arrang")
def npi_arrang(stack):
  """Arrangement (Y,X)

  >>> from stack import Stack
  >>> npi_arrang(Stack([10, 4]))
  [5040.0]
  >>> npi_arrang(Stack([4, 10]))
  Traceback (most recent call last):
  ...
  npi_errors.BadArgumentValue: arrang Error: Bad Argument Value
  >>> npi_arrang(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: arrang Error: Too Few Arguments
  """
  
  if len(stack) >= 2:
    k, n = int(stack.pop()), int(stack.pop())
    if k >= 0 and n >= 0 and k <= n:
      a = _factorial(n)
      b = _factorial(n-k)
      stack.push(a/b)
    else:
      stack.push(n)
      stack.push(k)
      raise BadArgumentValue(npi_arrang)
  else:
    raise TooFewArguments(npi_arrang)

  return stack


#
# Self-test.
#
def _test():
  import doctest
  print("Testing 'Proba' commands...", end="")
  (failure_count, test_count) = doctest.testmod()
  if not failure_count: print("Okey")
  else: print("Ko!")


#
# External entry point.
#
if __name__ == "__main__":
  _test()
