#
# NPI - Calculatrice en Notation Polonaise Inverse. 
# Copyright (C) 2005-2011 MiKael NAVARRO
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""NPI - Reverse Polish Notation Calculator. 
Copyright (C) 2005-2011 MiKael NAVARRO

Real operations.
"""

# Include directives
from npi_errors import TooFewArguments  # NPI errors
from npi_utils import func_name  # decorator: set func_name
from math import ceil, floor, modf, fabs, fmod
from math import pow


#
# Ceil.
#
@func_name("ceil")
def npi_ceil(stack):
  """Fisrt integer >= X

  >>> from stack import Stack
  >>> npi_ceil(Stack([3.14159265359]))
  [4]
  >>> npi_ceil(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: ceil Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(ceil(stack.pop()))
  else:
    raise TooFewArguments(npi_ceil)
  
  return stack


#
# Floor.
#
@func_name("floor")
def npi_floor(stack):
  """Last interger <= X

  >>> from stack import Stack
  >>> npi_floor(Stack([3.14159265359]))
  [3]
  >>> npi_floor(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: floor Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(floor(stack.pop()))
  else:
    raise TooFewArguments(npi_floor)
  
  return stack


#
# Truncate.
#
@func_name("trunc")
def npi_trunc(stack):
  """Truncate Y (at X)

  >>> from stack import Stack
  >>> npi_trunc(Stack([3.14159265359, 4]))
  [3.1415]
  >>> npi_trunc(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: trunc Error: Too Few Arguments
  """
  
  if len(stack) >= 2:
    n, y = int(stack.pop()), stack.pop()
    stack.push(modf(y * pow(10, n))[1] / pow(10, n))
  else:
    raise TooFewArguments(npi_trunc)
  
  return stack


#
# Round.
#
@func_name("rnd")
def npi_rnd(stack):
  """Round Y (at X)

  >>> from stack import Stack
  >>> npi_rnd(Stack([3.14159265359, 4]))
  [3.1416]
  >>> npi_rnd(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: rnd Error: Too Few Arguments
  """

  if len(stack) >= 2:
    n = stack.pop()
    fp, ip = modf(stack.pop() * pow(10, n))
    if fp >= 0.5:
      stack.push((ip+1) / pow(10, n))
    else:
      stack.push(ip / pow(10, n))
  else:
    raise TooFewArguments(npi_rnd)
  
  return stack


#
# Fabs.
#
@func_name("fabs")
def npi_fabs(stack):
  """Absolute value

  >>> from stack import Stack
  >>> npi_fabs(Stack([-3.7]))
  [3.7]
  >>> npi_fabs(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: fabs Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(fabs(stack.pop()))
  else:
    raise TooFewArguments(npi_fabs)

  return stack


#
# Integer part (prefix).
#
@func_name("ip")
def npi_ip(stack):
  """Integer part X

  >>> from stack import Stack
  >>> npi_ip(Stack([3.7]))
  [3.0]
  >>> npi_ip(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: ip Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(modf(stack.pop())[1])
  else:
    raise TooFewArguments(npi_ip)
  
  return stack


#
# Float part (postfix).
#
@func_name("fp")
def npi_fp(stack):
  """Float part X

  >>> from stack import Stack
  >>> npi_fp(Stack([3.7]))  # doctest:+ELLIPSIS
  [0.7...]
  >>> npi_fp(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: fp Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(modf(stack.pop())[0])
  else:
    raise TooFewArguments(npi_fp)
  
  return stack


#
# Return sign of X.
#
@func_name("sign")
def npi_sign(stack):
  """Sign of X

  >>> from stack import Stack
  >>> npi_sign(Stack([-3]))
  [-1.0]
  >>> npi_sign(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: sign Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    x = stack.pop()
    if x < 0:
      stack.push(-1.0)
    elif x > 0:
      stack.push(1.0)
    else:
      stack.push(0.0)
  else:
    raise TooFewArguments(npi_sign)

  return stack


#
# Mod.
#
@func_name("mod")
def npi_mod(stack):
  """Y modulo X

  >>> from stack import Stack
  >>> npi_mod(Stack([3, 2]))
  [1]
  >>> npi_mod(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: mod Error: Too Few Arguments
  """
  
  if len(stack) >= 2:
    op1, op2 = int(stack.pop()), int(stack.pop())
    stack.push(op2 % op1)
  else:
    raise TooFewArguments(npi_mod)
  
  return stack


#
# Fmod.
#
@func_name("fmod")
def npi_fmod(stack):
  """Remainder of X/Y

  >>> from stack import Stack
  >>> npi_fmod(Stack([3, 2]))
  [2.0]
  >>> npi_fmod(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: fmod Error: Too Few Arguments
  """
  
  if len(stack) >= 2:
    stack.push(fmod(stack.pop(), stack.pop()))
  else:
    raise TooFewArguments(npi_fmod)
  
  return stack


#
# Min.
#
@func_name("min")
def npi_min(stack):
  """Min X,Y

  >>> from stack import Stack
  >>> npi_min(Stack([3, 2]))
  [2]
  >>> npi_min(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: min Error: Too Few Arguments
  """
  
  if len(stack) >= 2:
    a, b = stack.pop(), stack.pop()
    stack.push(a < b and a or b)
  else:
    raise TooFewArguments(npi_min)
  
  return stack


#
# Max.
#
@func_name("max")
def npi_max(stack):
  """Max X,Y

  >>> from stack import Stack
  >>> npi_max(Stack([3, 2]))
  [3]
  >>> npi_max(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: max Error: Too Few Arguments
  """
  
  if len(stack) >= 2:
    a, b = stack.pop(), stack.pop()
    stack.push(a > b and a or b)
  else:
    raise TooFewArguments(npi_max)
  
  return stack


#
# Self-test.
#
def _test():
  import doctest
  print("Testing 'Real' commands...", end="")
  (failure_count, test_count) = doctest.testmod()
  if not failure_count: print("Okey")
  else: print("Ko!")


#
# External entry point.
#
if __name__ == "__main__":
  _test()


