#
# NPI - Calculatrice en Notation Polonaise Inverse. 
# Copyright (C) 2005-2011 MiKael NAVARRO
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""NPI - Reverse Polish Notation Calculator. 
Copyright (C) 2005-2011 MiKael NAVARRO

Stack manipulations.
"""

# Include directives
from npi_errors import TooFewArguments  # NPI errors
from npi_utils import func_name  # decorator: set func_name


#
# Delete.
#
@func_name("del")
def npi_del(stack):
  """Clear stack

  >>> from stack import Stack
  >>> npi_del(Stack([3, 2]))  # 3 2 del
  []
  """

  stack[:] = []

  return stack


#
# Drop.
#
@func_name("drop")
def npi_drop(stack):
  """Erase X

  >>> from stack import Stack
  >>> npi_drop(Stack([3, 2, 1]))  # 3 2 1 drop
  [3, 2]
  >>> npi_drop(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: drop Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.pop()
  else:
    raise TooFewArguments(npi_drop)

  return stack

@func_name("drop2")
def npi_drop2(stack):
  """Erase X, Y

  >>> from stack import Stack
  >>> npi_drop2(Stack([3, 2, 1]))  # 3 2 1 drop2
  [3]
  >>> npi_drop2(Stack([1]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: drop2 Error: Too Few Arguments
  """

  if len(stack) >= 2:
    for i in range(2):
      npi_drop(stack)
  else:
    raise TooFewArguments(npi_drop2)

  return stack

@func_name("dropn")
def npi_dropn(stack):
  """Erase X-N

  >>> from stack import Stack
  >>> npi_dropn(Stack([5, 4, 3, 2, 1, 3]))  # 5 4 3 2 1 3 dropn
  [5, 4]
  >>> npi_dropn(Stack([1]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: dropn Error: Too Few Arguments
  """

  if len(stack) >= 1:
    n = int(stack.pop())
    if len(stack) >= n:
      for i in range(n):
        npi_drop(stack)
    else:
      raise TooFewArguments(npi_dropn)
  else:
    raise TooFewArguments(npi_dropn)

  return stack


#
# Duplicate.
#
@func_name("dup")
def npi_dup(stack):
  """Duplicate X

  >>> from stack import Stack
  >>> npi_dup(Stack([3, 2]))  # 3 2 dup
  [3, 2, 2]
  >>> npi_dup(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: dup Error: Too Few Arguments
  """

  if len(stack) >= 1:
    stack.push(stack[len(stack)-1])
  else:
    raise TooFewArguments(npi_dup)

  return stack

@func_name("dup2")
def npi_dup2(stack):
  """Duplicate X,Y

  >>> from stack import Stack
  >>> npi_dup2(Stack([3, 2]))  # 3 2 dup2
  [3, 2, 3, 2]
  >>> npi_dup2(Stack([1]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: dup2 Error: Too Few Arguments
  """

  if len(stack) >= 2:
    stack.push(2)
    npi_dupn(stack)
  else:
    raise TooFewArguments(npi_dup2)

  return stack

@func_name("dupn")
def npi_dupn(stack):
  """Duplicate X-N

  >>> from stack import Stack
  >>> npi_dupn(Stack([4, 3, 2, 1, 3]))  # 4 3 2 1 3 dupn
  [4, 3, 2, 1, 3, 2, 1]
  >>> npi_dupn(Stack([2, 1, 3]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: dupn Error: Too Few Arguments
  """

  if len(stack) >= 1:
    n = int(stack.pop())
    if len(stack) >= n:
      for i in range(n):
        stack.push(stack[len(stack)-n])
    else:
      raise TooFewArguments(npi_dupn)
  else:
    raise TooFewArguments(npi_dupn)

  return stack


#
# Swap (X<->Y).
#
@func_name("swap")
def npi_swap(stack):
  """Swap X and Y

  >>> from stack import Stack
  >>> npi_swap(Stack([4, 3, 2]))  # 3 2 swap
  [4, 2, 3]
  >>> npi_swap(Stack([1]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: swap Error: Too Few Arguments
  """

  if len(stack) >= 2:
    op1, op2 = stack.pop(), stack.pop()
    stack.push(op1)
    stack.push(op2)
  else:
    raise TooFewArguments(npi_swap)

  return stack


#
# Depth of stack.
#
@func_name("depth")
def npi_depth(stack):
  """Depth of stack

  >>> from stack import Stack
  >>> npi_depth(Stack([3, 2]))  # 3 2 depth
  [3, 2, 2]
  """

  l = len(stack)
  stack.push(l)
  
  return stack


#
# Copy Y.
#
@func_name("over")
def npi_over(stack):
  """Duplicate Y

  >>> from stack import Stack
  >>> npi_over(Stack([4, 3, 2]))  # 4 3 2 over
  [4, 3, 2, 3]
  >>> npi_over(Stack([1]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: over Error: Too Few Arguments
  """

  if len(stack) >= 2:
    stack.push(stack[len(stack)-2])
  else:
    raise TooFewArguments(npi_over)

  return stack


#
# Copy N.
#
@func_name("pick")
def npi_pick(stack):
  """Duplicate N

  >>> from stack import Stack
  >>> npi_pick(Stack([4, 3, 2, 1, 3]))  # 4 3 2 1 3 pick
  [4, 3, 2, 1, 3]
  >>> npi_pick(Stack([2, 1, 3]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: pick Error: Too Few Arguments
  """

  if len(stack) >= 1:
    n = int(stack.pop())
    if len(stack) >= n:
      stack.push(stack[len(stack)-n])
    else:
      raise TooFewArguments(npi_pick)
  else:
    raise TooFewArguments(npi_pick)

  return stack


#
# Move N on top.
#
@func_name("roll")
def npi_roll(stack):
  """Move N on top of stack

  >>> from stack import Stack
  >>> npi_roll(Stack([4, 3, 2, 1, 3]))  # 4 3 2 1 3 roll
  [4, 2, 1, 3]
  >>> npi_roll(Stack([2, 1, 3]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: roll Error: Too Few Arguments
  """

  if len(stack) >= 1:
    n = int(stack.pop())
    if len(stack) >= n:
      stack.push(stack[len(stack)-n])
      del stack[len(stack)-(n+1)]
    else:
      raise TooFewArguments(npi_roll)
  else:
    raise TooFewArguments(npi_roll)

  return stack


#
# Move top on level N.
#
@func_name("rolld")
def npi_rolld(stack):
  """Move top on level N

  >>> from stack import Stack
  >>> npi_rolld(Stack([4, 3, 2, 1, 3]))  # 4 3 2 1 3 rolld
  [4, 1, 3, 2]
  >>> npi_rolld(Stack([2, 1, 3]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: rolld Error: Too Few Arguments
  """

  if len(stack) >= 1:
    n = int(stack.pop())
    if len(stack) >= n:
      e = stack.pop()
      stack.insert(len(stack)-(n-1), e)
    else:
      raise TooFewArguments(npi_rolld)
  else:
    raise TooFewArguments(npi_rolld)

  return stack


#
# Circular permutation (= 3 ROLL).
#
@func_name("rot")
def npi_rot(stack):
  """Circular permutation

  >>> from stack import Stack
  >>> npi_rot(Stack([4, 3, 2, 1]))  # 4 3 2 1 rot
  [4, 2, 1, 3]
  >>> npi_rot(Stack([2, 1]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: rot Error: Too Few Arguments
  """

  if len(stack) >= 3:
    stack.push(3)
    npi_roll(stack)
  else:
    raise TooFewArguments(npi_rot)

  return stack


#
# Invert order of levels 1-N.
#
@func_name("invst")
def npi_invst(stack):
  """Invert X-N levels

  >>> from stack import Stack
  >>> npi_invst(Stack([5, 4, 3, 2, 1, 3]))  # 5 4 3 2 1 3 invst
  [5, 4, 1, 2, 3]
  >>> npi_invst(Stack([2, 1, 3]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: invst Error: Too Few Arguments
  """

  if len(stack) >= 1:
    n = int(stack.pop())

    if len(stack) >= n:
      for i in range(1, n+1):
        stack.push(i)
        npi_roll(stack);
    else:
      raise TooFewArguments(npi_invst)
  else:
    raise TooFewArguments(npi_invst)

  return stack


#
# Reverse the stack.
#
@func_name("reverse")
def npi_reverse(stack):
  """Reverse the stack

  >>> from stack import Stack
  >>> npi_reverse(Stack([4, 3, 2, 1]))  # 4 3 2 1 reverse
  [1, 2, 3, 4]
  >>> npi_reverse(Stack([]))
  []
  """

  stack.reverse()

  return stack


#
# Self-test.
#
def _test():
  import doctest
  print("Testing 'Stack' commands...", end="")
  (failure_count, test_count) = doctest.testmod()
  if not failure_count: print("Okey")
  else: print("Ko!")


#
# External entry point.
#
if __name__ == "__main__":
  _test()
