# test_hpstats.py -- unittests for hpstats module

# Copyright (c) 2005 Floris Bruynooghe

# All rights reserved.

# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation files
# (the "Software"), to deal in the Software without restriction,
# including without limitation the rights to use, copy, modify, merge,
# publish, distribute, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, provided
# that the above copyright notice(s) and this permission notice appear
# in all copies of the Software and that both the above copyright
# notice(s) and this permission notice appear in supporting
# documentation.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE
# COPYRIGHT HOLDER OR HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR
# ANY CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
# WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
# ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
# OF THIS SOFTWARE.

# Except as contained in this notice, the name of a copyright holder
# shall not be used in advertising or otherwise to promote the sale,
# use or other dealings in this Software without prior written
# authorization of the copyright holder.


import unittest
import copy
import tempfile
import sys
import pickle
from test import test_support

import hpstats
import hprofile
import hstats


class BackEndTesting(unittest.TestCase):
    def setUp(self):
        def f():
            pass
        p = hprofile.Profile()
        p.runctx('f()', globals(), locals())
        p.dump_stats(test_support.TESTFN)

    def tearDown(self):
        test_support.unlink(test_support.TESTFN)

    def test_converted_to_seconds(self):
        # Data converted to seconds?
        hs = hstats.Stats(test_support.TESTFN)
        ps = hpstats._BackEnd(test_support.TESTFN)
        htime = hs._data.values()[0][hstats._SDATA_TIME]
        del ps._data[('<string>', 1, '?')]
        ptime = ps._data.values()[0][hstats._SDATA_TIME]
        self.failUnlessEqual(htime*0.000001, ptime)

    def test_string_stays(self):
        # The '<string>' function does not get stripped?
        s = hpstats._BackEnd(test_support.TESTFN)
        self.failUnless(('<string>', 1, '?') in s._data.keys())

    def test_get_data_order(self):
        # The .get_data() method returns data in expected order?
        order = ['call', 'time', 'avgtime',
                 'cumtime', 'avgcumtime', 'name', 'parents']
        s = hpstats._BackEnd(test_support.TESTFN)
        data, dd = s.get_data(extra=['parents'])
        self.failUnlessEqual(dd, order)


class BackEndBiasTesting(unittest.TestCase):
    def setUp(self):
        self.bias = 0.5
        def f():
            pass
        p = hprofile.Profile(bias=self.bias)
        p.runctx('f()', globals(), locals())
        p.dump_stats(test_support.TESTFN)

    def tearDown(self):
        test_support.unlink(test_support.TESTFN)
        
    def test_bias_used(self):
        # Bias is used?
        hs = hstats.Stats(test_support.TESTFN)
        ps = hpstats._BackEnd(test_support.TESTFN)
        htime = hs._data.values()[-1][hstats._SDATA_TIME]
        ptime = ps._data.values()[-1][hstats._SDATA_TIME]
        self.failUnlessAlmostEqual(htime*0.000001-self.bias, ptime, 3)


class StatsTesting(unittest.TestCase):
    def setUp(self):
        p = hprofile.Profile()
        p.run('pass')
        p.dump_stats(test_support.TESTFN)
        self.s = hpstats.Stats(test_support.TESTFN)

    def tearDown(self):
        test_support.unlink(test_support.TESTFN)

    def test_constructor_data_filled(self):
        # self._data gets created?
        s = hpstats.Stats(test_support.TESTFN)
        self.failUnless(isinstance(s._data, list))

    def test_constructor_order_set(self):
        # self._data_order is an empty list after instantiation.
        s = hpstats.Stats(test_support.TESTFN)
        self.failUnlessEqual(s._data_order, None)

    def test_constants_defined(self):
        # self._D_* constants are defined?
        s = hpstats.Stats(test_support.TESTFN)
        self.failUnless(hasattr(s, "_D_CALLS"))
        self.failUnless(hasattr(s, "_D_CALLS_P"))
        self.failUnless(hasattr(s, "_D_TOTTIME"))
        self.failUnless(hasattr(s, "_D_PERCALL_T"))
        self.failUnless(hasattr(s, "_D_CUMTIME"))
        self.failUnless(hasattr(s, "_D_PERCALL_C"))
        self.failUnless(hasattr(s, "_D_NAME_MODULE"))
        self.failUnless(hasattr(s, "_D_NAME_LINE"))
        self.failUnless(hasattr(s, "_D_NAME_FUNC"))
        self.failUnless(hasattr(s, "_D_NAME_LINE_STR"))
        self.failUnless(hasattr(s, "_D_PARENTS"))

    def test_line_as_string(self):
        # The line number is added as a string at the end of the data?
        s = hpstats.Stats(test_support.TESTFN)
        lines = get_column(s._data, s._D_NAME_LINE)
        lines_str = get_column(s._data, s._D_NAME_LINE_STR)
        for i, line in enumerate(lines[:]):
            lines[i] = str(line)
        self.failUnlessEqual(lines, lines_str)

    def test_parents_added(self):
        # The list of parents in in the data?
        parents = get_column(self.s._data, self.s._D_PARENTS)
        for row in parents:
            self.failUnless(isinstance(row, dict))

    def test_load_file_hotshot(self):
        # Stats._load_file() works with hotshot file?
        start_data = copy.copy(self.s._data)
        self.s._load_file(test_support.TESTFN)
        self.failUnlessEqual(start_data, self.s._data[-1:])

    def test_load_file_pickle(self):
        # Stats._load_file() works with pickle data?
        extra_data = [["dummy row"], ["dummy row 2"]]
        fileobject = file(test_support.TESTFN, "wb")
        pickle.dump(extra_data, fileobject, pickle.HIGHEST_PROTOCOL)
        fileobject.close()
        self.s._load_file(test_support.TESTFN)
        self.failUnlessEqual(extra_data, self.s._data[-2:])

    def test_load_file_invalid(self):
        # Stats._load_file() handles invalid filename?
        self.failUnlessRaises(ValueError, self.s._load_file, 12)

    def test_load_pickled(self):
        # Stats._load_pickled() adds data to self._data?
        extra_data = [["dummy row"], ["dummy row 2"]]
        fileobject = file(test_support.TESTFN, "wb")
        pickle.dump(extra_data, fileobject, pickle.HIGHEST_PROTOCOL)
        fileobject.close()
        self.s._load_pickled(test_support.TESTFN)
        self.failUnlessEqual(extra_data, self.s._data[-2:])

    def test_strip_dirs_returns_self(self):
        s = hpstats.Stats(test_support.TESTFN)
        self.failUnless(s is s.strip_dirs())

    def test_strip_dirs_extra_data(self):
        # .strip_dirs() leaves the extra data at the end of each row intact?
        s = hpstats.Stats(test_support.TESTFN)
        data_before = get_column(s._data, s._D_NAME_LINE_STR)
        s.strip_dirs()
        data_after = get_column(s._data, s._D_NAME_LINE_STR)
        data_before.sort()
        data_after.sort()
        self.failUnlessEqual(data_before, data_after)

    def test_add_returns_self(self):
        self.failUnless(self.s is self.s.add(test_support.TESTFN))

    def test_add_hotshot(self):
        # Stats._load_file() works with hotshot file?
        start_data = copy.copy(self.s._data)
        self.s.add(test_support.TESTFN)
        self.failUnlessEqual(start_data, self.s._data[-1:])

    def test_add_pickle(self):
        # Stats._load_file() works with pickle data?
        extra_data = [[1, 1, 4, 4, 4, 4, "module", 2, "function", '2', {}]]
        fileobject = file(test_support.TESTFN, "wb")
        pickle.dump(extra_data, fileobject, pickle.HIGHEST_PROTOCOL)
        fileobject.close()
        self.s.add(test_support.TESTFN)
        self.failUnlessEqual(extra_data, self.s._data[-1:])

    def test_add_multiple(self):
        # Stats.add() takes multiple files?
        start_data = copy.copy(self.s._data) # Shallow copy!
        extra_data = [[1, 1, 4, 4, 4, 4, "module", 2, "function", '2', {}]]
        #fileobject = tempfile.NamedTemporaryFile()
        fd, filename = tempfile.mkstemp()
        fileobject = file(filename, "wb")
        pickle.dump(extra_data, fileobject, pickle.HIGHEST_PROTOCOL)
        fileobject.close()
        self.s.add(test_support.TESTFN, filename)
        test_support.unlink(filename)
        expected = [start_data[0], extra_data[0]]
        expected.sort()
        self.s._data.sort()
        self.failUnlessEqual(expected, self.s._data)

    def test_add_invalid(self):
        # Stats._load_file() handles invalid filename?
        self.failUnlessRaises(ValueError, self.s.add, 12)

    def test_sort_stats_returns_self(self):
        # Also tests if empty sort works.
        s = hpstats.Stats(test_support.TESTFN)
        self.failUnless(s is s.sort_stats())

    def test_sort_stats_sets_data_order(self):
        # Stats.sort_stats() do set the data order correctly?
        self.s.sort_stats('pcalls', 'cumulative', 'line')
        self.failUnlessEqual(self.s._data_order,
                             ['pcalls', 'cumulative', 'line'])

    def test_reverse_order_returns_self(self):
        # Stats.reverse_order() returns self?
        s = hpstats.Stats(test_support.TESTFN)
        self.failUnless(s is s.sort_stats())

    def test_print_stats_returns_self(self):
        # Stats.print_stats() returns itself?
        self.failUnless(self.s is self.s.sort_stats())

    def test_print_preamble_data_order(self):
        # Stats._print_preamble prints out the data order correctly?
        bench = "   Ordered by: internal time, call count"
        self.s.sort_stats('time', 'calls')
        output = fetch_stdout(self.s._print_preamble)
        check = output.splitlines()[2]
        self.failUnlessEqual(bench, check)

    def test_print_preamble_data_order_unordered(self):
        # Stats._print_preamble prints out the data order when unordered?
        bench = "   Random listing order was used"
        output = fetch_stdout(self.s._print_preamble)
        check = output.splitlines()[2]
        self.failUnlessEqual(bench, check)

class FakeDataStubTesting(unittest.TestCase):
    def setUp(self):
        p = hprofile.Profile()
        p.run('pass')
        p.dump_stats(test_support.TESTFN)
        self.stub = hpstats.Stats(test_support.TESTFN)
        test_support.unlink(test_support.TESTFN)
        self.stub._data = [[2, 2, 10, 5, 10, 5, 'foo/mod', 5, 'func', '5',
                            {('<string>', 1, '?'): 2}],
                           [1, 1, 10, 10, 40, 40, '<string>', 1, '?', '1',
                            {}],
                           [2, 2, 8, 4, 8, 4, 'foo/mod', 10, 'func', '10',
                            {('foo/mod', 5, 'func'): 2}],
                           [2, 2, 10, 5, 10, 5, 'bar/mod', 5, 'func', '5',
                            {('foo/mod', 5, 'func'): 1,
                             ('bar/mod', 5, 'func'): 1}]]

    def test_dump_stats(self):
        # Check on the file created by dump_stats
        filename = test_support.TESTFN
        self.stub.dump_stats(filename)
        file_object = file(filename)
        loaded_data = pickle.load(file_object)
        file_object.close()
        test_support.unlink(filename)
        self.failUnlessEqual(self.stub._data, loaded_data)

    def test_dump_stats_extra_data(self):
        # Stats.dump_stats() does not save extra data appended to the rows?
        data_copy = copy.deepcopy(self.stub._data)
        for row_index in range(len(self.stub._data)):
            self.stub._data[row_index].extend(["extra", "cruft", 1])

        filename = test_support.TESTFN
        self.stub.dump_stats(filename)
        file_object = file(filename)
        loaded_data = pickle.load(file_object)
        file_object.close()
        test_support.unlink(filename)
        self.failUnlessEqual(data_copy, loaded_data)

    def test_strip_dirs(self):
        # .strip_dirs() behaves correctly?
        data = [[1, 1, 10, 10, 40, 40, '<string>', 1, '?', '1',
                 {}],
                [4, 4, 20, 5, 20, 5, 'mod', 5, 'func', '5',
                 {('<string>', 1, '?'): 2, ('mod', 5, 'func'): 2}],
                [2, 2, 8, 4, 8, 4, 'mod', 10, 'func', '10',
                 {('mod', 5, 'func'): 2}]]
        data.sort()
        self.stub.strip_dirs()
        self.stub._data.sort()
        self.failUnlessEqual(data, self.stub._data)

    def test_sort_stats_calls(self):
        # .sort_stats('calls') works?
        self.stub.sort_stats('calls')
        calls = get_column(self.stub._data, self.stub._D_CALLS)
        sorted_calls = copy.copy(calls)
        sorted_calls.sort()
        sorted_calls.reverse()
        self.failUnlessEqual(calls, sorted_calls)

    def test_sort_stats_time(self):
        # .sort_stats('time') works?
        self.stub.sort_stats('time')
        times = get_column(self.stub._data, self.stub._D_TOTTIME)
        sorted_times = copy.copy(times)
        sorted_times.sort()
        sorted_times.reverse()
        self.failUnlessEqual(times, sorted_times)

    def test_sort_stats_name(self):
        # .sort_stats('name') works?
        self.stub.sort_stats('name')
        names = get_column(self.stub._data, self.stub._D_NAME_FUNC)
        sorted_names = copy.copy(names)
        sorted_names.sort()
        self.failUnlessEqual(names, sorted_names)

    def test_sort_stats_stdname(self):
        # Stats.sort_stats(-1) (also 'stdname') works?
        known = copy.copy(self.stub._data)
        self.stub.sort_stats('stdname')
        self.failUnlessEqual(self.stub._data[0], known[1])
        self.failUnlessEqual(self.stub._data[1], known[3])
        self.failUnlessEqual(self.stub._data[2], known[2])
        self.failUnlessEqual(self.stub._data[3], known[0])

    def test_reverse_order(self):
        # Stats.reverse_order() really reverses?
        column = self.stub._D_PERCALL_C
        order = get_column(self.stub._data, column)
        self.stub.reverse_order()
        reversed = get_column(self.stub._data, column)
        order.reverse()
        self.failUnlessEqual(order, reversed)

    def test_print_preamble(self):
        # Preamle gets printed correctly?
        bench = """\
        7 function calls in 38.000 CPU seconds

   Random listing order was used

   ncalls  tottime  percall  cumtime  percall filename:lineno(function)
"""
        output = fetch_stdout(self.stub._print_preamble)
        self.failUnlessEqual(bench, output)

    def test_restrict_data_int(self):
        # Stats._restrict_data() works when used with an integer?
        data = self.stub._restrict_data(2)
        self.failUnlessEqual(2, len(data))

    def test_restrict_data_float(self):
        # Stats._restrict_data() works when used with a float?
        data = self.stub._restrict_data(.25)
        self.failUnlessEqual(1, len(data))

    def test_restrict_data_re(self):
        # Stats._restrict_data() works when used with a regular expression?
        data = self.stub._restrict_data(r'<string>:')
        string = get_column(data, self.stub._D_NAME_MODULE)
        self.failUnlessEqual([['<string>']], [string])

    def test_restrict_data_multiple(self):
        # Stats._restrict_data() works when using multiple restrictions?
        data = self.stub._restrict_data(2, 0.5)
        self.failUnlessEqual(1, len(data))

    def test_print_data(self):
        # Stats._print_data() prints out the stats correctly?
        bench = """\
        2   10.000    5.000   10.000    5.000 foo/mod:5(func)
        1   10.000   10.000   40.000   40.000 <string>:1(?)
        2    8.000    4.000    8.000    4.000 foo/mod:10(func)
        2   10.000    5.000   10.000    5.000 bar/mod:5(func)
"""
        output = fetch_stdout(self.stub._print_data, self.stub._data)
        self.failUnlessEqual(bench, output)

    def test_print_stats_simple(self):
        # Stats.print_stats() works normally?
        bench = """\
        7 function calls in 38.000 CPU seconds

   Random listing order was used

   ncalls  tottime  percall  cumtime  percall filename:lineno(function)
        2   10.000    5.000   10.000    5.000 foo/mod:5(func)
        1   10.000   10.000   40.000   40.000 <string>:1(?)
        2    8.000    4.000    8.000    4.000 foo/mod:10(func)
        2   10.000    5.000   10.000    5.000 bar/mod:5(func)


"""
        output = fetch_stdout(self.stub.print_stats)
        self.failUnlessEqual(bench, output)

    def test_print_stats_restriction(self):
        # Stats.print_stats() works when using restrictions?
        bench = """\
        7 function calls in 38.000 CPU seconds

   Random listing order was used

   ncalls  tottime  percall  cumtime  percall filename:lineno(function)
        2   10.000    5.000   10.000    5.000 foo/mod:5(func)
        1   10.000   10.000   40.000   40.000 <string>:1(?)


"""
        output = fetch_stdout(self.stub.print_stats, 2)
        self.failUnlessEqual(bench, output)

    def test_print_callers(self):
        # Stats.print_callers() works as expected?
        bench = """\
   Random listing order was used

Function                         was called by...
foo/mod:5(func)                   <string>:1(?)(2)    40.000
<string>:1(?)                     --
foo/mod:10(func)                  foo/mod:5(func)(2)    10.000
bar/mod:5(func)                   foo/mod:5(func)(1)    10.000
                                  bar/mod:5(func)(1)    10.000


"""
        output = fetch_stdout(self.stub.print_callers)
        self.failUnlessEqual(bench, output)

    def test_print_callers_restricted(self):
        # Stats.print_callers() works with restriction?
        bench = """\
   Random listing order was used

Function                         was called by...
foo/mod:5(func)                   <string>:1(?)(2)    40.000


"""
        output = fetch_stdout(self.stub.print_callers, 1)
        self.failUnlessEqual(bench, output)

    def test_print_preamble_callers(self):
        # Preamle of Stats.print_callers() gets printed correctly?
        bench = """\
   Random listing order was used

Function                         was called by...
"""
        output = fetch_stdout(self.stub._print_preamble_callers)
        self.failUnlessEqual(bench, output)

    def test_row_from_tuple(self):
        # Stats._row_from_tuple() finds indeed the row?
        row = self.stub._data[0]
        tuple_name = ('foo/mod', 5, 'func')
        result = self.stub._row_from_tuple(tuple_name)
        self.failUnlessEqual(row, result)

    def test_print_data_callers(self):
        # Data of Stats.print_callers() gets printed correctly?
        bench = """\
foo/mod:5(func)                   <string>:1(?)(2)    40.000
<string>:1(?)                     --
foo/mod:10(func)                  foo/mod:5(func)(2)    10.000
bar/mod:5(func)                   foo/mod:5(func)(1)    10.000
                                  bar/mod:5(func)(1)    10.000
"""
        output = fetch_stdout(self.stub._print_data_callers, self.stub._data)
        self.failUnlessEqual(bench, output)

    def test_print_callees(self):
        # Stats.print_callees() get printed correctly?
        bench = """\
   Random listing order was used

Function                         called...
foo/mod:5(func)                   foo/mod:10(func)(2)    8.000
                                  bar/mod:5(func)(1)    10.000
<string>:1(?)                     foo/mod:5(func)(2)    10.000
foo/mod:10(func)                  --
bar/mod:5(func)                   bar/mod:5(func)(1)    10.000


"""
        output = fetch_stdout(self.stub.print_callees)
        self.failUnlessEqual(bench, output)

    def test_print_preamble_callees(self):
        # Preamle of Stats.print_callees() gets printed correctly?
        bench = """\
   Random listing order was used

Function                         called...
"""
        output = fetch_stdout(self.stub._print_preamble_callees)
        self.failUnlessEqual(bench, output)

    def test_print_data_callees(self):
        # Data of Stats.print_callers() gets printed correctly?
        bench = """\
foo/mod:5(func)                   foo/mod:10(func)(2)    8.000
                                  bar/mod:5(func)(1)    10.000
<string>:1(?)                     foo/mod:5(func)(2)    10.000
foo/mod:10(func)                  --
bar/mod:5(func)                   bar/mod:5(func)(1)    10.000
"""
        self.stub._collect_callees()
        output = fetch_stdout(self.stub._print_data_callees, self.stub._data)
        self.failUnlessEqual(bench, output)

    def test_collect_callees(self):
        # Callees are collected properly?
        callees = [{('foo/mod', 10, 'func'): 2, ('bar/mod', 5, 'func'): 1},
                   {('foo/mod', 5, 'func'): 2},
                   {},
                   {('bar/mod', 5, 'func'): 1}]
        data = copy.deepcopy(self.stub._data)
        for i in range(len(data)):
            data[i].append(callees[i])
        len_before = len(self.stub._data[0])
        self.stub._collect_callees()
        len_after = len(self.stub._data[0])
        self.failUnlessEqual(data, self.stub._data)
        self.failUnlessEqual(len_before+1, len_after)


class EmptyStubTesting(unittest.TestCase):
    """Tests of methods that don't need object data."""
    def setUp(self):
        class Stub(hpstats.Stats):
            def __init__(self):
                pass
        self.stub = Stub()
        self.make_order = self.stub._make_order
        self.expand = self.stub._expand_abbrev

    def test_merge_duplicate_data(self):
        # Duplicate data gets merged properly by ._merge_duplicate_data?
        self.stub._data = [[1, 1, 2, 2, 2, 2, '<string>', 1, '?', '1',
                            {('mod', 5, 'func'): 3}],
                           [1, 1, 4, 4, 4, 4, '<string>', 1, '?', '1',
                            {('mod', 5, 'func'): 2}]]
        expected_data = [[2, 2, 6, 3, 6, 3, '<string>', 1, '?', '1',
                          {('mod', 5, 'func'): 5}]]
        self.stub._merge_duplicate_data()
        self.failUnlessEqual(expected_data, self.stub._data)

    def test_make_order_simple(self):
        # Test one normal descending and one normal ascending
        order = self.make_order(['time', 'name'])
        expected = [(self.stub._D_TOTTIME, 'd'),
                    (self.stub._D_NAME_FUNC, 'a')]
        self.failUnlessEqual(order, expected)

    def test_make_order_nfl(self):
        # 'nfl' gives the right list?
        order = self.make_order(['nfl'])
        expected = [(self.stub._D_NAME_FUNC, 'a'),
                    (self.stub._D_NAME_MODULE, 'a'),
                    (self.stub._D_NAME_LINE, 'a')]
        self.failUnlessEqual(order, expected)

    def test_make_order_stdname(self):
        # 'stdname' give the right list?
        order = self.make_order(['stdname'])
        expected = [(self.stub._D_NAME_MODULE, 'a'),
                    (self.stub._D_NAME_LINE_STR, 'a'),
                    (self.stub._D_NAME_FUNC, 'a')]
        self.failUnlessEqual(order, expected)

    def test_expand_abbrev_simple(self):
        # All abbreviations are expanded?
        expanded = self.expand(['ca', 'cum', 'f', 'm', 'p',
                                'l', 'nam', 'nf', 's', 't'])
        fully = ['calls', 'cumulative', 'file', 'module',
                 'pcalls', 'line', 'name', 'nfl', 'stdname', 'time']
        self.failUnlessEqual(expanded, fully)

    def test_tuple_name_from_row(self):
        # The string of a name gets made correctly?
        row = [0, 0, 0, 0, 0, 0, 'foo/mod', 0, 'func', '0']
        expected = ('foo/mod', 0, 'func')
        result = self.stub._tuple_name_from_row(row)
        self.failUnlessEqual(expected, result)

    def test_rper_name_from_row(self):
        # The string of a name gets made correctly?
        row = [0, 0, 0, 0, 0, 0, 'foo/mod', 0, 'func', '0']
        expected = 'foo/mod:0(func)'
        result = self.stub._repr_name_from_row(row)
        self.failUnlessEqual(expected, result)

    def test_repr_name_from_tuple(self):
        # The string of a name gets made correctly?
        bench = "module:23(function)"
        tuple_input = ('module', 23, 'function')
        result = self.stub._repr_name_from_tuple(tuple_input)
        self.failUnlessEqual(bench, result)

    def test_get_order_text(self):
        # Stats._get_order_text() works?
        s = self.stub
        s._data_order = None
        result = s._get_order_text()
        expected = "Random listing order was used"
        self.failUnlessEqual(expected, result)
        s._data_order = ['line', 'nfl', 'file']
        result = s._get_order_text()
        expected = "Ordered by: line number, name/file/line, file name"
        self.failUnlessEqual(expected, result)


class HelperTesting(unittest.TestCase):
    def test_fetch_stdout(self):
        string = "hello"
        def f():
            print string,
        output = fetch_stdout(f)
        self.failUnlessEqual(string, output)

# Helper Functions

def get_column(data, column):
    """Return one column as a list."""
    result = list()
    for row in data:
        result.append(row[column])
    return result


def fetch_stdout(call, *args, **kwargs):
    """Execute `call' and return the resulting stdout as string.

    call - function or method call to execute.

    glob - dictionary of the global symbols as returned by `globals()'

    loc - dictionary of the local symbols as returned by `locals()'

    The output to stdout resulting from executing the call is caught
    and returned as a string.
    """
    outfile = tempfile.TemporaryFile()
    stdout_bak = sys.stdout
    sys.stdout = outfile
    call(*args, **kwargs)
    sys.stdout = stdout_bak
    outfile.flush()
    outfile.seek(0)
    return outfile.read()


def test_main():
    test_support.run_unittest(BackEndTesting,
                              BackEndBiasTesting,
                              StatsTesting,
                              FakeDataStubTesting,
                              EmptyStubTesting,
                              HelperTesting)


if __name__ == '__main__':
    test_main()
