/*
 * 	Random Access Machine.
 * 	Debugger stuff -- common functions.
 *
 * 	Copyright (C) 2002, 2003  Dmitry Rutsky	<rutsky@school.ioffe.rssi.ru>
 * 	
 * 	This program is free software; you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation; either version 2 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program; if not, write to the 
 * 	Free Software Foundation, Inc.,
 * 	59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "debugger.h"

/* ========	Constructors and destructors	======== */

static RAM_Watchpoint *ram_watchpoint_new ()
{
   RAM_Watchpoint *wp;

   wp = (RAM_Watchpoint *) calloc (1, sizeof (RAM_Watchpoint));

   if (! wp)
      err_fatal_perror ("calloc",
		      "cannot allocate memory for RAM_Watchpoint");

   return wp;
}

static inline void ram_watchpoint_struct_delete (RAM_Watchpoint *wp)
{
   free (wp);
}

static inline void ram_watchpoint_delete (RAM_Watchpoint *wp)
{
   if (wp -> expression)
      free (wp -> expression);
   
   ram_watchpoint_struct_delete (wp);
}

static void ram_watchpoint_delete_list (RAM_Watchpoint *wp)
{
   while (wp)
   {
      RAM_Watchpoint *next = (wp -> next);

      ram_watchpoint_delete (wp);

      wp = next;
   }
}

static RAM_Catchpoint *ram_catchpoint_new ()
{
   RAM_Catchpoint *cp;

   cp = (RAM_Catchpoint *) calloc (1, sizeof (RAM_Catchpoint));

   if (! cp)
      err_fatal_perror ("calloc",
		      "cannot allocate memory for RAM_Catchpoint...");

   return cp;
}

static inline void ram_catchpoint_struct_delete (RAM_Catchpoint *cp)
{
   free (cp);
}

static inline void ram_catchpoint_delete (RAM_Catchpoint *cp)
{
   if (cp -> expression)
      free (cp -> expression);
   
   ram_catchpoint_struct_delete (cp);
}

static void ram_catchpoint_delete_list (RAM_Catchpoint *cp)
{
   while (cp)
   {
      RAM_Catchpoint *next = (cp -> next);

      ram_catchpoint_delete (cp);

      cp = next;
   }
}

void ram_disable_watchpoint (RAM_Debug *rd, RAM_Watchpoint *wp)
{
   RAM_Watchpoint *w = (rd -> watchpoints);
   
   if (w == wp)
   {
      (rd -> watchpoints) = (wp -> next);
      ram_watchpoint_delete (wp);
      return;
   }
   
   while (w && ((w -> next) != wp))
      w = (w -> next);

   if (!w)
      err_programming
	    ("ram_disable_watchpoint:\ntried to disable invalid watchpoint");
   
   (w -> next) = (wp -> next);
   ram_watchpoint_delete (wp);
}

void ram_disable_catchpoint (RAM_Debug *rd, RAM_Catchpoint *cp)
{
   RAM_Catchpoint *c = (rd -> catchpoints);
   
   if (c == cp)
   {
      (rd -> catchpoints) = (cp -> next);
      ram_catchpoint_delete (cp);
      return;
   }
   
   while (c && ((c -> next) != cp))
      c = (c -> next);

   if (!c)
      err_programming
	    ("ram_disable_catchpoint:\ntried to disable invalid catchpoint");
   
   (c -> next) = (cp -> next);
   ram_catchpoint_delete (cp);
}

RAM_Debug *ram_debug_new ()
{
   RAM_Debug *rd;

   rd = (RAM_Debug *) calloc (1, sizeof (RAM_Debug));
  
   if (! rd)
      err_fatal_perror ("calloc", "cannot allocate memory for RAM_Debug");
   
   return rd;
}

// Do not touch memory and FILEs inside
void ram_debug_only_delete (RAM_Debug *rd)
{
   if (rd -> breakpoint)
      free (rd -> breakpoint);

   if (rd -> watchpoints)
      ram_watchpoint_delete_list (rd -> watchpoints);

   if (rd -> catchpoints)
      ram_watchpoint_delete_list (rd -> watchpoints);

   free (rd);
}

RAM_Debug *ram_debug_new_by (RAM *rm, FILE *input, FILE *output, RAM_Text *text)
{
   RAM_Debug *rd;

   rd = ram_debug_new ();

   (rd -> machine) = rm;
   (rd -> text) = text;
   (rd -> input) = input;
   (rd -> output) = output;

   if (rm)
   {
      (rd -> is_running) = ram_is_running (rm) &&
	      mpz_sgn (rm -> instructions_done);

      if (rd -> is_running)
         (rd -> is_stopped) = 1;
   }

   return rd;
}

// rm can be NULL, but input and output have to be correct streams,
// because they'll be used on a machine creation.
int ram_debug_by (RAM *rm, FILE *input, FILE *output,
			RAM_Text *text)
{
   RAM_Debug *rd;
   int return_this;

   rd = ram_debug_new_by (rm, input, output, text);

   return_this = ram_debug (rd);
   
   ram_debug_only_delete (rd);

   return return_this;
}

/* ========	Debugging stuff ======== */

void inline ram_debug_reset_machine (RAM_Debug *rd)
{
   ram_reset (rd -> machine);

   (rd -> is_running) = 1;
   (rd -> is_stopped) = 0;
}

static int compare_breakpoints (const void *a, const void *b)
{
   if (*((unsigned int *)a) > *((unsigned int *)b))
      return 1;
   
   if (*((unsigned int *)a) < *((unsigned int *)b))
      return -1;

   return 0;
}

// Returns pointer to the breakpoint if found, or NULL if there is no one.
unsigned int *find_breakpoint (RAM_Debug *rd, unsigned int point)
{
   return bsearch (&point, (rd -> breakpoint), (rd -> breakpoints),
		   sizeof (unsigned int), compare_breakpoints);
}

// `watchpoint' have to be a correct expression.
// It is copied, not referenced.
RAM_Watchpoint *add_watchpoint (RAM_Debug *rd, const char *watchpoint)
{
   RAM_Watchpoint *wp = ram_watchpoint_new ();

   (wp -> expression) = strdup (watchpoint);
   (wp -> next) = NULL;

   if (rd -> watchpoints)
   {
      RAM_Watchpoint *w = (rd -> watchpoints);
      while (w -> next)
	 w = (w -> next);

      (w -> next) = wp;
   }
   else
      (rd -> watchpoints) = wp;

   return wp;
}

static int last_catchpoint_index = 0;

RAM_Catchpoint *add_catchpoint (RAM_Debug *rd, const char *catchpoint)
{
   RAM_Catchpoint *cp = ram_catchpoint_new ();

   (cp -> expression) = strdup (catchpoint);
   (cp -> next) = NULL;

   last_catchpoint_index ++;
   (cp -> index) = last_catchpoint_index;

   if (rd -> catchpoints)
   {
      RAM_Catchpoint *c = (rd -> catchpoints);
      while (c -> next)
	 c = (c -> next);

      (c -> next) = cp;
   }
   else
      (rd -> catchpoints) = cp;

   return cp;
}

// This function assumes that its args are correct.
// The value returned is the number of the breakpoint hit,
// or 0 if none found.
int instruction_by_line (RAM_Debug *rd, int line)
{
   // Customized bsearch here.
   unsigned int *left = (rd -> text -> instruction_line),
   		*right = (rd -> text -> instruction_line) +
			(rd -> machine -> program -> n) - 1;

   line --;

   if (line >= (rd -> text -> text_size) || line < 0)
      return 0;

   if (line <= *left)
      return 1;
   if (line > *right)
      return 0;

   while (right - left > 1)
   {
      unsigned int *middle = left + (right - left) / 2;

      if (*middle > line)
	 right = middle;
      else
	 left = middle;
   }

   if (line == *left)
      return left - (rd -> text -> instruction_line) + 1;

   return right - (rd -> text -> instruction_line) + 1;
}

// Useful inline function, but it doesn't perform any error checking.
inline int line_by_instruction (RAM_Debug *rd, int instruction)
{
   return (rd -> text -> instruction_line) [instruction];
}

// Returns true if program is not stopped yet
int emulation_step (RAM_Debug *rd)
{
   if (!(rd -> is_running) || (rd -> is_stopped))
      return 0;

   if (! (rd -> bypass_breakpoint))
   {
      unsigned int *breakpoint = find_breakpoint
	      (rd, (rd -> machine -> current_instruction));

      if (breakpoint)
      {
	 (rd -> is_stopped) = 1;
	 (rd -> interrupt_reason) = RAM_DEBUG_BREAKPOINT;
	 (rd -> interrupt_reason_parameter) = breakpoint - (rd -> breakpoint);

	 (rd -> bypass_breakpoint) = 1;

	 return 0;
      }
   }
   else
      (rd -> bypass_breakpoint) = 0;

   if (! ram_do_instruction (rd -> machine))
   {
      (rd -> is_running) = 0;
      (rd -> is_stopped) = 1;
      return 0;
   }

   {
      RAM_Catchpoint *c = (rd -> catchpoints);
      mpz_t value;

      mpz_init (value);

      while (c)
      {
         if (ram_evaluate_expression ((rd -> machine), (c -> expression),
				 	&value))
	 {
	    if (mpz_sgn (value))
	    {
	       if (! (c -> is_ignored))
	       {
	          (rd -> is_stopped) = 1;
	          (rd -> interrupt_reason) = RAM_DEBUG_CATCHPOINT;
	          (rd -> interrupt_reason_parameter) = (c -> index);

		  (c -> is_ignored) = 1;

	          break;
	       }
	    }
	    else
	       (c -> is_ignored) = 0;
	 }
	 else
	    err_warning ("invalid catchpoint %d, `%s'", (c -> index),
			    	(c -> expression));

	 c = (c -> next);
      }

      mpz_clear (value);
   }

   return ! (rd -> is_stopped);
}

// Closes program.  This function assumes that there is correct session.
void ram_close_program (RAM_Debug *rd)
{
   ram_clear (rd -> machine);
   ram_text_reset (rd -> text);

   if (rd -> breakpoints)
   {
      free (rd -> breakpoint);
      (rd -> breakpoints) = 0;
   }

   if (rd -> watchpoints)
   {
      ram_watchpoint_delete_list (rd -> watchpoints);
      (rd -> watchpoints) = 0;
   }

   if (rd -> catchpoints)
   {
      ram_catchpoint_delete_list (rd -> catchpoints);
      (rd -> catchpoints) = 0;
   }

   (rd -> is_running) = 0;
   (rd -> is_stopped) = 0;

   rewind (rd -> input);
}
