/*
 * 	Random Access Machine.
 * 	Debugger command-line interface.
 *
 * 	Copyright (C) 2002, 2003  Dmitry Rutsky	<rutsky@school.ioffe.rssi.ru>
 * 	
 * 	This program is free software; you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation; either version 2 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program; if not, write to the 
 * 	Free Software Foundation, Inc.,
 * 	59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <signal.h>
#include <setjmp.h>
#include <readline/readline.h>
#include <readline/history.h>

#include "debugger.h"
#include "legal_attributes.h"
#include "config.h"

/* ========	Some prototypes		======== */

static int ask_yn (const char *what);

static inline void message (const char *, ...);
static inline void message_noeol (const char *, ...);

static inline void no_repetition (RAM_Debug *);

static void report_state (RAM_Debug *);

/* ========	Some global options	======== */

static const char *history_file_name = NULL;

// Note: name should either point to the valid string all the time of debugger
// operation, or be NULL.
const char *ram_debug_set_history_file (const char *name)
{
   const char *old = history_file_name;

   history_file_name = name;

   return old;
}

static int history_max = 0;

int ram_debug_set_history_max (int max)
{
   int old = history_max;

   history_max = max;

   return old;
}

/* ========	Some interface-specific debugger stuff	======== */

// Returns whether breakpoint was successfully added
static int add_breakpoint (RAM_Debug *rd, unsigned int point)
{
   int i;

   if (find_breakpoint (rd, point))
   {
      printf ("There is a breakpoint at instruction %d already.\n",
		      point + 1);
      return 0;
   }
   
   (rd -> breakpoint) = (unsigned int *) realloc ((rd -> breakpoint),
						  (rd -> breakpoints) + 1);
   if (! (rd -> breakpoint))
      err_fatal_perror ("realloc",
	    "add_breakpoint: couldn't allocate memory for breakpoint");

   // Keep the breakpoints sorted
   for (i = 0; (i < (rd -> breakpoints)) && ((rd -> breakpoint) [i] < point); 
		   i ++);

   if (i < (rd -> breakpoints))
      memmove ((rd -> breakpoint) + i + 1, (rd -> breakpoint) + i, 
		sizeof (unsigned int) * ((rd -> breakpoints) - i));
	   
   (rd -> breakpoint) [i] = point;
   (rd -> breakpoints) ++;

   return 1;
}

// Displays the naked instruction, as it looks to the emulator.
static void display_instruction (RAM_Debug *rd)
{
   RAM_Instruction *i;
   
   const char *instruction_name [] =
   {
      "<no instruction>", "READ", "WRITE", "LOAD", "STORE", 
      "ADD", "NEG", "HALF", "JUMP", "JGTZ", "HALT"
   };
   
   if ((rd -> machine -> current_instruction) >= 
		   (rd -> machine -> program -> n))
      return;

   i = (rd -> machine -> program -> instructions)
	   		[rd -> machine -> current_instruction];
   message_noeol ("Instruction %d:\t%s",
		   (rd -> machine -> current_instruction) + 1,
		   instruction_name [i -> instruction]);

   switch (i -> parameter_type)
   {
   case RAM_INSTRUCTION:
      message_noeol ("\t%Zd", (i -> parameter));
      break;
   case RAM_POINTER:
      message_noeol ("\t[%Zd]", (i -> parameter));
      break;
   case RAM_INDIRECT_POINTER:
      message_noeol ("\t[[%Zd]]", (i -> parameter));
      break;
   case RAM_CONSTANT:
      message_noeol ("\t");
      mpz_out_str (stdout, 10, (i -> parameter));
      break;
   case RAM_NO_PARAMETER:
      break;
   }

   message ("");
}

// Returns `true' if everything was displayed correctly.
static int display_watchpoint (RAM_Debug *rd, RAM_Watchpoint *wp)
{
   if (!ram_display_expression ((rd -> machine), "`%s'\t:\t",
			   (wp -> expression), stdout, (wp -> expression)))
   {
      message ("Disabling invalid watchpoint `%s'.", (wp -> expression));
      ram_disable_watchpoint (rd, wp);
      return 1;
   }

   message ("");

   return 1;
}

// This function assumes that index is index of correct existing catchpoint.
static int display_catchpoint (RAM_Debug *rd, int index)
{
   RAM_Catchpoint *c = (rd -> catchpoints);

   while ((c -> index) != index)
      c = (c -> next);

   ram_display_expression ((rd -> machine), "`%s'\t:\t",
			   (c -> expression), stdout, (c -> expression));
   message ("");

   return 1;
}

static void display_watchpoints (RAM_Debug *rd)
{
   RAM_Watchpoint *wp = (rd -> watchpoints);

   while (wp)
   {
      display_watchpoint (rd, wp);
      
      wp = (wp -> next);
   }
}

// Returns true if it has successfuly closed the program.
static int close_the_program (RAM_Debug *rd)
{
   if (rd -> machine)
      if (!(rd -> is_running) ||
	  ask_yn ("Close the session for the program being debugged? (y/n) "))
      {
	 ram_close_program (rd);

	 return 1;
      }

   no_repetition (rd);
   
   return 0;
}

static void state_of_the_machine (RAM_Debug *rd)
{
   if (rd -> machine)
   {
      if (rd -> machine -> program)
      {
         message ("The program has %d instruction%s.\n",
			 (rd -> machine -> program -> n),
			 ((rd -> machine -> program -> n) > 1) ?
			 	"s" : "");
	 if (rd -> is_running)
	 {
	    message ("The program is running, next instruction is "
			    "instruction %d.",
			    (rd -> machine -> current_instruction) + 1);
	    display_instruction (rd);
	    message ("\nThe machine has performed %Zd instruction%s, "
			 "it has taken %Zd time unit%s.",
			    (rd -> machine -> instructions_done),
			    (mpz_cmp_ui ((rd -> machine -> instructions_done),
				    1) > 0) ?
			    	"s" : "",
			    (rd -> machine -> time_consumed),
			    (mpz_cmp_ui ((rd -> machine -> time_consumed),
				    1) > 0) ?
			    	"s" : ""
				      );
	    
	    if (rd -> is_stopped)
	       message (
		"The emulator had paused its execution at instruction %d,\n"
			    "%s.",
			    (rd -> machine -> current_instruction),
		((rd -> interrupt_reason) == RAM_DEBUG_TRACING) ?
			"because of the debugging activities" :
		(((rd -> interrupt_reason) == RAM_DEBUG_BREAKPOINT) ?
		 	"due to breakpoint met" :
		((((rd -> interrupt_reason) == RAM_DEBUG_CATCHPOINT) ?
		  	"due to catchpoint met" :
		(((((rd -> interrupt_reason) == RAM_DEBUG_INTERRUPT) ?
		   	"because emulation was interrupted" :
			"don't know why")))))));
	 }
	 else if (rd -> is_stopped)
	    message ("The machine had halted at position %d.", 
			    (rd -> machine -> current_instruction));
	 else
	    message ("The program is not running.");
      }
      else
	 message ("No program is loaded.");

      message_noeol ("\nThe emulator has allocated memory for %d "
	"register%s", (rd -> machine -> memory -> allocated),
		      ((rd -> machine -> memory -> allocated) > 1) ? "s" : "");
      message (" in %d segment%s;", (rd -> machine -> memory -> segment_count),
		   ((rd -> machine -> memory -> segment_count) > 1) ? "s" : "");
      message ("%s arranged in a balanced binary tree of height %d.",
	((rd -> machine -> memory -> segment_count) > 1) ? "they are" : "it is",
		      ram_memory_tree_height (rd -> machine -> memory));
   }
   else
      message ("No even a machine structure was created yet.");
}

static void state_of_the_debugger (RAM_Debug *rd)
{
   if (rd -> machine)
   {
      if (rd -> breakpoints)
      {
	 unsigned int i;
	 
	 message_noeol ("There %s %d breakpoint%s at instruction%s%s ",
			 ((rd -> breakpoints) > 1) ? "are" : "is",
			 (rd -> breakpoints),
			 ((rd -> breakpoints) > 1) ? "s" : "",
			 ((rd -> breakpoints) > 1) ? "s" : "",
			 ((rd -> breakpoints) > 1) ? ":" : "");
	 for (i = 0; i < (rd -> breakpoints); i ++)
	    message_noeol ("%d%s", (rd -> breakpoint) [i] + 1,
			    (i < ((rd -> breakpoints) - 1)) ? ", " : ".\n");
      }

      {
	 RAM_Catchpoint *c = (rd -> catchpoints);

	 if (c)
	 {
	    message ("There %s catchpoint%s:", (c -> next) ? "are" : "is a",
			    (c -> next) ? "s" : "");
	    while (c)
	    {
	       message ("%d:\t`%s'", (c -> index), (c -> expression));

	       c = (c -> next);
	    }
	 }
      }
      
      if (rd -> text)
	 if (rd -> text -> text_size)
	    message ("The program text is loaded, it has %d line%s.",
			    (rd -> text -> text_size),
			    ((rd -> text -> text_size) > 1) ? "s" : "");
   }
   else
      message ("No even a machine structure created yet.");
}

// Check if program is halted and try to reset the emulator.
// Return `true' if it is successfully done.
static int check_if_running (RAM_Debug *rd)
{
   if (rd -> is_running)
      return 1;

   if (rd -> is_stopped)
      if (! ask_yn ("The machine is halted.  Restart it? (y/n) "))
      {
	 no_repetition (rd);
	 return 0;
      }

   ram_debug_reset_machine (rd);
   return 1;
}

static void get_running (RAM_Debug *rd)
{
   (rd -> is_running) = 1;
   (rd -> is_stopped) = 0;
   do
      if (verbosity_is_enough (VERBOSITY_EXTENSIVE_DEBUG))
	 display_instruction (rd);
   while (emulation_step (rd));
}

/* ========	Command handlers definition part	======== */

typedef int (CommandHandler)(RAM_Debug *rd, int is_repeat, char *parameter);

typedef struct
{
   const char *name;
   const char *abbreviation;
   const char *short_description;
   const char *description;
   
   CommandHandler *handler;
}
DebuggerCommand;

static CommandHandler quit, help, print, display, breakpoint, catch,
			run, step, list, open, state, cont, verbosity,
			copyright, warranty, version;

// Note that for the correct command matching commands with abbreviations
// containing other abbreviations (e.g. "continue" and "copyright") have to be
// before the commands abbreviations of which they contain.
static const DebuggerCommand command [] =
{
   {"breakpoint", "b", "insert breakpoint", 
"Set breakpoint at given position.\n"
"Argument can be line number in input file, or instruction number\n"
"if it is prefixed by sharp.", breakpoint},
   {"catch", "ca", "catch a condition", 
"Terminate machine emulation if the given conditional expression evaluates as\n"
"`true' (nonzero value).  See `help expressions' for online information on\n"
"allowed expressions."
"Argument can be line number in input file, or instruction number\n"
"if it is prefixed by sharp.", catch},
   {"copyright", "co", "copyright notice",
"Provide the copyright information for the emulator.", copyright},
   {"continue", "c", "continue program execution",
"Continue program execution.\n"
"If integer argument N is specified,\n"
"the next (N - 1) breakpoints (if there are so many) will be ignored.", cont},
   {"display", "di", "add watchpoint", "This command is similar to the "
"`print' one,\n"
"but it adds the expression to the watchpoint list, a list of expressions\n"
"being automatically displayed within program tracing, or when a breakpoint\n"
"is reached.  See also `help expressions' for description of allowed\n"
"expressions", display},
   {"help", "h", "get help on commands", 
"Get the description of the specified command.\n"
"If command not specified, list of commands with short descriptions\n"
"is displayed.", help},
   {"list", "l", "view part of program text", "View part of program text.\n"
"If line number specified, display 10 lines of program text from "
"that line.\nIf line number and number of lines specified, "
"display that number of lines.", list},
   {"open", "o", "open a program",
"Load the program from a given file.", open},
   {"print", "p", "print expression value", 
"Print an expression value.  See `help expressions' for further information.\n",
	print},
   {"quit", "q", "exit current debug session", "Exit this debug session.",
	quit},
   {"run", "r", "start program", "Start the program being debugged.", run},
   {"state", "st", "display the machine state",
"Display the current state of the machine.  If the argument `debugger` is\n"
"given, display some debugger statistics.", state},
   {"step", "s", "program tracing",
"Execute one instruction and stop.\n"
"If integer argument N specified, N steps will be done\n"
"(at most N steps, machine may stop during this operation).", step},
   {"version", "ve", "version information",
"Provide version information of the emulator.", version},
   {"verbosity", "v", "verbosity level",
"Display the verbosity level, or set it to N where N is integer argument\n"
"of the command.", verbosity},
   {"warranty", "w", "warranty information",
"Provide warranty information for this emulator.", warranty},
   {NULL, NULL, NULL, NULL, NULL}
};

/* ========	Miscellaneous interface stuff	======== */

// Do not repeat the current command at `Enter'.
static inline void no_repetition (RAM_Debug *rd)
{
   if (rd -> previous_command)
   {
      free (rd -> previous_command);
      (rd -> previous_command) = 0;
   }
}

static inline void command_error (RAM_Debug *rd, const char *error, ...)
{
   va_list args;
   
   va_start (args, error);

   gmp_vprintf (error, args);
   printf ("\n");

   va_end (args);

   no_repetition (rd);
}

static int ask_yn (const char *what)
{
   int c = 0;
  
   do
   {
      if (c)
	 message ("Please answer `y' or `n'.");

      message_noeol ("%s", what);

      c = tolower (getchar ());
      {
	 int d = 0, count = 0;
	 do
	 {
	    d = getchar ();
	    count ++;
	 }
	 while (d != '\n');

	 if (count != 1)
	    c = 1;
      }
   }
   while (!strchr ("yn", c));
   
   return (c == 'y');
}

static inline void message (const char *m, ...)
{
   if (verbosity_is_enough (VERBOSITY_ERRORS))
   {
      va_list args;

      va_start (args, m);

      gmp_vprintf (m, args);
      printf ("\n");

      va_end (args);
   }
}

static inline void message_noeol (const char *m, ...)
{
   if (verbosity_is_enough (VERBOSITY_ERRORS))
   {
      va_list args;

      va_start (args, m);

      gmp_vprintf (m, args);

      va_end (args);
   }
}

static const char *DELIMITERS = " \n\t,";

// Return the pointer to the next token in line if integer parameter is read.
// Make `*parameters' point to the next argument, or to NULL.
int get_unsigned_int_parameter (char **parameters, unsigned int *parameter)
{
   char *p = strsep (parameters, DELIMITERS);

   if (! p)
      return 0;

   while (*p == '\0')
      p = strsep (parameters, DELIMITERS);

   return (sscanf (p, "%u", parameter) == 1);
}

static int get_int_parameter (char **parameters, int *parameter)
{
   char *p = strsep (parameters, DELIMITERS);

   if (! p)
      return 0;

   while (*p == '\0')
      p = strsep (parameters, DELIMITERS);

   return (sscanf (p, "%d", parameter) == 1);
}

// Skip all delimiters, and tell if there is something at all.
static int is_any_parameter (char **s)
{
   if (*s == NULL)
      return 0;

   while (strchr (DELIMITERS, **s))
      (*s) ++;

   return (**s != '\0');
}

// Get the `instruction pointer' parameter, which can be either text line
// number, or instruction number if it is prefixed with sharp `#'.
// Returns `true' if parameter was read successfully, `false' if not,
// and then `instruction' will be set to `true' if there actually were no
// parameters, or to `false' if the parameter is bogus.
static int get_instruction_pointer (RAM_Debug *rd, char **s, int *instruction)
{
   int is_instruction_no = 0;
   char *next;
   
   if (!is_any_parameter (s))
   {
      *instruction = 1;
      return 0;
   }

   if (**s == '#')
   {
      (*s) ++;
      is_instruction_no = 1;
   }

   if (!is_any_parameter (s))
   {
      (*instruction) = !is_instruction_no;
      return 0;
   }

   next = *s;
   while (strchr (DELIMITERS, *next) == NULL)
      next ++;

   if (*next == '\0')
      next = NULL;
   else
   {
      *next = '\0';
      next ++;
   }
   
   if (sscanf (*s, "%d", instruction) != 1)
   {
      message ("Unrecognized argument.");
      *instruction = 0;
      return 0;
   }

   if (!is_instruction_no)
   {
      int i = instruction_by_line (rd, *instruction);
      if (i <= 0)
      {
	 message ("Line %d is out of the program.", *instruction);
	 *instruction = 0;
	 return 0;
      }
      
      *instruction = i - 1;
   }
   else
   {
      if (*instruction < 1 || *instruction > (rd -> machine -> program -> n))
      {
	 message ("There is no instruction %d in the program.", *instruction);
	 *instruction = 0;
	 return 0;
      }

      (*instruction) --;
   }

   *s = next;

   return 1;
}

// Print the string `s' and several tabulations trying to make it `size'
// chars width.
void nputs (const char *s, const unsigned int size)
{
   int i;
	 
   printf ("%s", s);
   i = strlen (s);
   
   //	Assuming our tab is always 8 spaces long
   while (i < size)
   {
      printf ("\t");
      i /= 8;
      i ++;
      i *= 8;
   }
}

static int is_the_command (const DebuggerCommand *dc, const char *s)
{
   if (!strcmp (s, (dc -> name)))
      return 1;

   if (!strncmp (s, (dc -> name), strlen (s)) &&
		   (strlen (s) >= strlen (dc -> abbreviation)))
      return 1;

   return 0;
}

/* ========	Command handlers	========= */

static int cont (RAM_Debug *rd, int is_repeat, char *parameter)
{
   int n = 1;

   if (! (rd -> machine))
   {
      command_error (rd, "No program is loaded.");
      return 0;
   }
  
   if (! (rd -> is_running))
   {
      command_error (rd, "The program is not being run.");
      return 0;
   }
   
   if (parameter)
   {
      if (! get_int_parameter (&parameter, &n))
      {
	 command_error (rd, "Unrecognized parameter -- try `help step'.");
	 return 0;
      }

      if (n < 1)
      {
	  command_error (rd, "Cannot omit %d breaks.", n);
	  return 0;
      }
   }
   
   (rd -> is_running) = 1;
   (rd -> is_stopped) = 0;
   while (n > 0)
   {
      get_running (rd);

      if (!(rd -> is_stopped))
	 n = 1;

      n --;
   }

   report_state (rd);
   display_watchpoints (rd);
  
   return 0;
}

static int state (RAM_Debug *rd, int is_repeat, char *parameters)
{
   no_repetition (rd);
   
   if (parameters)
   {
      if (!strcmp (parameters, "debugger"))
         state_of_the_debugger (rd);
      else
	 printf ("Unrecognized parameter -- try `help state'.\n");
   }
   else
      state_of_the_machine (rd);
   
   return 0;
}

static int open (RAM_Debug *rd, int is_repeat, char *parameters)
{
   FILE *f;
   RAM_Program *program;

   no_repetition (rd);
   
   if (!parameters)
   {
      command_error (rd, "You should specify a program to open.");
      return 0;
   }
   
   f = fopen (parameters, "r");
   if (!f)
   {
      command_error (rd, "Cannot open `%s'.", parameters);
      perror ("fopen");
      return 0;
   }
   
   if (rd -> machine)
      if (! close_the_program (rd))
      {
	 fclose (f);
	 return 0;
      }

   err_debug ("Parsing the program %s...", parameters);

   if (!(rd -> text))
      (rd -> text) = ram_text_new ();

   program = ram_program_parse (f, (rd -> text));

   fclose (f);
   if (! program)
      return 0;
 
   if (rd -> machine)
      ram_delete (rd -> machine);

   (rd -> machine) = ram_new_by_program (program);
   (rd -> machine -> input) = (rd -> input);
   (rd -> machine -> output) = (rd -> output);

   return 0;
}

static int quit (RAM_Debug *rd, int is_repeat, char *parameters)
{
   if (rd -> machine)
      return close_the_program (rd);
   else
      return 1;
	
   return 0;
}

static int help (RAM_Debug *rd, int is_repeat, char *parameters)
{	
   const DebuggerCommand *dc;

   no_repetition (rd);

   if (!parameters)
      printf (
"Use `help' command with the name or abbreviation of a command\n"
"to get its full description.\n\n"
"If the line starts with exclamation mark, everything after it\n"
"will be passed to shell for execution.\n\n"
"Command name\tShortest abbreviation\tShort description\n"
"=============== ======================= =================================\n");
   
   if (parameters)
      if (!strcmp (parameters, "expressions"))
      {
	 message (
"Expressions may contain integer numbers, standard arithmetic (`+', `-', `*',\n"
"`/'), as well as raise in power (`^') and integer factorial (`!').\n"
"Round braces (`(' and `)') may be used in usual way, square braces (`[' and\n"
"`]') perform dereference of their content (e.g. [ 2 + 3 ] means value stored\n"
"in register [5]).\n\n"
"Logical values are handled in the same way as integers, zero means `false'\n"
"and nonzero 'true' respectively.  Usial conditionals `>' (greater), `>='\n"
"(greater or equal), `<' (less), `<=' (less or equal), `=' (equal), and '!='\n"
"(not equal) are allowed, as well as logical operators `&' (and), `|' (or),\n"
"and '^' (not).\n\n"
"Examples:\n"
"\"[0]\":\t\tcontent of register 0;\n"
"\"[2] >= 10! & [5] != 2^[4]\":\tif content of register 2 is greater or equal\n"
"\tto factorial of 10 and content of register 5 is not equal to 2 raised\n"
"\tin power of content of register 4."
		);

	 return 0;
      }

   dc = command;
   while (dc -> name)
   {
      if (parameters)
      {
	 if (is_the_command (dc, parameters))
	 {
	    message ("%s\t--\t%s\n\n%s", (dc -> name),
			    (dc -> short_description),
			    (dc -> description));
	    return 0;
	 }
      }
      else
      {
 	 nputs ((dc -> name), 16);
	 nputs ((dc -> abbreviation), 24);

	 message (dc -> short_description);
      }
      
      dc ++;
   }

   if (parameters)
      message ("There is no command to match a `%s'.", parameters); 
   
   return 0;
}

static int breakpoint (RAM_Debug *rd, int is_repeat, char *parameters)
{
   int point;

   no_repetition (rd);
   
   if (! (rd -> machine))
   {
      command_error (rd, "There is no program to insert a breakpoint in.");
      return 0;
   }
  
   if (parameters)
   {
      if (get_instruction_pointer (rd, &parameters, &point))
      {
	 if (is_any_parameter (&parameters))
	 {
	    command_error (rd, "breakpoint: unrecognized argument.");
	    return 0;
	 }
      }
      else
      {
	 if (point)
            command_error (rd, "breakpoint: command requires an argument.");
	 
         return 0;
      }
   }
   else
   {
      command_error (rd, "breakpoint: command requires an argument.");
      return 0;
   }

   if (! add_breakpoint (rd, point))
      return 0;

   message ("Breakpoint at instruction %d, line %d added.",
		      point + 1, line_by_instruction (rd, point) + 1);
   
   return 0;

}

static void report_state (RAM_Debug *rd)
{
   unsigned int instruction = (rd -> machine -> current_instruction),
      		line;

   if (instruction < (rd -> machine -> program -> n))
      line = (rd -> text -> instruction_line) [instruction];

   if ((rd -> is_running) && (rd -> is_stopped))
   {
      if ((rd -> interrupt_reason) == RAM_DEBUG_BREAKPOINT)
	 message ("Breakpoint %d reached at instruction %d, line %d:",
			 (rd -> interrupt_reason_parameter) + 1,
			  instruction + 1, line + 1);
      else if ((rd -> interrupt_reason) == RAM_DEBUG_CATCHPOINT)
      {
	 message ("Catchpoint %d reached at instruction %d, line %d:",
			 (rd -> interrupt_reason_parameter),
			  instruction + 1, line + 1);
	 display_catchpoint (rd, (rd -> interrupt_reason_parameter));
      }
      else
         message ("The %s at instruction %d, line %d:",
	   ((rd -> interrupt_reason) == RAM_DEBUG_INTERRUPT) ?
		"emulator was interrupted" :
	   (((rd -> interrupt_reason) == RAM_DEBUG_TRACING) ?
		"machine is" : "bad thing happened"),
		      instruction + 1, line + 1);
      message ((rd -> text -> text) [line]);
   }
   else if (!(rd -> is_running) && (rd -> is_stopped))
   {
      if (instruction < (rd -> machine -> program -> n))
      {
         message ("The machine had halted at instruction %d, line %d:",
		      instruction + 1, line + 1);
         message ((rd -> text -> text) [line]);
      }
      else
	 message ("The machine had halted at position %d.",
			 instruction);
   }
}

static int run (RAM_Debug *rd, int is_repeat, char *parameter)
{
   if (! (rd -> machine))
   {
      command_error (rd, "There is no program to start.");
      return 0;
   }
   
   if (rd -> is_running)
      if (! ask_yn ("There is a running program.  "
			      "Start it from the beginnig? (y/n) "))
      {
	 no_repetition (rd);
	 return 0;
      }
   
   err_debug ("Starting the program...");

   ram_debug_reset_machine (rd);

   get_running (rd);

   report_state (rd);
   display_watchpoints (rd);
   
   return 0;
}

static int copyright (RAM_Debug *rd, int is_repeat, char *parameter)
{
   if (parameter)
   {
      command_error (rd, "Unexpected parameter -- try `help copyright'.");
      return 0;
   }

   message (COPYRIGHT);

   no_repetition (rd);

   return 0;
}

static int warranty (RAM_Debug *rd, int is_repeat, char *parameter)
{
   if (parameter)
   {
      command_error (rd, "Unexpected parameter -- try `help warranty'.");
      return 0;
   }

   message (WARRANTY);

   no_repetition (rd);

   return 0;
}

static int version (RAM_Debug *rd, int is_repeat, char *parameter)
{
   if (parameter)
   {
      command_error (rd, "Unexpected parameter -- try `help version'.");
      return 0;
   }

   message (PACKAGE_VERSION);

   no_repetition (rd);

   return 0;
}

static int verbosity (RAM_Debug *rd, int is_repeat, char *parameter)
{
   int level;

   if (parameter)
   {
      if (!get_int_parameter (&parameter, &level))
      {
	 command_error (rd, "Unrecognized parameter -- try `help verbosity'.");
	 return 0;
      }

      err_debug ("The verbosity level was %d (%s).", verbosity_get (), 
		      verbosity_description_string (verbosity_get ()));

      verbosity_set (level);
      message ("The verbosity level has been changed to %d (%s).", level,
		      verbosity_description_string (level));
   }
   else
      message ("The verbosity level is %d (%s).", verbosity_get (),
		      verbosity_description_string (verbosity_get ()));

   no_repetition (rd);

   return 0;
}

static int step (RAM_Debug *rd, int is_repeat, char *parameter)
{
   int n = 1;

   if (! (rd -> machine))
   {
      command_error (rd, "There is no program for tracing.");
      return 0;
   }
   
   if (parameter)
   {
      if (!get_int_parameter (&parameter, &n))
      {
	 command_error (rd, "Unrecognized parameter -- try `help step'.");
	 return 0;
      }

      if (n < 1)
      {
	  command_error (rd, "Cannot execute %d steps.", n);
	  no_repetition (rd);
	  return 0;
      }
   }

   if (! check_if_running (rd))
      return 0;
  
   // This prevents duplicate instruction display
   if (! verbosity_is_enough (VERBOSITY_EXTENSIVE_DEBUG))
      display_instruction (rd);

   (rd -> is_running) = 1;
   (rd -> is_stopped) = 0;
   while (n > 0)
   {
      if (! emulation_step (rd))
	 n = 1;

      n --;
   }

   if ((rd -> is_running) && !(rd -> is_stopped))
   {
      (rd -> is_stopped) = 1;
      (rd -> interrupt_reason) = RAM_DEBUG_TRACING;
   }

   report_state (rd);
   display_watchpoints (rd);

   if (! (rd -> is_running))
      no_repetition (rd);
  
   return 0;
}

static int display (RAM_Debug *rd, int is_repeat, char *parameter)
{
   RAM_Watchpoint *wp;
   
   no_repetition (rd);

   if (!parameter)
   {
      if (! (rd -> watchpoints))
	 message ("No watchpoints.");
      else
         message ("Active watchpoints:");
      
      display_watchpoints (rd);

      return 0;
   }
  
   wp = add_watchpoint (rd, parameter);
   err_debug ("Watchpoint `%s' has been added.", parameter);
 
   display_watchpoint (rd, wp);
   
   return 0;
}

static int catch (RAM_Debug *rd, int is_repeat, char *parameter)
{
   RAM_Catchpoint *cp;
   
   no_repetition (rd);

   if (!parameter)
   {
      message ("A condition have to be specified.");
      return 0;
   }
  
   cp = add_catchpoint (rd, parameter);
   err_debug ("Catchpoint %d, `%s' has been added.", (cp -> index), parameter);
   
   return 0;
}

static int print (RAM_Debug *rd, int is_repeat, char *parameter)
{
   no_repetition (rd);
   
   if (!parameter)
   {
      command_error (rd, "An expression have to be specified.");
      return 0;
   }
  
   if (ram_display_expression ((rd -> machine), "", parameter, stdout))
      message ("");
   
   return 0;
}

static int list (RAM_Debug *rd, int is_repeat, char *parameter)
{
   static int prev_line_n = -1, next = 0, n_lines = 10, prev_n_lines = -1;
   int line_n, i, instruction, have_param = 0;
   char **line;

   if (!(rd -> machine) || !(rd -> machine -> program))
   {
      command_error (rd, "There is no program to list.");
      return 0;
   }
   
   if (parameter && !is_repeat)
   {
      if (!get_int_parameter (&parameter, &line_n))
      {
	 command_error (rd, "Unrecognized parameter.", parameter);
	 return 0;
      }

      if (line_n != prev_line_n)
      {
         have_param = 1;
	 prev_line_n = line_n;
      }
      
      line = (rd -> text -> text) + line_n - 1;

      if (is_any_parameter (&parameter))
      {
         if (! get_int_parameter (&parameter, &n_lines))
	 {
	    command_error (rd, "list: unrecognized parameter.");
	    return 0;
	 }
	 else if (n_lines < 1)
	 {
	    command_error (rd, "list: cannot display %d lines of the text.",
			    n_lines);
	    return 0;
	 }

	 if (n_lines != prev_n_lines)
	 {
	    prev_n_lines = n_lines;
	    have_param = 1;
	 }
      }
   }
   else if (!is_repeat)
   {
      instruction =
	      ((rd -> is_running) ? (rd -> machine -> current_instruction) : 0)
	      		+ 1;
      line_n = (rd -> text -> instruction_line) [instruction - 1];
      line = (rd -> text -> text) + line_n;
      line_n ++;
   }

   if (!have_param && is_repeat && next)
   {
      line_n = next;
      line = (rd -> text -> text) + line_n - 1;
   }

   if (line_n < 1 || line_n > (rd -> text -> text_size))
   {
      command_error (rd, "There is no line %d in the text.", line_n);
      return 0;
   }

   if (line_n + n_lines >= (rd -> text -> text_size))
      n_lines = (rd -> text -> text_size) - line_n + 1;
   
   for (i = 0; i < n_lines; i ++, line ++)
      message ("%d\t%s", i + line_n, *line);

   next = line_n + n_lines;
   if (next >= (rd -> text -> text_size))
   {
      next = 0;
      n_lines = 10;
      no_repetition (rd);
   }
  
   return 0;
}

/* ========	Interface part	======== */

static void print_prompt ()
{
   message ("Debugger mode.  Type 'h' or 'help' for interactive help.");
}

static char *command_generator (const char *text, int state)
{
   static const DebuggerCommand *dc;
   static int length;

   if (!state)
   {
      dc = command;
      length = strlen (text);
   }

   while (dc -> name)
   {
      const char *name = (dc -> name);
      
      dc ++;

      if (! strncmp (name, text, length))
	 return strdup (name);
   }

   return NULL;
}

static inline char *command_state_generator (const char *text, int state)
{
   return state ? NULL : "debugger";
}

// Record the `text' to show it later if `state' is -1.
static inline char *braces_generator_ctl (const char *text, int state)
{
   static const char *t = NULL;

   if (state == -1)
   {
      t = text;
      return NULL;
   }

   return state ? NULL : ((char *) t);
}

static inline char *braces_generator (const char *text, int state)
{
   return braces_generator_ctl (text, state ? 1 : 0);
}

static inline char *empty_generator (const char *text, int state)
{
   return NULL;
}

typedef struct _Brace	// We trace the braces :-)
{
   char brace;
   struct _Brace *next;
}
Brace;

static inline Brace *brace_new ()
{
   Brace *b = (Brace *) malloc (sizeof (Brace));

   if (! b)
      err_fatal_perror ("calloc",
	"brace_new:  couldn't allocate memory for `Brace' struct");

   return b;
}

// Add a new element to the head of the list
static inline void brace_extend_list (Brace **list)
{
   Brace *b = brace_new ();
   
   (b -> next) = *list;
   *list = b;
}

static void brace_delete_list (Brace *list)
{
   while (list)
   {
      Brace *next = (list -> next);
      free (list);
      list = next;
   }
}

// Delete the brace from the head of the list
static inline void brace_delete (Brace **list)
{
   Brace *next = ((*list) -> next);

   free (*list);
   (*list) = next;
}

static void brace_add (Brace **list, char brace)
{
   brace_extend_list (list);
   ((*list) -> brace) = brace;
}

// Try to complete the user input.  This function makes some smart things (like
// braces completion in the arithmetical expressions), but it looks as it have
// a good bit of the mess in the code.
static char **completion_list (const char *text, int start, int end)
{
   if (start)	// Try to analyze what we have so far
   {
      
      const char *p = rl_line_buffer + end,
      			*first = NULL,	// First word in the expression
      			*last = NULL;	// Last word in the expression
      int word = 0;	// If we are currently at the word making pass backward
      int single = 1;	// If we have any word, it is single

      Brace *braces = NULL; 	// This is a head of the list being created
      		// during the pass in the backward direction, so it acts like
      		// a stack.
      int braces_depth = 0;	// How many braces we have so far
      
      while (end > 0)	// Make single pass in the backward direction
      {
         p --;
	 end --;

	 if (! (! (*p) || isspace (*p)))
	 {
	    word = 1;

	    if (*p == ')' || *p == ']')
	    {
	       brace_add (&braces, *p);
	       braces_depth ++;
	    }
	    else if (*p == '(' || *p == '[')
	    {
	       if (braces)
	       {
		  if ((braces -> brace) == ')' || (braces -> brace) == ']')
		  {
		     if ((*p == '(' && (braces -> brace) != ')') ||
				(*p == '[' && (braces -> brace) != ']'))
		     {
		        rl_message ("Braces do not match");
		     
			brace_delete_list (braces);
		     
		        return rl_completion_matches (text, empty_generator);
		     }
		     else
		     {
			brace_delete (&braces);
		        braces_depth --;
		     }
		  }
		  else
		  {
		     brace_add (&braces, *p); 
		     braces_depth ++;
		  }
	       }
	       else
	       {
		  brace_add (&braces, *p);
		  braces_depth ++;
	       }
	    }
	 }
	 else if (word)
	 {
	    if (!last)
	       last = p + 1;
	    else
	    {
	       first = p + 1;
	       single = 0;
	    }
	 }
      }

      if ((word && !first) || !isspace (*p))
         first = p;
     
      if (first && !last)
	 last = first;

      // Complete braces in the expression if any
      if (first && braces)
         if (! strncmp (first, "p", strlen ("p")) ||
		! strncmp (first, "di", strlen ("di")) ||
		! strncmp (first, "ca", strlen ("ca")))
	 {
	    char *s = (char *) malloc ((braces_depth + 
				    	strlen (text) + 1) * sizeof (char));
	    if (!s)
	       err_fatal_perror ("malloc", 
		       "completion_list:  couldn't allocate memory for "
			       "braces");

	    strcpy (s, text);
	    {
	       char *c = s + braces_depth + strlen (text);
	       
	       *c = '\0';
	       while (braces)
	       {
		  c --;
		  if ((braces -> brace) == '(')
		     *c = ')';
		  else if ((braces -> brace) == '[')
		     *c = ']';
		  else if ((braces -> brace) == ')' ||
				(braces -> brace) == ']')
		  {
		     rl_message ("`%c' without `%s%s'", (braces -> brace),
				   ((braces -> brace) == ')') ? "(" : "",
				   ((braces -> brace) == ']') ? "[" : "");
		     brace_delete_list (braces);
		     return rl_completion_matches (text, empty_generator);
		  }
				
		  brace_delete (&braces);
	       }
	    }

	    braces_generator_ctl (s, -1);

	    return rl_completion_matches (text, braces_generator);
	    // The allocated string `s' should not be freed here
	 }
      
      if (last)
      {
         if (! strncmp (last, "state", strlen ("state")))
	    return rl_completion_matches (text, command_state_generator);
	 else
	    return NULL;
      }
   }
   
   return rl_completion_matches (text, command_generator);
}

static RAM_Debug *current_rd = NULL;

static jmp_buf discard_readline;

static void interrupt_handler (int n)
{
   if (!current_rd)
   {
      fprintf (stderr, "\n");
      err_programming ("sighal %d caught by debugger's interrupt handler,\n"
		      "but there is no active debugging session.", n);
      rl_on_new_line ();
      rl_redisplay ();
      return;
   }

   fprintf (stderr, "\n");
   err_debug ("Signal %d caught.", n);

   if (current_rd -> is_running)
   {
      (current_rd -> is_stopped) = 1;
      (current_rd -> interrupt_reason) = RAM_DEBUG_INTERRUPT;
   }

   {
      sigset_t set;

      sigemptyset (&set);
      sigaddset (&set, SIGINT);
      sigprocmask (SIG_UNBLOCK, &set, NULL);
   }

   if (!(current_rd -> is_running) ||
		((current_rd -> is_running) && (current_rd -> is_stopped)))
      longjmp (discard_readline, 1);
}

static inline int string_is_empty (const char *s)
{
   while (*s)
   {
      if (!isspace (*s))
	 return 0;

      s ++;
   }

   return 1;
}

// rm can be NULL.
// This function always returns non-zero value.
int ram_debug (RAM_Debug *rd)
{
   void (*old_interrupt_handler)(int);

   print_prompt ();
  
   rl_readline_name = "RAMEmu";
   rl_attempted_completion_function = completion_list;
	   
   using_history ();

   if (history_file_name && history_max >= 0)
   {
      err_debug ("Reading the command history from `%s'.",
		      history_file_name);
      
      if (read_history (history_file_name))
	 err_error_perror ("read_history",
		"cannot not read history file `%s'.", history_file_name);
   }
   else
      err_debug ("No command history will be used.");

   if (history_max != 0)
      stifle_history ((history_max > 0) ? history_max : 0);
   else
      unstifle_history ();

   current_rd = rd;
   old_interrupt_handler = signal (SIGINT, interrupt_handler);
   {
      sigset_t set;

      sigemptyset (&set);
      sigaddset (&set, SIGINT);
      sigprocmask (SIG_UNBLOCK, &set, NULL);
   }

   while (1)
   {
      const char *DELIMITERS = " \t\n";
      
      char *line, *token;
      const DebuggerCommand *dc;

      int is_repeated = 0;

      line = NULL;
      while (!line)
	 if (setjmp (discard_readline))
	    line = NULL;
         else
            line = readline ("RAM> ");
      if (!string_is_empty (line))
      {
	 HIST_ENTRY *r = previous_history ();
	 int add = 0;
	 if (r)
	    add = strcmp (line, (r -> line));
	 else
	    add = 1;

	 if (add)
            add_history (line);
      }
      
      {
	 char *line_preserved = strdup (line);
      
         token = strtok (line, DELIMITERS);
         if (token)
	    if (!strcmp (token, ""))
	       token = NULL;
      
         if (!token && (rd -> previous_command))
         {
	    token = strtok ((rd -> previous_command), DELIMITERS);
	    free (line_preserved);
	    is_repeated = 1;
         }
         else if (!token)
         {
	    free (line);
	    free (line_preserved);
	    continue;
         }
         else
         {
	    no_repetition (rd);
	    (rd -> previous_command) = line_preserved;
         }
      }
      
      if (*token == '!')
      {
	 char *command = (rd -> previous_command); // Which is current for now.
	 
	 while (*command != '!')
	    command ++;
	 command ++;
	 
	 system (command);

	 no_repetition (rd);
	 continue;
      }
      
      dc = command;
      while ((dc -> name) && !is_the_command (dc, token))
         dc ++;
      
      if ((dc -> name) && is_the_command (dc, token))
      {
         int return_value;
	 char *parameter = NULL;

	 parameter = strtok (NULL, "");	// The rest of the string.

	 if (parameter)
	 { // Kill delimiters at the end
	    char *c = parameter + strlen (parameter);
	    while (c > parameter)
	    {
	       c --;
	       if (isspace (*c))
		  *c = '\0';
	       else
		  break;
	    }

	    if (*parameter == '\0')
	       parameter = NULL;
	 }
	 
	 return_value = (dc -> handler) (rd, is_repeated, parameter);

	 if (return_value)
	 {
	    if (history_file_name && history_max >= 0)
	    {
	       err_debug ("Writing the command history to `%s'.",
				  history_file_name);
	       
	       if (write_history (history_file_name))
		  err_error_perror ("write_history",
			"could not properly write the history file `%s'.",
			history_file_name);
	    }
	    
	    no_repetition (rd);
	    free (line);
	   
	    signal (SIGINT, old_interrupt_handler);
	    current_rd = NULL;

            return return_value;
	 }
      }
      else
      {
         message ("Unknown command or command abbreviation.\n"
			 "Try 'h' or 'help'.");
	 no_repetition (rd);
      }
   }

   return 0;
}
