/*
 * 	Header for the error processing module.
 *
 * 	Copyright (C) 2003  Dmitry Rutsky	<rutsky@school.ioffe.rssi.ru>
 * 	
 * 	This program is free software; you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation; either version 2 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program; if not, write to the 
 * 	Free Software Foundation, Inc.,
 * 	59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifndef __error_h
#define __error_h

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>

#include "verbosity.h"

typedef Verbosity ErrorLevel;

// These are error levels for corresponding verbosity levels.
const ErrorLevel ERROR_FATAL;
const ErrorLevel ERROR_ERROR;
const ErrorLevel ERROR_IMPORTANT_WARNING;
const ErrorLevel ERROR_WARNING;
const ErrorLevel ERROR_DEBUG;
const ErrorLevel ERROR_EXTENSIVE_DEBUG;

const ErrorLevel ERROR_NONE;

const ErrorLevel ERROR_PROGRAMMING;

const ErrorLevel ERROR_MESSAGE;	// Just message to stdout.  Better than printf,
				// because it wraps long lines.

// The `severity' corresponds the verbosity level value, however, to make the
// code more natural, it is opposite to corresponding verbosity level.
// E.g. you have to make "verbosity_is_enough (-severity)" check to determine if
// this error is to be reported.
//
// ERROR_PROGRAMMING value indicates programming error, that is something
// equivalent to "fixme" and requires special treatment.
typedef void (*ErrorHandler)(const ErrorLevel severity, const char *error,
							va_list args);

extern ErrorHandler err_handler;

inline void err_report (const ErrorLevel severity, const char *error, ...);
inline void err_report_perror (const ErrorLevel severity,
				const char *caller, const char *error, ...);

inline void err_error (const char *, ...);
inline void err_fatal (const char *, ...);
inline void err_warning (const char *, ...);
inline void err_important_warning (const char *, ...);
inline void err_debug (const char *, ...);
inline void err_extensive_debug (const char *, ...);
inline void err_programming (const char *, ...);
inline void err_message (const char *, ...);

inline void err_error_perror (const char *caller, const char *error, ...);
inline void err_fatal_perror (const char *caller, const char *error, ...);
inline void err_warning_perror (const char *caller, const char *error, ...);
inline void err_debug_perror (const char *caller, const char *error, ...);

// NOTE:  the above functions assume that the err_handler points to a
// correct handler.  It is set to stderr reporting function by default.
ErrorHandler err_install_handler (ErrorHandler);

typedef int (*ErrorSNPrintFHandler)(char *, size_t, const char *, ...);
typedef int (*ErrorVSNPrintFHandler)(char *, size_t, const char *, va_list);

// If you wish to use default error handler with generic printf-family
// functions, register them with the following calls before any possible err_
// calls.
ErrorSNPrintFHandler err_use_snprintf_handler (ErrorSNPrintFHandler);
ErrorVSNPrintFHandler err_use_vsnprintf_handler (ErrorVSNPrintFHandler);

// This variable is maintained by the default error handler to have
// highest severity of any error met, custom handler may do it as well.
// It is initialized with ERROR_NONE, and does not count programming errors.
extern ErrorLevel err_highest_met_severity;

// Use this routine in your custom error handlers to update the previous
// variable.
ErrorLevel err_update_highest_met_severity (const ErrorLevel severity);

int err_set_output_width (int);

#endif /* __error_h */
