/*
 * 	Random Access Machine.
 * 	Command-line interface.
 *
 * 	Copyright (C) 2002, 2003  Dmitry Rutsky	<rutsky@school.ioffe.rssi.ru>
 * 	
 * 	This program is free software; you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation; either version 2 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program; if not, write to the 
 * 	Free Software Foundation, Inc.,
 * 	59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <signal.h>
#include <setjmp.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <getopt.h>
#include <gmp.h>

#include "ram.h"
#include "configuration.h"
#include "ram_config.h"
#include "legal_attributes.h"

#if BUILD_DEBUGGER
#include "debugger.h"
#endif

static char *current_source = NULL;
static RAM *current_machine = NULL;

static jmp_buf machine_interrupted;

static void set_interrupt_handler ();
static void unset_interrupt_handler ();

static void display_status ()
{
   if (current_machine)
   {
      if (!ram_is_running (current_machine))
	 err_debug ("The machine had halted at position %d.",
			 (current_machine -> current_instruction) + 1);

      err_message ("The machine had performed %Zd instruction%s "
			"taken %Zd time unit%s.",
		(current_machine -> instructions_done),
		(mpz_cmp_ui ((current_machine -> instructions_done), 1) > 0)
			? "s, they have" : ", it has",
		(current_machine -> time_consumed),
		(mpz_cmp_ui ((current_machine -> time_consumed), 1) > 0)
			? "s" : ""
			);
   }
   else
      err_debug ("No machine created yet.");
}

// This function is called upon keyboard interrupt or similar signal.
static void interrupt (int n)
{
   fflush (stdout);

   err_debug ("Signal %d caught, interrupt.", n);

   if (!current_machine)
   {
      err_debug ("No machine is running, ignoring interrupt.");
      return;
   }

   {
      int c = 0, again = 0;
      do
      {
	 if (again)
	    err_message (
		"Please enter single letter corresponding requested action.");

         if (current_machine)
            printf (
      "Select preferable action:  i)gnore, q)uit, or run builtin d)ebugger:  ");
         else
            err_message ("Select preferable action:  i)gnore or q)uit:  ");

         c = getchar ();
	 {
	    int d = getchar ();
	    if (d != '\n')
	    {
	       while (d != '\n')
		  d = getchar ();

	       c = 1;
	    }
	 }

	 again ++;
      }
      while (!strchr (current_machine ? "IiQqDd" : "IiQq", c));

      switch (c)
      {
      case 'I':
      case 'i':
	 return;
      case 'Q':
      case 'q':
	 exit (0);
	 break;
      case 'D':
      case 'd':
      {
	 RAM_Text *text = ram_text_new ();
	 FILE *f = fopen (current_source, "r");

	 if (!f)
	 {
	    err_error_perror ("fopen", "Couldn't open `%s'.", current_source);
	    return;
	 }

	 ram_program_delete (current_machine -> program);
	 (current_machine -> program) = ram_program_parse (f, text);

	 fclose (f);

	 unset_interrupt_handler ();

	 if (! (current_machine -> program))
	    exit (1);

	 err_debug ("Entering debugging session for `%s'...",
			 current_source);
	 ram_debug_by (current_machine, (current_machine -> input),
			 (current_machine -> output), text);
	
	 ram_text_delete (text);

	 set_interrupt_handler ();

	 longjmp (machine_interrupted, 1);
      }
      }

      return;
   }
   
   display_status ();
}

static void (*old_interrupt_handler) (int) = NULL;
static int handler_is_set = 0;

static void set_interrupt_handler ()
{
   if (old_interrupt_handler)
      err_programming ("set_interrupt_handler:  handler set already.");
   else
   {
      old_interrupt_handler = signal (SIGINT, interrupt);

      handler_is_set = 1;
   }
}

static void unset_interrupt_handler ()
{
   if (! handler_is_set)
      err_programming ("unset_interrupt_handler:  handler was not set.");
   else
   {
      signal (SIGINT, old_interrupt_handler);
      old_interrupt_handler = NULL;

      handler_is_set = 0;
   }
}

static void copyright ()
{
   err_message (COPYRIGHT);
}

static void warranty ()
{
   err_message (WARRANTY);
}

static void help (const char *program_name)
{
   err_message (
"Usage:  ramemu [ list of options ] [ list of RAM programs ]\n\n"
"Perform specified actions with given programs (default is to simulate).\n\n"
"Options:\n"
"  -h, --help                  print help message (this one)\n"
"  -C, --copyright             print copyright notice\n"
"  -W, --warranty              print warranty notice\n"
"  -V, --version               print version number\n"
"  -v, --verbose               increase verbosity level, cumulative\n"
"  -q, --quiet                 decrease verbosity level, also cumulative\n"
"  -i, --input [FILE]          use [FILE] as input of emulated programs\n"
"  -o, --output [FILE]         use [FILE] as output of emulated programs\n"
"      --home [DIR]            use [DIR] as emulator's home directory\n"
#if BUILD_DEBUGGER
"  -d, --debugger              run as command-line debugger\n"
"      --history [FILE]        use [FILE] as debugger command history\n"
#endif
"      --configuration [FILE]  read configuration options from file [FILE]\n"
"                              (options before this one may be overridden)\n"
"  -p, --prompts               print prompts for READ/WRITE instructions\n"
"      --no-prompts            disable prompts for READ/WRITE instructions\n"
"  -B, --segment-size [N]      specify default memory segment size to be used\n"
"                              in RAM memory emulation, measured in registers\n"
"                              (default is 4)\n\n"
"If no `--home' option specified, RAMEMU_HOME environment variable is used as "
"home directory name.  If this variable is not set, the path defaults to "
"$HOME/.ramemu;  if HOME variable is not set, no home directory will be used."
"\n\n"
"Report bugs to <" PACKAGE_BUGREPORT ">.");
}

static void version ()
{
   err_message (PACKAGE_VERSION);
}

// Find last `option' option in the command line parameters and return its
// argument.  If no `option' option found, return NULL.  NULL is also returned
// in case no argument found or bad option string.  No error messages are
// produced in this function.
static char *locate_option (int argc, char **argv,
				const char *option)
{
   char *prev = NULL, *option_found = NULL;

   argv ++;

   while (argc > 1)
   {
      if (!strncmp (*argv, option, strlen (option)))
      {
	 if (strlen (*argv) > strlen (option))
	 {
	    if ((*argv) [strlen (option)] != '=' || !(*argv) [strlen (option)])
	       return NULL;

	    option_found = (*argv) + strlen (option) + 1;
	 }
	 else if ((*prev) != '-')
	    option_found = prev;
	 else
	    return NULL;
      }

      prev = *argv;
      argv ++;
      argc --;
   }

   return option_found;
}

// Scan options array and apply every verbosity level modification option met.
static void parse_verbosity_options (int argc, char **argv)
{
   argv ++;

   while (argc > 1)
   {
      const char *verbose = "--verbose", *quiet = "--quiet";

      if (!strcmp (*argv, verbose))
         verbosity_increase ();
      else if (!strcmp (*argv, quiet))
	 verbosity_decrease ();
      else if (**argv == '-')
      {
	 char *p = *argv + 1;

	 while (isalpha (*p))
	 {
	    if (*p == 'v')
	       verbosity_increase ();
	    else if (*p == 'q')
	       verbosity_decrease ();

	    p ++;
	 }
      }

      argv ++;
      argc --;
   }
}

// Try to open and read configuration from `directory'/`name'.  Return status
// of the operation.  Don't produce any error messages if failed to open the
// configuration.
static int try_to_read_config (const char *directory, const char *name)
{
   char *n = (char *) alloca ((strlen (directory) + strlen (name) + 2) *
		   					sizeof (char));
   if (!n)
      err_fatal ("couldn't allocate memory for a configuration filename.");

   sprintf (n, "%s/%s", directory, name);

   return ram_read_config (n);
}

int main (int argc, char **argv)
{
   int option;
   FILE *input = stdin, *output = stdout;
   char *home = NULL;	// Allocated.
   
#if BUILD_DEBUGGER
   int debugger = 0;
   char *history_name = NULL;
#endif

   int blank_run = 1;

   const struct option options [] =
   {
      {"help", 0, NULL, 'h'},
      {"copyright", 0, NULL, 'C'},
      {"warranty", 0, NULL, 'W'},
      {"version", 0, NULL, 'V'},
      {"verbose", 0, NULL, 'v'},
      {"quiet", 0, NULL, 'q'},
      {"input", 1, NULL, 'i'},
      {"output", 1, NULL, 'o'},
      {"home", 1, NULL, '\04'},
      {"debugger", 0, NULL, 'd'},
      {"prompts", 0, NULL, 'p'},
      {"no-prompts", 0, NULL, '\02'},
      {"configuration", 1, NULL, '\03'},
      {"history", 1, NULL, '\01'},
      {"segment-size", 1, NULL, 'B'},
      {NULL, 0, NULL, 0}
   };
 
   parse_verbosity_options (argc, argv);

   err_use_snprintf_handler (gmp_snprintf);
   err_use_vsnprintf_handler (gmp_vsnprintf);

   home = locate_option (argc, argv, "--home");
   if (home)
   {
      home = strdup (home);
      if (!home)
	 err_fatal_perror ("strdup",
		"couldn't duplicate `--home' option string `%s'.", home);
   }
   else
   {
      char *c_home = getenv ("HOME"), *r_home = getenv ("RAMEMU_HOME");

      if (!c_home && !r_home)
         err_warning ("neither HOME, nor RAMEMU_HOME environment variables "
		"set.  No configuration will be read.");

      if (r_home)
      {
	 home = strdup (r_home);
	 if (!home)
            err_fatal_perror ("strdup", "couldn't duplicate string");
      }
      else if (c_home)
      {
	 const char *suffix = ".ramemu";

	 home = (char *) malloc ((strlen (c_home) + strlen (suffix) + 2) *
			 			sizeof (char));
	 if (!home)
	    err_fatal_perror ("malloc", "couldn't allocate memory for "
			    "home directory name string");

	 strcpy (home, c_home);
	 strcat (home, "/");
	 strcat (home, suffix);
      }

      if (home)
      {
	 DIR *test = opendir (home);

	 if (!test)
	 {
	    if (errno == ENOENT)
	    {
	       if (mkdir (home, 0700))
	       {
		  err_error_perror ("mkdir",
			"couldn't create home directory `%s'.", home);

		  free (home);
		  home = NULL;
	       }
	       else
		  err_debug ("New home directory `%s' was created.", home);
	    }
	    else
	    {
	       err_error_perror
			("opendir", "couldn't open home directory `%s'.", home);

	       free (home);
	       home = NULL;
	    }
	 }
	 else
	    closedir (test);
      }
   }

   if (home)
      err_debug ("Using `%s' as home directory.", home);
   else
      err_debug ("No home directory found or specified.");

   if (home)
   {
      const char *config_name = "ramemu.config";

      if (!try_to_read_config (home, config_name))
	 err_debug ("There were errors when tried to use configuration "
			 "stored in `%s/%s'.", home,
			 				config_name);
   }

   opterr = 0;
   
   while ((option = getopt_long (argc, argv, "hCWVvqi:o:dp\02\03:\01B:",
				   options, NULL))
		   != -1)
   {
      switch (option)
      {
      case 'h':
	 if (!blank_run)
	    printf ("\n");
	 help (*argv);
	 blank_run = 0;
	 break;
      case 'C':
	 if (!blank_run)
	    printf ("\n");
	 copyright ();
	 blank_run = 0;
	 break;
      case 'W':
	 if (!blank_run)
	    printf ("\n");
	 warranty ();
	 blank_run = 0;
	 break;
      case 'V':
	 if (!blank_run)
	    printf ("\n");
	 version ();
	 blank_run = 0;
	 break;
      case 'v':
      case 'q':
	 break;
      case 'i':
	 if (input != stdin)
	 {
	    if (input)
	    {
               err_important_warning ("multiple '--input' options");
	       fclose (input);
	    }
	 }
	 
	 input = fopen (optarg, "r");
	 if (! input)
	    err_error_perror ("fopen", "couldn't open `%s' for reading",
			    			optarg);
	 
	 break;
      case 'o':
	 if (output != stdout)
	 {
	    if (output)
	    {
	       err_warning ("multiple '--output' options");
	       fclose (output);
	    }
	 }

	 output = fopen (optarg, "w");
	 if (! output)
            err_error_perror ("fopen", "couldn't open `%s' for writing",
			    					optarg);
	 break;
      case '\04':
	 {
	    static int was = 0;

	    if (was == 1)
	    {
	       if (home)
	          err_important_warning (
		   "multiple `--home' options, last one (`%s') was used.",
		   		home);
	       else
		  err_important_warning
			  ("multiple `--home' options, last one is bad");
	    }

	    was ++;
	 }
	 break;
#if BUILD_DEBUGGER
      case 'd':
	 debugger = 1;
	 break;
      case '\01':
	 ram_debug_set_history_file (optarg);
	 break;
#else
      case 'd':
      case '\01':
	 err_error (
"debugger-related option `%c' found, but built-in debugger was\n"
"disabled at compilation time.", option);
	 break;
#endif
      case 'p':
	 ram_enable_IO_prompts ();
	 break;
      case '\02':
	 ram_disable_IO_prompts ();
	 break;
      case '\03':
	 {
	    int have = 0;

	    if (ram_read_config (optarg) == -12)
	    {
	       if (home)
		  if (try_to_read_config (home, optarg) != -12)
		     have = 1;
	    }
	    else
	       have = 1;

	    if (!have)
	       err_error ("couldn't open `%s' or `%s/%s' as configuration file "
			       "for reading.", optarg, home, optarg);
	 }
	 break;
      case 'B':
	 {
	    int size;
	    if (sscanf (optarg, "%d", &size) != 1)
	       err_error ("bad parameter `%s' for --segment-size.", optarg);
	    else if (size <= 0)
	       err_error ("bad segment size %d.", size);
	    else
	    {
	       err_debug ("Setting segment size to %d.", size);
	       ram_set_block_size (size);
	    }
	 }
	 break;
      case ':':
	 err_error ("option `%s' requires an argument.  Try `--help'.",
			 argv [optind - 1]);
	 break;
      case '?':
	 err_error ("`%s':  unrecognized option.  Try `--help'.",
			 argv [optind - 1]);
	 break;
      }
   }

   if (err_highest_met_severity >= ERROR_ERROR)
      return err_highest_met_severity;

#if BUILD_DEBUGGER
   if (debugger && !history_name && home)
   {
      const char *default_history = "debugger.history";
      

      history_name = (char *) malloc
	      ((strlen (home) + strlen (default_history) + 2) * sizeof (char));
      if (!history_name)
	 err_fatal_perror ("malloc",
		"couldn't allocate memory for history filename.");

      strcpy (history_name, home);
      strcat (history_name, "/");
      strcat (history_name, default_history);

      ram_debug_set_history_file (history_name);
   }
#endif
   
   if (optind == argc)
   {
#if BUILD_DEBUGGER
      if (debugger)
	 ram_debug_by (NULL, input, output, NULL);
      else
#endif
      if (blank_run)
         err_important_warning ("no programs given.");
   }

   while (optind < argc)
   {
      FILE *program_file;
      RAM_Program *program;
      RAM *machine;
      RAM_Text *text = NULL;
    
      if (!blank_run)
	 err_message ("");

      program_file = fopen (argv [optind], "r");
      if (! program_file)
      {
	 err_error_perror ("fopen",
			 "couldn't open `%s' for reading.", argv [optind]);
	 optind ++;
	 continue;
      }

#if BUILD_DEBUGGER
      if (debugger)
	 text = ram_text_new ();
#endif
      
      program = ram_program_parse (program_file, text);
      if (! program)
         exit (1);
     
      fclose (program_file);

      machine = ram_new_by_program (program);
      (machine -> input) = input;
      (machine -> output) = output;

#if BUILD_DEBUGGER
      current_source = argv [optind];
#endif
      current_machine = machine;

#if BUILD_DEBUGGER
      if (debugger)
      {
	 err_debug ("Starting debugger session for %s...", 
			 		argv [optind]);
	 ram_debug_by (machine, input, output, text);

	 ram_text_delete (text);
      }
      else
#endif
      {
	 err_debug ("Starting `%s'...", argv [optind]);

	 set_interrupt_handler ();

	 if (! setjmp (machine_interrupted))
	 {
            while (ram_do_instruction (machine));

	    display_status ();
	 }
	 else
	 {
	    sigset_t mask;
	    
	    sigemptyset (&mask);
	    sigaddset (&mask, SIGINT);

	    sigprocmask (SIG_UNBLOCK, &mask, NULL);
	 }

	 unset_interrupt_handler ();
      }

      current_machine = NULL;
#if BUILD_DEBUGGER
      current_source = NULL;
#endif

      ram_delete (machine);
      
      optind ++;
   }

   if (input != stdin)
      fclose (input);

   if (output != stdout)
      fclose (output);
  
#if BUILD_DEBUGGER
   if (debugger && history_name)
      free (history_name);
#endif

   if (home)
      free (home);

   return 0;
}
