/*
 * 	Random Access Machine.
 * 	RAM configuration parser module.
 *
 * 	Copyright (C) 2003  Dmitry Rutsky	<rutsky@school.ioffe.rssi.ru>
 * 	
 * 	This program is free software; you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation; either version 2 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program; if not, write to the 
 * 	Free Software Foundation, Inc.,
 * 	59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include "config.h"

#include "ram_config.h"
#include "error.h"

#if BUILD_DEBUGGER
#include "debugger.h"
#endif

static int match_version (const char *option, const char *value)
{
   int v1, v2, v3, o1, o2, o3;

   if (sscanf (value, "%d.%d.%d", &v1, &v2, &v3) != 3)
   {
      err_error ("invalid `%s' version option value.", value);
      return 0;
   }

   if (sscanf (PACKAGE_VERSION, "%d.%d.%d", &o1, &o2, &o3) != 3)
   {
      err_programming ("invalid `%s' PACKAGE_VERSION value.", PACKAGE_VERSION);
      return 0;
   }

   if (v1 > o1 || ((v1 == o1) && (v2 > o2)) ||
		   ((v1 == o1) && (v2 == o2) && (v3 > o3)))
   {
      err_important_warning ("configuration file version number is greater "
		      "than the emulator's version number (%s > %s).",
		      value, PACKAGE_VERSION);
   }
   else
      err_debug ("Configuration file version %s matches %s of ours.", value,
		   			PACKAGE_VERSION);

   return 1;
}

#if BUILD_DEBUGGER
static int set_history_max (const char *option, const char *value)
{
   int m;

   if (sscanf (value, "%d", &m) != 1)
   {
      err_error ("Invalid value `%s' for option `%s'.", value, option);
      return 0;
   }

   ram_debug_set_history_max (m);

   if (m > 0)
      err_debug ("The debugger history was limited to %d line%s.", m,
		      				(m > 1) ? "s" : "");
   else if (m < 0)
      err_debug ("No history file will be written.");
   else
      err_debug ("The debugger history is set unlimited.");

   return 1;
}
#else
static int debugger_option (const char *option, const char *value)
{
   err_debug ("Ignoring debugger's option `%s'.", option);

   return 1;
}
#endif

typedef int (*OptionHandler)(const char *option, const char *value);

typedef struct
{
   const char *name;
   OptionHandler handler;
}
Option;

static Option options [] =
{
   {"version", match_version},
#if BUILD_DEBUGGER
   {"debugger_history_max", set_history_max},
#else
   {"debugger_history_max", debugger_option},
#endif
   {NULL, NULL}
};

// Parse the configuration.  Return the operation's status.
int ram_config_parse (Conf_Option *configuration)
{
   int status = 1;

   while (configuration)
   {
      Option *o = options;
      while (o -> name)
      {
         if (!strcasecmp ((configuration -> name), (o -> name)))
	 {
            status = (o -> handler) ((o -> name), (configuration -> value)) ?
		    					status : 0;
	    break;
	 }

	 o ++;
      }

      if (!(o -> name))
      {
	 err_error ("unknown option `%s'", (configuration -> name));
	 status = 0;
      }
	      
      configuration = (configuration -> next);
   }

   return status;
}

// Read configuration from file `filename'.  Return the operation status:  -12
// if we couldn't open the file, 0 if other error, other value if there were no
// errors.
int ram_read_config (const char *filename)
{
   FILE *f;
   int status;

   err_debug ("Opening `%s'...", filename);

   f = fopen (filename, "r");
   if (!f)
   {
      err_debug_perror ("fopen", "Couldn't open `%s' for reading.",
			 filename);

      return -12;
   }
   
   err_debug ("Reading configuration from `%s'...", filename);

   {
      Conf_Option *c = conf_read (f);
   
      fclose (f);

      if (!c)
         return 0;

      status = ram_config_parse (c);

      conf_delete (c);
   }

   return status;
}
