-------------------------------------------------------------------
--           RAPID - RAPID ADA PORTABLE INTERFACE DESIGNER
--           TCL PEER FOR THE MCC GUI PACKAGE LIBRARY
--           Copyright (C) 1999 Martin C. Carlisle.
--
-- RAPID is free software;  you can  redistribute it  and/or modify
-- it under terms of the  GNU General Public License as published
-- by the Free Software  Foundation;  either version 2,  or (at your
-- option) any later version.  RAPID is distributed in the hope that
-- it will be useful, but WITHOUT ANY WARRANTY;  without even the
-- implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
-- PURPOSE.  See the GNU General Public License for more details.
-- You should have  received  a copy of the GNU General Public License
-- distributed with RAPID; see file COPYING.  If not, write to the
-- Free Software Foundation,  59 Temple Place - Suite 330,  Boston,
-- MA 02111-1307, USA.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.  This exception does not apply to executables which
-- are GUI design tools, or that could act as a replacement
-- for RAPID.
--
-- Copyright (C) 2005, SA Griffin <stephen.griffin.external@eads.com>
------------------------------------------------------------------------------

with Text_IO;
with Ada.Exceptions;
with Ada.Unchecked_Conversion;
with Ada.Characters.Handling;  use Ada.Characters.Handling;

package body mcc.Gui.Widget.Textbox is

   Max_Text_Length : constant := 13680;

   Text : String (1 .. Max_Text_Length);

   ---------------
   -- Highlight --
   ---------------

   procedure Highlight
     (Obj   : in out Textbox;
      Start : in Natural;
      Stop  : in Natural)
   is
   begin

      peer.Eval
        (Obj.My_Peer.Name.all & " selection range " &
         mcc.Img (Start) & mcc.Img (Stop));

      peer.Eval ("focus " & Obj.My_Peer.Name.all);

   exception

      when E : others =>
         Text_IO.New_Line;
         Text_IO.Put
           (
"**** EXCEPTION **** : Mcc.Gui.Widget.Textbox.Highlight.Obj.Start.Stop " &
            Ada.Exceptions.Exception_Information (E));

   end Highlight;

   procedure Highlight (Obj : in out Textbox) is
   begin

      peer.Eval (Obj.My_Peer.Name.all & " selection range 0 end");
      peer.Eval ("focus " & Obj.My_Peer.Name.all);

   exception

      when E : others =>
         Text_IO.New_Line;
         Text_IO.Put
           ("**** EXCEPTION **** : Mcc.Gui.Widget.Textbox.Highlight.Obj " &
            Ada.Exceptions.Exception_Information (E));

   end Highlight;

   -----------
   -- Clear --
   -----------

   procedure Clear (Obj : in out Textbox) is
   begin

      peer.Eval (Obj.My_Peer.Name.all & ".t delete 1.0 end");

   exception

      when E : others =>
         Text_IO.New_Line;
         Text_IO.Put
           ("**** EXCEPTION **** : Mcc.Gui.Widget.Textbox.Clear " &
            Ada.Exceptions.Exception_Information (E));

   end Clear;

   procedure Create
     (Obj                  : in out Textbox;
      Parent               : in mcc.Gui.Container.Container'Class;
      X                    : in Integer;
      Y                    : in Integer;
      Width                : in Natural;
      Height               : in Natural;
      Horizontal_Scrollbar : in Boolean;
      Vertical_Scrollbar   : in Boolean)
   is

      type Parent_Access is access constant mcc.Gui.Container.Container'Class;
      function Convert is new Ada.Unchecked_Conversion (
         Parent_Access,
         mcc.Gui.Container.Container_Pointer);
   begin

      Obj.My_Peer := peer.Create_Peer (mcc.Gui.Object (Parent).My_Peer);

      peer.Eval ("frame " & Obj.My_Peer.Name.all);

      peer.Eval
        ("text " & Obj.My_Peer.Name.all &
         ".t -setgrid true -wrap word" &
         " -width " & mcc.Img (Width) &
         " -height " & mcc.Img (Height) &
         " -yscrollcommand """ & Obj.My_Peer.Name.all & ".sy set""");

      peer.Eval
        ("scrollbar " & Obj.My_Peer.Name.all & ".sy" &
         " -orient vertical" &
         " -command """ & Obj.My_Peer.Name.all & ".t yview""");

      peer.Eval ("pack " & Obj.My_Peer.Name.all & ".sy -side right -fill y");

      peer.Eval ("pack " & Obj.My_Peer.Name.all & ".t -side left");

      peer.Eval
        ("place " & Obj.My_Peer.Name.all &
         " -anchor nw" &
         " -x " & mcc.Img (X) &
         " -y " & mcc.Img (Y) &
         " -width " & mcc.Img (Width) &
         " -height " & mcc.Img (Height));

      peer.Eval
        ("bindtags " &
         Obj.My_Peer.Name.all &
         ".t ""[bindtags " &
         Obj.My_Peer.Name.all &
         ".t] " &
         Obj.My_Peer.Name.all &
         """");
      Obj.Parent := Convert (Parent'Unchecked_Access);

   exception

      when E : others =>
         Text_IO.New_Line;
         Text_IO.Put
           ("**** EXCEPTION **** : Mcc.Gui.Widget.Textbox.Create " &
            Ada.Exceptions.Exception_Information (E));

   end Create;

   procedure Set_Text (Obj : in out Textbox; Text : in String) is
   begin

      Clear (Obj);

      --text_io.new_line ;
      --text_io.put (" Mcc.Gui.Widget.Textbox.Set_Text : Text = " & Text ) ;

      peer.Eval
        (Obj.My_Peer.Name.all &
         ".t insert 1.0 """ &
         peer.Fix_Quotes (Text) &
         """");

   exception

      when E : others =>
         Text_IO.New_Line;
         Text_IO.Put
           ("**** EXCEPTION **** : Mcc.Gui.Widget.Textbox.Set_Text " &
            Ada.Exceptions.Exception_Information (E));

   end Set_Text;

   --------------
   -- Get Text --
   --------------

   function Get_Text (Obj : in Textbox) return String is
      Last : Natural;
   begin

      peer.Eval
        ("set entry_text [ " &
         Obj.My_Peer.Name.all &
         ".t get 1.0 {end -1c}]");

      peer.Get_Value (Name => "entry_text", Result => Text, Last => Last);

      return Text (Text'First .. Last);

   exception

      when E : others =>
         Text_IO.New_Line;
         Text_IO.Put
           ("**** EXCEPTION **** : Mcc.Gui.Widget.Textbox.Get_Text.String " &
            Ada.Exceptions.Exception_Information (E));

         return " ";

   end Get_Text;

   ---------------------
   -- Get_Text_Length --
   ---------------------

   function Get_Text_Length (Obj : in Textbox) return Natural is
      Result : String := Get_Text (Obj);

   begin

      --text_io.new_line ;
      --text_io.put ("Mcc.Gui.Widget.Textbox.Get_Text_Length_Box : Text = " &
      --Result & " : Length = " & Natural'Image ( Result'Length ) ) ;

      return Result'Length;

   exception

      when E : others =>
         Text_IO.New_Line;
         Text_IO.Put
           ("**** EXCEPTION **** : Mcc.Gui.Widget.Textbox.Get_Text_Length " &
            Ada.Exceptions.Exception_Information (E));

         return 0;

   end Get_Text_Length;

end Mcc.Gui.Widget.Textbox;
