---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  EDIT_MENU.ADB
--  Description : Implements choices from the Edit menu
--
--  Copyright (C) 1999, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with gui;                         use type gui.String_Pointer;
with gui.Widget, gui.Window;
use type gui.Widget.Widget_Access, gui.Window.Window_Pointer;
with Gui_Enum;
with state;
with Subwindow_Actions;
with change_window_dialog_window;
with mcc.Common_Dialogs;
with mcc.tki.Widget.Text_Entry;
with mcc.tki.Widget.Dropdown;
with mcc.tki.Widget.Button.Check;
with gui.Widget.Button;
with gui.Menu;
--with Common_Dialogs; -- for debugging
with Ada.Strings.Unbounded; use Ada.Strings.Unbounded;

package body Edit_Menu is
   Old_Name : Unbounded_String;
   -------------------------------------------------------------
   -- procedure Change_Window
   --
   -- used to change window's name, width or height
   --
   -- Algorithm
   -- 1) Find current window
   -- 2) Generate change dialog
   --    dialog (change_window_dialog) has 3 entries
   --      entry1: name
   --      entry2: width
   --      entry3: height
   --      entry4: title (added 2/22/99 using suggestion by Dmitri
   --                     Anisimkov)
   -- 3) Make change dialog transient (changes appearance)
   -- 4) Main window is stored as ".", but display as "main"
   --    Set text in entry1, 2 and 3
   -- 5) Callbacks implement changes, so we are done!
   -------------------------------------------------------------
   procedure Change_Window is
      Window : gui.Window.Window_Pointer := state.Get_Current_Window;
   begin
      change_window_dialog_window.Generate_Window;
      mcc.tki.Widget.Text_Entry.Highlight
        (Obj   => change_window_dialog_window.entry1,
         Start => 0,
         Stop  => 0);

      Old_Name := To_Unbounded_String (Window.Window_Name.all);

      mcc.tki.Widget.Text_Entry.Set_Text
        (Obj  => change_window_dialog_window.entry1,
         Text => Window.Window_Name.all);

      if Window.Title /= null then
         mcc.tki.Widget.Text_Entry.Set_Text
           (Obj  => change_window_dialog_window.entry4,
            Text => Window.Title.all);
      end if;

      mcc.tki.Widget.Text_Entry.Set_Text
        (Obj  => change_window_dialog_window.entry2,
         Text => mcc.Img (Window.Width));

      mcc.tki.Widget.Text_Entry.Set_Text
        (Obj  => change_window_dialog_window.entry3,
         Text => mcc.Img (Window.Height));

      mcc.tki.Widget.Dropdown.Select_Item
        (Obj    => change_window_dialog_window.Main_Win_DD,
         Number => Gui_Enum.Window_Kind_T'Pos (Window.Kind) + 1);

      if Window.Parent_Frame /= null then
         mcc.tki.Widget.Text_Entry.Set_Text
           (Obj  => change_window_dialog_window.Parent_TE,
            Text => Window.Parent_Frame.all);
      end if;

      mcc.tki.Widget.Dropdown.Select_Item
        (Obj    => change_window_dialog_window.Snaptogrid_DD,
         Number => Gui_Enum.Snaptogrid_T'Pos (Window.Snap) + 1);

      mcc.tki.Widget.Button.Check.Set_Check
        (Obj => change_window_dialog_window.novice,
         To  => Window.Novice_Mode);

   end Change_Window;

   -------------------------------------------------------------
   -- function Check_Changes
   --
   -- verify that entries have legal values
   -- highlight illegal entry (if it is negative)
   -------------------------------------------------------------
   function Check_Changes
     (Window : in gui.Window.Window_Pointer)
      return   Boolean
   is
   begin
      if Window.Width < 0 then
         mcc.tki.Widget.Text_Entry.Highlight
           (Obj => change_window_dialog_window.entry2);
         return False;
      elsif Window.Height < 0 then
         mcc.tki.Widget.Text_Entry.Highlight
           (Obj => change_window_dialog_window.entry3);
         return False;
      else
         return True;
      end if;
   end Check_Changes;

   -------------------------------------------------------------
   -- procedure Set_Changes/Set_Name
   --
   -- put values from entries in Window record
   --
   -- These used to be one procedure, but we don't want to reset
   -- name until after we have undisplayed the old window
   --
   -- Algorithm
   -- Set_Name
   -- 1) Read entry1 into Word
   -- 2) Put Word into Window.Name, replacing "." with "_"
   --    (Ada doesn't like . in its names)
   -- Set_Changes
   -- 3) Read entry2 directly into Window.Width
   -- 4) Read entry3 directly into Window.Height
   -- 5) Read entry4 into Word, then into Window.Title (added 2/22/99)
   -------------------------------------------------------------
   procedure Set_Name (Window : in gui.Window.Window_Pointer) is
      Word : String :=
         mcc.tki.Widget.Text_Entry.Get_Text
           (change_window_dialog_window.entry1);
   begin

      if Word'Last > Word'First then
         Window.Window_Name := new String'(Word);
      else
         Window.Window_Name := new String'("Default");
      end if;
   end Set_Name;

   procedure Set_Changes (Window : in gui.Window.Window_Pointer) is
      package TE renames mcc.tki.Widget.Text_Entry;
      Title  : String := TE.Get_Text (change_window_dialog_window.entry4);
      Parent : String := TE.Get_Text (change_window_dialog_window.Parent_TE);
      Parent_Last : Natural := 0;
   begin

      if Title'Last > Title'First then
         Window.Title := new String'(Title);
      else
         mcc.Common_Dialogs.Ok_Box ("Bad title");
      end if;

      begin
         Window.Width :=
            mcc.tki.Widget.Text_Entry.Get_Text
              (change_window_dialog_window.entry2);
      exception
         when others =>
            mcc.Common_Dialogs.Ok_Box ("Bad width");
      end;

      begin
         Window.Height :=
            mcc.tki.Widget.Text_Entry.Get_Text
              (change_window_dialog_window.entry3);
      exception
         when others =>
            mcc.Common_Dialogs.Ok_Box ("Bad height");
      end;

      begin
         Window.Kind :=
            Gui_Enum.Window_Kind_T'Val
              (mcc.tki.Widget.Dropdown.Get_Selected
                 (change_window_dialog_window.Main_Win_DD) - 1);
      exception
         when others =>
            mcc.Common_Dialogs.Ok_Box ("Bad window kind");
      end;

      -- eliminate trailing spaces
      if Parent /= "" then
         Parent_Last := Parent'Last;
         while Parent_Last > 0 and then Parent (Parent_Last) = ' ' loop
            Parent_Last := Parent_Last - 1;
         end loop;
         if Parent_Last > 0 then
            if Window.Parent_Frame = null or else
              Window.Parent_Frame.all /= Parent (1 .. Parent_Last) then
               Window.Parent_Frame := new String' (Parent (1 .. Parent_Last));
            end if;
         else
            Window.Parent_Frame := null;
         end if;
      end if;

      begin
         Window.Snap :=
            Gui_Enum.Snaptogrid_T'Val
              (mcc.tki.Widget.Dropdown.Get_Selected
                 (change_window_dialog_window.Snaptogrid_DD) - 1);
      exception
         when others =>
            mcc.Common_Dialogs.Ok_Box ("Bad snap to grid");
      end;

      Window.Novice_Mode :=
         mcc.tki.Widget.Button.Check.Is_Checked
           (change_window_dialog_window.novice);
   end Set_Changes;

   -------------------------------------------------------------
   -- procedure Change_Done
   --
   -- Change is Done (hit ok)-- update display
   --
   -- Algorithm
   -- 1) Apply changes -- copy of below b/c we don't
   --    wan't to destroy window if changes not ok.
   --    Reset name between undisplay and display
   -- 2) Destroy dialog only if changes ok
   -------------------------------------------------------------
   procedure Change_Done is
      Window : gui.Window.Window_Pointer := state.Get_Current_Window;
   begin
      Set_Changes (Window);
      if Check_Changes (Window) then
         gui.Window.Undisplay_Window (Window.all);
         Set_Name (Window);
         gui.Window.Display_Window (Window.all);
         mcc.tki.Destroy
           (mcc.tki.Object (
           change_window_dialog_window.change_window_dialog_window));
         if Window.Window_Name.all /= To_String (Old_Name) then
            gui.Menu.Update_Actions
              (Window.Menu,
               To_String (Old_Name),
               Window.Window_Name.all);
            declare
               Current_Widget_Position :
                 gui.Widget.Widget_List_Package.Position;
               Found                   : Boolean := False;
               Widget_Ptr              : gui.Widget.Widget_Access;
            begin
               Current_Widget_Position :=
                  gui.Widget.Widget_List_Package.First
                    (L => Window.Widget_List);
               while not gui.Widget.Widget_List_Package.IsPastEnd
                           (L => Window.Widget_List,
                            P => Current_Widget_Position)
               loop
                  Widget_Ptr :=
                     gui.Widget.Widget_List_Package.Retrieve
                       (L => Window.Widget_List,
                        P => Current_Widget_Position);
                  if Widget_Ptr.all in gui.Widget.Button.Button'Class then
                     gui.Widget.Button.Update_Action
                       (gui.Widget.Button.Button'Class (Widget_Ptr.all),
                        To_String (Old_Name),
                        Window.Window_Name.all);
                  end if;
                  gui.Widget.Widget_List_Package.GoAhead
                    (L => Window.Widget_List,
                     P => Current_Widget_Position);
               end loop;
            end;
         end if; -- window name changed
      end if;
      state.Set_Changed (True);
   end Change_Done;

   procedure Change_Done (Obj : in out mcc.tki.Widget.Button.Button'Class) is
   begin
      Change_Done;
   end Change_Done;

   -------------------------------------------------------------
   -- procedure Apply_Changes
   --
   -- Update window with changes so far
   --
   -- Algorithm
   -- 1) Set changes into record
   -- 2) If OK, undisplay and redisplay, changing name between
   -- 3) Note the window has changed
   -------------------------------------------------------------
   procedure Apply_Changes is
      Window : gui.Window.Window_Pointer := state.Get_Current_Window;
   begin
      Set_Changes (Window);
      if Check_Changes (Window) then
         gui.Window.Undisplay_Window (Window.all);
         Set_Name (Window);
         gui.Window.Display_Window (Window.all);
      end if;
      state.Set_Changed (True);
   end Apply_Changes;

   procedure Apply_Changes
     (Obj : in out mcc.tki.Widget.Button.Button'Class)
   is
   begin
      Apply_Changes;
   end Apply_Changes;

   -------------------------------------------------------------
   -- procedure Cancel_Changes
   --
   -- Destroy dialog without doing anything
   -------------------------------------------------------------
   procedure Cancel_Changes is
      Window : gui.Window.Window_Pointer := state.Get_Current_Window;
   begin
      if Check_Changes (Window) then
         mcc.tki.Destroy
           (mcc.tki.Object (
           change_window_dialog_window.change_window_dialog_window));
      end if;
   end Cancel_Changes;

   procedure Cancel_Changes
     (Obj : in out mcc.tki.Widget.Button.Button'Class)
   is
   begin
      Cancel_Changes;
   end Cancel_Changes;

   -------------------------------------------------------------
   -- procedure Duplicate_Choice
   --
   -- duplicate a widget
   --
   -- 1) Can't do this if dialog is running, so just exit (ignore)
   --    Same if there is no window or selection
   -- 2) Create a new widget of the same class
   -- 3) Call Subwindow_Actions.Duplicate to complete
   -- 4) Note window has changed
   -------------------------------------------------------------
   procedure Duplicate_Choice
     (Obj : in out mcc.tki.Widget.Button.Button'Class)
   is
   begin
      Duplicate_Choice;
   end Duplicate_Choice;

   procedure Duplicate_Choice is
      Window    : gui.Window.Window_Pointer := state.Get_Current_Window;
      Selection : gui.Widget.Widget_Access  := state.Get_Selection;
      New_Guy   : gui.Widget.Widget_Access;
   begin -- Duplicate_Choice
      -- don't do if dialog already running
      if state.Dialog_Running then
         return;
      end if;

      if Window /= null and then Selection /= null then
         New_Guy := new gui.Widget.GUI_Widget'Class'(Selection.all);
         Subwindow_Actions.Duplicate (New_Guy);
         state.Set_Changed (True);
      end if;
   end Duplicate_Choice;

   -------------------------------------------------------------
   -- procedure Delete_Choice
   --
   -- duplicate a widget
   --
   -- 1) Can't do this if dialog is running, so just exit (ignore)
   --    Same if there is no window or selection
   -- 2) Find selected widget in list (using name)
   -- 3) Undisplay widget
   -- 4) Remove highlighting (if widget was highlighted)
   -- 5) Delete from list
   -- 6) Note window has changed
   -------------------------------------------------------------
   procedure Delete_Choice
     (Obj : in out mcc.tki.Widget.Button.Button'Class)
   is
   begin
      Delete_Choice;
   end Delete_Choice;

   procedure Delete_Choice is
      Window      : gui.Window.Window_Pointer := state.Get_Current_Window;
      Position    : gui.Widget.Widget_Pointer;
      OldPosition : gui.Widget.Widget_Pointer;
      Selection   : gui.Widget.Widget_Access;
   begin -- Delete_Choice
      -- don't do if dialog already running
      if state.Dialog_Running then
         return;
      end if;

      if Window /= null then
         Position :=
            gui.Widget.Widget_List_Package.First (Window.Widget_List);
         while not gui.Widget.Widget_List_Package.IsPastEnd
                     (Window.Widget_List,
                      Position)
         loop
            Selection :=
               gui.Widget.Widget_List_Package.Retrieve
                 (Window.Widget_List,
                  Position);
            if Selection.Is_Selected then
               gui.Widget.Undisplay_Widget (Widget => Selection.all);
               gui.Widget.Unhighlight (Widget => Selection.all);
               OldPosition := Position;
               gui.Widget.Widget_List_Package.GoAhead
                 (Window.Widget_List,
                  Position);
               gui.Widget.Widget_List_Package.Delete
                 (L => Window.Widget_List,
                  P => OldPosition);
               state.Set_Changed (True);
            else
               gui.Widget.Widget_List_Package.GoAhead
                 (Window.Widget_List,
                  Position);
            end if;
         end loop;
      end if;

   end Delete_Choice;

end Edit_Menu;
