---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-MENU.ADB
--  Description : Root of GUI MENU Hierarchy
--
--  Copyright (C) 2003, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
-- Change log:
-- 09/21/99 (mcc) : fixed problem with some menu names not
--                  generating legal ada identifiers
---------------------------------------------------------------
with Ada.Characters.Handling;
with Generate_Helpers;
with File_Helpers;
with mcc;

package body gui.Menu is

   function To_Lower (Item : String) return String
                   renames Ada.Characters.Handling.To_Lower;

   package Listpkg renames Menu_List_Package;

   function Convert_Menu_Name (Menu_Name : in String) return String is
      Result        : String := "M" & Menu_Name;
      Start, Finish : Integer;
   begin
      -- make sure we don't begin with non letter
      if Ada.Characters.Handling.Is_Letter (Result (Result'First + 1)) then
         Start := Result'First + 1;
      else
         Start := Result'First;
      end if;

      for I in Result'Range loop
         if (Result (I) = '.' or Result (I) = ' ')
           or else not Ada.Characters.Handling.Is_Alphanumeric (Result (I))
         then
            -- need to make sure we don't have two underscores in a row
            if Result (I - 1) = '_' then
               Result (I) := 'x';
            else
               Result (I) := '_';
            end if;
         end if;
      end loop;

      -- make sure we don't end in underscore
      Finish := Result'Last;
      while Result (Finish) = '_' loop
         Finish := Finish - 1;
      end loop;

      return Result (Start .. Finish);
   end Convert_Menu_Name;

   function Separator_Name (Menu : in Menu_Access) return String is
   begin
      return "Separator" & mcc.Img (Menu.Number);
   end Separator_Name;

   -- update novice mode actions if window name changes.
   procedure Update_Actions
     (Menu       : in Menu_Pointer;
      Old_Window : in String;
      New_Window : in String)
   is
      procedure Update_Action (Menu : in out gui.Menu.Menu'Class) is
         First : constant Positive := Menu.Action'First;
      begin
         if Menu.Action'Length <= Old_Window'Length + 8 then
            return;
         end if;
         if To_Lower
               (Menu.Action (First .. First + Old_Window'Length + 7)) =
            To_Lower (Old_Window) & "_actions"
         then
            Menu.Action :=
              new String'
                (New_Window & "_actions" &
                 Menu.Action (First + Old_Window'Length + 8 ..
                              Menu.Action'Last));
         end if;
      end Update_Action;
      Current_Position : gui.Menu.Menu_Position := Listpkg.First (Menu);
      Current_Menu     : gui.Menu.Menu_Access;
   begin
      while not Listpkg.IsPastEnd (Menu, Current_Position)
      loop
         Current_Menu := Listpkg.Retrieve (Menu, Current_Position);

         if Current_Menu.all in gui.Menu.Submenu'Class then
            Update_Action (Current_Menu.all);
            Update_Actions
              (gui.Menu.Submenu (Current_Menu.all).Items,
               Old_Window,
               New_Window);
         elsif Current_Menu.all in gui.Menu.Menu_Item'Class then
            Update_Action (Current_Menu.all);
         elsif Current_Menu.all in gui.Menu.Separator'Class then
            null;
         else
            raise Constraint_Error;
         end if;
         Listpkg.GoAhead (Menu, Current_Position);
      end loop;
   end Update_Actions;

   -- wbw 6/7/99
   procedure Generate_Menu_Declaration (Menu : in Menu_Pointer) is

      Current_Position : gui.Menu.Menu_Position := Listpkg.First (Menu);
      Current_Menu     : gui.Menu.Menu_Access;
   begin
      while not Listpkg.IsPastEnd (Menu, Current_Position)
      loop
         Current_Menu := Listpkg.Retrieve (Menu, Current_Position);

         if Current_Menu.all in gui.Menu.Submenu'Class then
            File_Helpers.P
              (Convert_Menu_Name (Current_Menu.all.Name.all) &
               "_SubMenu : aliased mcc.tki.Menu.SubMenu;");
            Generate_Menu_Declaration
              (gui.Menu.Submenu (Current_Menu.all).Items);
         elsif Current_Menu.all in gui.Menu.Menu_Item'Class then
            File_Helpers.P
              (Convert_Menu_Name (Current_Menu.all.Name.all) &
               "_Choice : aliased mcc.tki.Menu.Choice;");
         elsif Current_Menu.all in gui.Menu.Separator'Class then
            File_Helpers.P
              (Separator_Name (Current_Menu) &
               " : aliased mcc.tki.Menu.Separator;");
         else
            raise Constraint_Error;
         end if;
         Listpkg.GoAhead (Menu, Current_Position);
      end loop;
   end Generate_Menu_Declaration;

   procedure Generate_Menu_Novice
     (Menu      : in Menu_Pointer;
      Spec_File : in Ada.Text_IO.File_Type;
      Body_File : in Ada.Text_IO.File_Type;
      Callback  : in Already_Callback)
   is
      use Ada.Text_IO;

      Current_Position : gui.Menu.Menu_Position := Listpkg.First (Menu);
      Current_Menu     : gui.Menu.Menu_Access;
      procedure Write_Spec is
      begin
         if not Callback.all
                  (True,
                   Current_Menu.all.Action.all,
                   Convert_Menu_Name (Current_Menu.all.Name.all))
         then
            Put_Line
              (Spec_File,
               "   procedure " &
               Convert_Menu_Name (Current_Menu.all.Name.all) &
               "_Selected;");
         end if;
      end Write_Spec;
      procedure Write_Body is
      begin
         if not Callback.all
                  (True,
                   Current_Menu.all.Action.all,
                   Convert_Menu_Name (Current_Menu.all.Name.all))
         then
            New_Line (Body_File);
            Put_Line
              (Body_File,
               "   procedure " &
               Convert_Menu_Name (Current_Menu.all.Name.all) &
               "_Selected is");
            Put_Line (Body_File, "   begin");
            Put_Line (Body_File, "      null;");
            Put_Line
              (Body_File,
               "   end " &
               Convert_Menu_Name (Current_Menu.all.Name.all) &
               "_Selected;");
         end if;
      end Write_Body;
   begin
      while not Listpkg.IsPastEnd (Menu, Current_Position)
      loop
         Current_Menu := Listpkg.Retrieve (Menu, Current_Position);

         if Current_Menu.all in gui.Menu.Submenu'Class then
            Write_Spec;
            Write_Body;
            Generate_Menu_Novice
              (gui.Menu.Submenu (Current_Menu.all).Items,
               Spec_File,
               Body_File,
               Callback);
         elsif Current_Menu.all in gui.Menu.Menu_Item'Class then
            Write_Spec;
            Write_Body;
         elsif Current_Menu.all in Separator'Class then
            null;
         else
            raise Constraint_Error;
         end if;
         Listpkg.GoAhead (Menu, Current_Position);
      end loop;
   end Generate_Menu_Novice;

   procedure Generate_Action_Context_Clause (Menu : in Menu_Pointer) is
      Current_Menu_Position : gui.Menu.Menu_Position := Listpkg.First (Menu);
      Current_Menu          : gui.Menu.Menu_Access;
   begin
      while not Listpkg.IsPastEnd
                  (L => Menu,
                   P => Current_Menu_Position)
      loop
         Current_Menu := Listpkg.Retrieve
              (L => Menu,
               P => Current_Menu_Position);
         if Current_Menu.all in Menu_Item'Class then
            Generate_Helpers.Generate_With_For_FQN (Current_Menu.Action.all);
         end if;
         if Current_Menu.all in Submenu'Class then
            Generate_Action_Context_Clause
              (Menu => Submenu (Current_Menu.all).Items);
         end if;
         Listpkg.GoAhead (L => Menu, P => Current_Menu_Position);
      end loop;
   end Generate_Action_Context_Clause;

   procedure Generate_Menu_Creation
     (Menu   : in Menu_Pointer;
      Parent : in String)
   is
      Current_Position : gui.Menu.Menu_Position := Listpkg.First (Menu);
      Current_Menu     : gui.Menu.Menu_Access;

      use File_Helpers;
   begin
      while not Listpkg.IsPastEnd (Menu, Current_Position)
      loop
         Current_Menu := Listpkg.Retrieve (Menu, Current_Position);

         if Current_Menu.all in gui.Menu.Submenu'Class then
            P ("mcc.tki.Menu.Add_SubMenu");
            P ("  (Obj         => " &
               Convert_Menu_Name (Current_Menu.all.Name.all) &
               "_SubMenu,");
            P ("   Text        => """ &
               Current_Menu.all.Name.all & """,");
            P ("   Underline   =>" &
               Integer'Image (Current_Menu.all.Underline_Position) & ",");
            P ("   Parent_Menu => " & Parent, Newline => False);
            if Current_Menu.all.Action /= null
              and then Current_Menu.all.Action.all /= "null"
            then
               P (",", Indent => False);
               P ("   On_Post     => " & Current_Menu.all.Action.all &
                  "'access);");
            else
               P (");", Indent => False);
            end if;
            Generate_Menu_Creation
              (Menu   => gui.Menu.Submenu (Current_Menu.all).Items,
               Parent => Convert_Menu_Name (Current_Menu.all.Name.all) &
                         "_SubMenu");
         elsif Current_Menu.all in gui.Menu.Menu_Item'Class then
            P ("mcc.tki.Menu.Add_Choice");
            P ("  (Obj         => " &
               Convert_Menu_Name (Current_Menu.all.Name.all) & "_Choice,");
            P ("   To_Menu     => " & Parent & ",");
            P ("   Text        => """ & Current_Menu.all.Name.all & """,");
            P ("   Action      => ", Newline => False);
            -- although action is an attribute of menu, this will change in
            -- the future, so menu_item is used here
            declare
               Action : constant gui.String_Pointer
                      := gui.Menu.Menu_Item (Current_Menu.all).Action;
            begin
               if Action /= null and then
                 (Action.all /= "" and To_Lower (Action.all) /= "null")
               then
                  Put
                    (Generate_Helpers.Undash_Name (Current_Menu.Action.all)
                     & "'Access,");
               else
                  Put ("null,");
               end if;
            end;
            NL;
            P ("   Underline   =>" &
               Integer'Image (Current_Menu.all.Underline_Position) & ",");
            P ("   Accelerator => " & '"', Newline => False);
            if gui.Menu.Menu_Item (Current_Menu.all).Accelerator /= null
            then
               Put (gui.Menu.Menu_Item (Current_Menu.all).Accelerator.all);
            end if;
            P (""");", Indent => False);
         elsif Current_Menu.all in gui.Menu.Separator'Class then
            P ("mcc.tki.Menu.Add_Separator");
            P ("  (Obj         => " & Separator_Name (Current_Menu) & ",");
            P ("   To_Menu     => " & Parent & ");");
         else
            raise Constraint_Error;
         end if;
         Listpkg.GoAhead (Menu, Current_Position);
      end loop;
   end Generate_Menu_Creation;

end Gui.Menu;
