---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET-DROPDOWN.ADB
--  Description : GUI Widget Dropdown
--
--  Copyright (C) 2000, Jonathan Busch and Martin Carlisle
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with Gui_Enum;
with File_Helpers;
with Widget_IO;
with mcc.Common_Dialogs;
with dropdown_dialog_window;
with Ada.Exceptions;
with mcc.tki.Widget.Dropdown; use type mcc.tki.Widget.Widget_Pointer;
with mcc.tki.Colors;
with Generate_Helpers;
with state;

package body gui.Widget.Dropdown is

   use type mcc.tki.Colors.Optional_Color;

   -- reads information from file into Dropdown,
   -- assumes keyword already read.
   procedure Read_Widget (Widget : in out Dropdown) is
   begin -- Read_Widget
      Read_Widget (GUI_Widget (Widget));
      -- Get number of rows for list when open
      Widget.Number_Rows       :=
         Natural'Value (File_Helpers.Token (File_Helpers.Token_Index).all);
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      -- Get Colors
      Widget_IO.Get_FG_BG_Color (Widget.Color.FG, Widget.Color.BG);

      if File_Helpers.Token_Index > File_Helpers.N_Tokens then
         return;
      end if;
      Widget.Data_Item   := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
      Widget.Data_Type  := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
   end Read_Widget;

   -- Writes information to file from Dropdown
   procedure Write_Widget (Widget : in Dropdown) is
   begin -- Write_Widget
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.Dropdown) & " ");
      Write_Widget (GUI_Widget (Widget));

      File_Helpers.Put (Natural'Image (Widget.Number_Rows));
      Widget_IO.Put_FG_BG_Color (Widget.Color.FG, Widget.Color.BG);
      File_Helpers.Put (" """);
      if Widget.Data_Item /= null then
         File_Helpers.Put (Widget.Data_Item.all);
      end if;
      File_Helpers.Put (""" """);
      if Widget.Data_Type /= null then
         File_Helpers.Put (Widget.Data_Type.all);
      end if;
      File_Helpers.P ("""");
   end Write_Widget;

   -- mcc 05/23/00
   procedure Generate_Action_Context_Clause (Widget : in Dropdown) is
   begin
      gui.Widget.Typed_Object.Generate_Action_Context_Clause
        (gui.Widget.Typed_Object.Object (Widget));
   end Generate_Action_Context_Clause;

   -- SAG 02.Dec.1999
   procedure Generate_Callback_Context_Clause
     (Widget     : in Dropdown;
      Windowname : in String)
   is
   begin
      Generate_Action_Context_Clause (Widget);  -- CHECK: Do we need this?
      Generate_Helpers.Generate_With ("mcc.tki.Widget.Dropdown");
      Generate_Helpers.Generate_With (Windowname);
   end Generate_Callback_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Context_Clause (Widget : in Dropdown) is
   begin
      Generate_Helpers.Generate_With ("mcc.tki.Widget.Dropdown");
      Generate_Helpers.Generate_With ("mcc.tki.Colors");
   end Generate_Widget_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Declaration (Widget : in Dropdown) is
   begin
      File_Helpers.P
        (Widget.Name.all &
         " : aliased mcc.tki.Widget.Dropdown.Dropdown;");
   end Generate_Widget_Declaration;

   function Normalize (Name : String) return String
      renames Generate_Helpers.Undash_Name;

   -- wbw 5/10/99 modified by mcc 05/24/00
   procedure Generate_Widget_Creation
     (Widget      : in Dropdown;
      Window_Name : in String)
   is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      P ("mcc.tki.Widget.Dropdown.Create");
      P ("  (Obj         => " & Widget.Name.all & ",");
      P ("   Parent      => " & Window_Name & ",");
      P ("   X           =>" & Integer'Image (Widget.x) & ",");
      P ("   Y           =>" & Integer'Image (Widget.y) & ",");
      P ("   Width       =>" & Integer'Image (Widget.Width) & ",");
      P ("   Height      =>" & Integer'Image (Widget.Height) & ",");
      P ("   Number_Rows =>" & Integer'Image (Widget.Number_Rows) &
         ");");

      if Widget.Data_Type /= null
        and then Widget.Data_Type.all /= ""
      then
         P ("for E in " & Normalize (Widget.Data_Type.all) & " loop");
         P ("   mcc.tki.Widget.Dropdown.Add_Entry");
         P ("     (Obj      => " & Widget.Name.all & ",");
         P ("      Location => " & Normalize (Widget.Data_Type.all) & "'Pos(E),");
         P ("      Text     => " &
            Normalize (Widget.Data_Type.all) &
            "'Image(E));");
         P ("end loop;");
      end if;

      Generate_Helpers.Gen_Set_FG_BG_Color
        ("Dropdown", Widget.Name.all, Widget.Color.FG, Widget.Color.BG);
   end Generate_Widget_Creation;

   function Has_Anything_To_Fill_Or_Read
     (Widget : in Dropdown)
      return   Boolean
   is
   begin
      return Widget.Data_Item /= null
            and then Widget.Data_Item.all /= ""
            and then Widget.Data_Type /= null
            and then Widget.Data_Type.all /= "";
   end Has_Anything_To_Fill_Or_Read;

   procedure Generate_Fill_Action (Widget : in Dropdown) is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      if Has_Anything_To_Fill_Or_Read (Widget) then
         P ("mcc.tki.Widget.Dropdown.Select_Item");
         P ("  (Obj      => " & Widget.Name.all & ",");
         P ("   Number   => " & Normalize (Widget.Data_Type.all) &
            "'Pos (" & Normalize (Widget.Data_Item.all) & ") + 1);");
      end if;
   end Generate_Fill_Action;

   procedure Generate_Read_Action (Widget : in Dropdown) is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      if Has_Anything_To_Fill_Or_Read (Widget) then
         P ("declare");
         P ("   Position : Integer := " &
            "mcc.tki.Widget.Dropdown.Get_Selected (" &
            Widget.Name.all & ") - 1;");
         P ("begin");
         P ("   " & Normalize (Widget.Data_Item.all) & " := " &
            Normalize (Widget.Data_Type.all) & "'Val (Position);");
         P ("end;");
      end if;
   end Generate_Read_Action;

   -- display the widget to a window
   procedure Display_Widget
     (Widget    : in out Dropdown;
      Container : in out mcc.tki.Container.Container'Class)
   is

   begin
      if Widget.The_Widget = null then
         Widget.The_Widget := new mcc.tki.Widget.Dropdown.Dropdown;
      end if;

      mcc.tki.Widget.Dropdown.Create
        (Obj         =>
           mcc.tki.Widget.Dropdown.Dropdown (Widget.The_Widget.all),
         Parent      => Container,
         X           => Widget.x,
         Y           => Widget.y,
         Width       => Widget.Width,
         Height      => Widget.Height,
         Number_Rows => Widget.Number_Rows);

      if Widget.Color.FG /= mcc.tki.Colors.Default then
         mcc.tki.Widget.Set_Foreground_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.tki.Colors.Named_Color (Widget.Color.FG));
      end if;
      if Widget.Color.BG /= mcc.tki.Colors.Default then
         mcc.tki.Widget.Set_Background_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.tki.Colors.Named_Color (Widget.Color.BG));
      end if;

      Display_Widget (GUI_Widget (Widget), Container);

   exception
      when e : others =>
         mcc.Common_Dialogs.Ok_Box
           ("Can't display: " &
            Widget.Name.all &
            ASCII.LF &
            Ada.Exceptions.Exception_Information (e));
   end Display_Widget;

   procedure Set_Properties (Widget : in out Dropdown) is
   begin
      dropdown_dialog_window.Generate_Window;
      Widget.Properties      :=
        dropdown_dialog_window.dropdown_dialog_window'Access;
      Widget.Name_Entry      := dropdown_dialog_window.entry1'Access;
      Widget.X_Entry         := dropdown_dialog_window.entry2'Access;
      Widget.Y_Entry         := dropdown_dialog_window.entry3'Access;
      Widget.Width_Entry     := dropdown_dialog_window.entry4'Access;
      Widget.Height_Entry    := dropdown_dialog_window.entry5'Access;
      Widget.Color.FG_DD     := dropdown_dialog_window.fg_dd'Access;
      Widget.Color.BG_DD     := dropdown_dialog_window.bg_dd'Access;
      Widget.Rows_Entry      := dropdown_dialog_window.Row_Entry'Access;
      Widget.Data_Type_Entry := dropdown_dialog_window.entry8'Access;
      Widget.Data_Item_Entry := dropdown_dialog_window.entry9'Access;
      Set_Properties (GUI_Widget (Widget));

      mcc.tki.Widget.Text_Entry.Set_Text
        (Obj  => Widget.Rows_Entry.all,
         Text => Widget.Number_Rows);

      mcc.tki.Colors.Fill_Colors (Widget.Color);

      if state.Get_Current_Window.Novice_Mode then
         mcc.tki.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Data_Item_Entry.all,
            Text => "disabled for novice");
         mcc.tki.Widget.Text_Entry.Disable (Widget.Data_Item_Entry.all);
         mcc.tki.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Data_Type_Entry.all,
            Text => "disabled for novice");
         mcc.tki.Widget.Text_Entry.Disable
           (Widget.Data_Type_Entry.all);
      else
         if Widget.Data_Item /= null then
            mcc.tki.Widget.Text_Entry.Set_Text
              (Obj  => Widget.Data_Item_Entry.all,
               Text => Widget.Data_Item.all);
         end if;
         if Widget.Data_Type /= null then
            mcc.tki.Widget.Text_Entry.Set_Text
              (Obj  => Widget.Data_Type_Entry.all,
               Text => Widget.Data_Type.all);
         end if;
      end if;
   end Set_Properties;

   procedure Apply_Properties (Widget : in out Dropdown) is
   begin
      Apply_Properties (GUI_Widget (Widget));

      begin
         Widget.Number_Rows :=
            mcc.tki.Widget.Text_Entry.Get_Text (Widget.Rows_Entry.all);
      exception
         when others =>
            Widget.Number_Rows := -99;
      end;
      mcc.tki.Colors.Read_Colors (Widget.Color);
      declare
         Enumeration_Var : constant String :=
            mcc.tki.Widget.Text_Entry.Get_Text
              (Widget.Data_Item_Entry.all);
      begin
         if Enumeration_Var /= "disabled for novice" then
            Widget.Data_Item := new String'(Enumeration_Var);
         end if;
      end;
      declare
         Enumeration_Type : constant String :=
            mcc.tki.Widget.Text_Entry.Get_Text
              (Widget.Data_Type_Entry.all);
      begin
         if Enumeration_Type /= "disabled for novice" then
            Widget.Data_Type := new String'(Enumeration_Type);
         end if;
      end;
   end Apply_Properties;

   procedure Check_Properties (Widget : in out Dropdown; Ok : out Boolean) is
   begin
      Check_Properties (GUI_Widget (Widget), Ok);

      if Ok and then Widget.Number_Rows < 0 then
         Ok := False;
         mcc.tki.Widget.Text_Entry.Highlight (Widget.Rows_Entry.all);
      end if;

      if not Ok then
         mcc.tki.Bell;
      end if;
   end Check_Properties;

end Gui.Widget.Dropdown;
