---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET-LISTBOX.ADB
--  Description : GUI Widget Listbox
--
--  Copyright (C) 1999, Jonathan Busch and Martin Carlisle
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with Gui_Enum;
with File_Helpers;
with Widget_IO;
with Generate_Helpers;
with mcc.Msg;
with listbox_dialog_window;
with Ada.Exceptions;
with mcc.tki.Widget.Button.Check;
with mcc.tki.Widget.Listbox;      use type mcc.tki.Widget.Widget_Pointer;
with mcc.tki.Colors;

package body gui.Widget.Listbox is

   use type mcc.tki.Colors.Optional_Color;

   -- reads information from file into Listbox,
   -- assumes keyword already read.
   procedure Read_Widget (Widget : in out Listbox) is
   begin
      Read_Widget (GUI_Widget (Widget));
      -- Check for horizontal scrollbar
      Widget.HZ_Scroll         :=
         Boolean'Value (File_Helpers.Token (File_Helpers.Token_Index).all);
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      -- Check for vertical scrollbar
      Widget.VR_Scroll         :=
         Boolean'Value (File_Helpers.Token (File_Helpers.Token_Index).all);
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      -- Get Colors
      Widget_IO.Get_FG_BG_Color (Widget.Color.FG, Widget.Color.BG);
   end Read_Widget;

   -- Writes information to file from Listbox
   procedure Write_Widget (Widget : in Listbox) is
   begin
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.Listbox) & " ");
      Write_Widget (GUI_Widget (Widget));
      File_Helpers.Put (" " & Boolean'Image (Widget.HZ_Scroll));
      File_Helpers.Put (" " & Boolean'Image (Widget.VR_Scroll));
      Widget_IO.Put_FG_BG_Color (Widget.Color.FG, Widget.Color.BG);
      File_Helpers.NL;
   end Write_Widget;

   -- wbw 6/6/99
   procedure Generate_Widget_Context_Clause (Widget : in Listbox) is
   begin
      Generate_Helpers.Generate_With ("mcc.tki.Widget.Listbox");
      Generate_Helpers.Generate_With ("mcc.tki.Colors");
   end Generate_Widget_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Declaration (Widget : in Listbox) is
   begin
      File_Helpers.P
        (Widget.Name.all &
         " : aliased mcc.tki.Widget.Listbox.Listbox;");
   end Generate_Widget_Declaration;

   -- wbw 5/10/99
   procedure Generate_Widget_Creation
     (Widget      : in Listbox;
      Window_Name : in String)
   is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      P ("mcc.tki.Widget.ListBox.Create");
      P ("  (Obj                  => " & Widget.Name.all & ",");
      P ("   Parent               => " & Window_Name & ",");
      P ("   X                    =>" & Integer'Image (Widget.x) & ",");
      P ("   Y                    =>" & Integer'Image (Widget.y) & ",");
      P ("   Width                =>" & Integer'Image (Widget.Width) & ",");
      P ("   Height               =>" & Integer'Image (Widget.Height) & ",");
      P ("   Horizontal_Scrollbar =>" &
         Boolean'Image (Widget.HZ_Scroll) & ",");
      P ("   Vertical_Scrollbar   =>" &
         Boolean'Image (Widget.VR_Scroll) & ");");
      Generate_Helpers.Gen_Set_FG_BG_Color
        ("Listbox", Widget.Name.all, Widget.Color.FG, Widget.Color.BG);
   end Generate_Widget_Creation;

   -- display the widget to a window
   procedure Display_Widget
     (Widget    : in out Listbox;
      Container : in out mcc.tki.Container.Container'Class)
   is

   begin
      if Widget.The_Widget = null then
         Widget.The_Widget := new mcc.tki.Widget.Listbox.Listbox;
      end if;

      mcc.tki.Widget.Listbox.Create
        (Obj                  =>
           mcc.tki.Widget.Listbox.Listbox (Widget.The_Widget.all),
         Parent               => Container,
         X                    => Widget.x,
         Y                    => Widget.y,
         Width                => Widget.Width,
         Height               => Widget.Height,
         Horizontal_Scrollbar => Widget.HZ_Scroll,
         Vertical_Scrollbar   => Widget.VR_Scroll);

      if Widget.Color.FG /= mcc.tki.Colors.Default then
         mcc.tki.Widget.Set_Foreground_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.tki.Colors.Named_Color (Widget.Color.FG));
      end if;
      if Widget.Color.BG /= mcc.tki.Colors.Default then
         mcc.tki.Widget.Set_Background_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.tki.Colors.Named_Color (Widget.Color.BG));
      end if;

      Display_Widget (GUI_Widget (Widget), Container);

   exception
      when e : others =>
         mcc.Msg.Error (E, "Can't display " & Widget.Name.all);
   end Display_Widget;

   procedure Set_Properties (Widget : in out Listbox) is
   begin
      listbox_dialog_window.Generate_Window;
      Widget.Properties      :=
        listbox_dialog_window.listbox_dialog_window'Access;
      Widget.Name_Entry      := listbox_dialog_window.entry1'Access;
      Widget.X_Entry         := listbox_dialog_window.entry2'Access;
      Widget.Y_Entry         := listbox_dialog_window.entry3'Access;
      Widget.Width_Entry     := listbox_dialog_window.entry4'Access;
      Widget.Height_Entry    := listbox_dialog_window.entry5'Access;
      Widget.VR_Scroll_Check := listbox_dialog_window.vertical'Access;
      Widget.HZ_Scroll_Check := listbox_dialog_window.horizontal'Access;
      Widget.Color.FG_DD     := listbox_dialog_window.fg_dd'Access;
      Widget.Color.BG_DD     := listbox_dialog_window.bg_dd'Access;
      Set_Properties (GUI_Widget (Widget));

      --vertical scrollbar?
      if (Widget.VR_Scroll = True) then
         mcc.tki.Widget.Button.Check.Select_Check
           (Obj => Widget.VR_Scroll_Check.all);
      else
         mcc.tki.Widget.Button.Check.Unselect_Check
           (Obj => Widget.VR_Scroll_Check.all);
      end if;

      --horizontal scrollbar?
      if (Widget.HZ_Scroll = True) then
         mcc.tki.Widget.Button.Check.Select_Check
           (Obj => Widget.HZ_Scroll_Check.all);
      else
         mcc.tki.Widget.Button.Check.Unselect_Check
           (Obj => Widget.HZ_Scroll_Check.all);
      end if;

      mcc.tki.Colors.Fill_Colors (Widget.Color);
   end Set_Properties;

   procedure Apply_Properties (Widget : in out Listbox) is
   begin
      Apply_Properties (GUI_Widget (Widget));

      --Vertical scrollbar?
      Widget.VR_Scroll :=
         mcc.tki.Widget.Button.Check.Is_Checked (Widget.VR_Scroll_Check.all);

      --Horizontal scrollbar?
      Widget.HZ_Scroll :=
         mcc.tki.Widget.Button.Check.Is_Checked (Widget.HZ_Scroll_Check.all);

      mcc.tki.Colors.Read_Colors (Widget.Color);
   end Apply_Properties;

   procedure Check_Properties (Widget : in out Listbox; Ok : out Boolean) is
   begin
      Check_Properties (GUI_Widget (Widget), Ok);

      if not Ok then
         mcc.tki.Bell;
      end if;
   end Check_Properties;

end Gui.Widget.Listbox;
