---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET-TEXTBOX.ADB
--  Description : GUI Widget Textbox
--
--  By: Jonathan Busch and Martin Carlisle
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
--
-- Copyright (C) 2005, SA Griffin <stephen.griffin.external@eads.com>
-- Copyright (C) 2009, OM Kellogg <okellogg@users.sourceforge.net>
----------------------------------------------------------------------
with Gui_Enum;
with File_Helpers;
with mcc.Msg;
with Textbox_dialog_window;
with state;
with Ada.Strings.Fixed;
with Generate_Helpers;
with Widget_IO;
with mcc.tki.Widget.Button.Check;
with mcc.tki.Widget.Textbox;      use type mcc.tki.Widget.Widget_Pointer;
with mcc.tki.Colors;

package body gui.Widget.Textbox is

   use type mcc.tki.Colors.Optional_Color;

   -- reads information from file into Textbox,
   -- assumes keyword already read.

   procedure Read_Widget (Widget : in out Textbox) is
   begin

      Read_Widget (GUI_Widget (Widget));

      -- Check for horizontal scrollbar
      Widget.HZ_Scroll         :=
         Boolean'Value (File_Helpers.Token (File_Helpers.Token_Index).all);
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      -- Check for vertical scrollbar
      Widget.VR_Scroll         :=
         Boolean'Value (File_Helpers.Token (File_Helpers.Token_Index).all);
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      -- Get Colours
      Widget_IO.Get_FG_BG_Color (Widget.Color.FG, Widget.Color.BG);

      Widget.Data_Item         := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      Widget.Base_Type         := mcc.text_entry_types.Default_Base_Type;

      -- test for EOL for backward compatibility
      if File_Helpers.Token_Index > File_Helpers.N_Tokens then
         return;
      end if;

      Widget.Data_Type         := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      declare
         Base_Type : constant String :=
            File_Helpers.Token (File_Helpers.Token_Index).all;
      begin
         Widget.Base_Type := mcc.text_entry_types.String_Types'Value (Base_Type);
      exception
         when others =>
            mcc.Msg.Error
              ("Syntax error at base type: " &
               Base_Type &
               ", using default (" &
               mcc.text_entry_types.Base_Type'Image
                  (mcc.text_entry_types.Default_Base_Type) &
               ")");
      end;
   exception
      when E : others =>
         mcc.Msg.Error (E);
   end Read_Widget;

   -- Writes information to file from Textbox
   procedure Write_Widget (Widget : in Textbox) is
   begin
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.Textbox) & " ");
      Write_Widget (GUI_Widget (Widget));

      File_Helpers.Put (" " & Boolean'Image (Widget.HZ_Scroll));
      File_Helpers.Put (" " & Boolean'Image (Widget.VR_Scroll));
      Widget_IO.Put_FG_BG_Color (Widget.Color.FG, Widget.Color.BG);
      File_Helpers.Put (" ");
      File_Helpers.Put_String (Widget.Data_Item);
      if Widget.Data_Type /= null then
         File_Helpers.Put_String (Widget.Data_Type);
         File_Helpers.Put
           (" " & mcc.text_entry_types.String_Types'Image (Widget.Base_Type));
      end if;
      File_Helpers.NL;
   end Write_Widget;

   -- wbw 6/6/99
   procedure Generate_Widget_Context_Clause (Widget : in Textbox) is
   begin
      Generate_Helpers.Generate_With ("mcc.tki.Widget.Textbox");
      Generate_Helpers.Generate_With ("mcc.tki.Colors");
   end Generate_Widget_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Declaration (Widget : in Textbox) is
   begin
      File_Helpers.P
        (Widget.Name.all & " : aliased mcc.tki.Widget.Textbox.Textbox;");
   end Generate_Widget_Declaration;

   function Normalize (Name : String) return String
      renames Generate_Helpers.Undash_Name;

   procedure Generate_Text_Conversion
     (Widget     : Textbox;
      Subprogram : String) is

      use File_Helpers;
      use type mcc.Text_Entry_Types.Base_Type;
   begin
      case Widget.Base_Type is
         when mcc.text_entry_types.String_Subtype =>
            P ("   Text   => " & Normalize (Widget.Data_Item.all) & ");");
         when mcc.text_entry_types.Unbounded_String
            | mcc.text_entry_types.Bounded_String =>
            if Widget.Data_Type /= null then
               declare
                  Last_Dot : Natural := Ada.Strings.Fixed.Index
                    (Widget.Data_Type.all, ".", Ada.Strings.Backward);
               begin
                  P ("   Text   => " &
                     Normalize (Widget.Data_Type (1 .. Last_Dot)) &
                     "To_String (" & Normalize (Widget.Data_Item.all) & "));");
               end;
            elsif Widget.Base_Type = mcc.text_entry_types.Bounded_String then
               mcc.Msg.Error
                 ("Gui.Widget.Textbox." & Subprogram & " (" & Widget.Name.all &
                  ") : user data type required for BOUNDED_STRING");
            else
               P ("   Text   => Ada.Strings.Unbounded.To_String (" &
                  Normalize (Widget.Data_Item.all) & "));");
            end if;
      end case;
   end Generate_Text_Conversion;

   --
   -- wbw 5/10/99
   -- SAG 07.08.2005
   --
   procedure Generate_Widget_Creation
     (Widget      : in Textbox;
      Window_Name : in String)
   is
      use File_Helpers;
   begin
      P ("mcc.tki.Widget.Textbox.Create ");
      P ("  (Obj                  => " & Widget.Name.all & ",");
      P ("   Parent               => " & Window_Name & ",");
      P ("   X                    => " & mcc.Img (Widget.x) & ",");
      P ("   Y                    => " & mcc.Img (Widget.y) & ",");
      P ("   Width                => " & mcc.Img (Widget.Width) & ",");
      P ("   Height               => " & mcc.Img (Widget.Height) & ",");
      P ("   Horizontal_Scrollbar => " &
         Boolean'Image (Widget.HZ_Scroll) & ",");
      P ("   Vertical_Scrollbar   => " &
         Boolean'Image (Widget.VR_Scroll) & ");");
      NL;
      if Widget.Data_Item /= null and then Widget.Data_Item.all /= "" then
         P ("mcc.tki.Widget.Textbox.Set_Text");
         P ("  (Obj  => " & Widget.Name.all & ",");
         Generate_Text_Conversion (Widget, "Generate_Widget_Creation");
      end if;
      Generate_Helpers.Gen_Set_FG_BG_Color
        ("Textbox", Widget.Name.all, Widget.Color.FG, Widget.Color.BG);
   end Generate_Widget_Creation;

   -- SAG 08.07.2005
   procedure Generate_Action_Context_Clause (Widget : in Textbox) is
      use type mcc.Text_Entry_Types.Base_Type;
   begin
      gui.Widget.Typed_Object.Generate_Action_Context_Clause
        (gui.Widget.Typed_Object.Object (Widget));

      if Widget.Data_Item /= null
        and then Widget.Data_Item.all /= ""
        and then Widget.Base_Type = mcc.text_entry_types.Unbounded_String
        and then Widget.Data_Type = null then
         Generate_Helpers.Generate_With ("Ada.Strings.Unbounded");
      end if;
   end Generate_Action_Context_Clause;

   -- SAG 08.07.2005
   procedure Generate_Callback_Context_Clause
     (Widget     : in Textbox;
      Windowname : in String)
   is
   begin
      Generate_Action_Context_Clause (Widget);
      Generate_Helpers.Generate_With ("mcc.tki.Widget.Textbox");
      Generate_Helpers.Generate_With (Windowname);
   end Generate_Callback_Context_Clause;

   function Has_Anything_To_Fill_Or_Read
     (Widget : in Textbox)
      return   Boolean
   is
   begin
      return Widget.Data_Item /= null and then Widget.Data_Item.all /= "";
   end Has_Anything_To_Fill_Or_Read;

   -- SAG 08.07.2005
   procedure Generate_Fill_Action (Widget : in Textbox) is
      use File_Helpers;
   begin
      if Widget.Data_Item = null or else Widget.Data_Item.all = "" then
         return;
      end if;
      -- This code will read from a data item and write the value to the
      -- selected Text Box Widget.

      P ("mcc.tki.Widget.Textbox.Set_Text");
      P ("  (Obj    => " & Widget.Name.all & ",");

      Generate_Text_Conversion (Widget, "Generate_Fill_Action");
   end Generate_Fill_Action;

   -- SAG 02.Dec.1999
   procedure Generate_Read_Action (Widget : in Textbox) is
      use File_Helpers;

      procedure Print_Declare_Block is
      begin
         P ("declare");
         P ("   X : constant String := mcc.tki.Widget.Textbox.Get_Text");
         P ("                            (" & Widget.Name.all & ");");
         P ("begin");
         P ("   " & Normalize (Widget.Data_Item.all) & " := ", Newline => False);
      end Print_Declare_Block;
   begin
      if Widget.Data_Item = null or else Widget.Data_Item.all = "" then
         return;
      end if;

      case Widget.Base_Type is
         when mcc.text_entry_types.String_Subtype =>
            Print_Declare_Block;
            P ("mcc.Pad (X, " & Normalize (Widget.Data_Item.all) & "'Length);");
         when mcc.text_entry_types.Unbounded_String =>
            Print_Declare_Block;
            if Widget.Data_Type /= null
              and then Widget.Data_Type.all /= "" then
               declare
                  Last_Dot : constant Natural :=
                     Ada.Strings.Fixed.Index (Widget.Data_Type.all, ".",
                                              Going => Ada.Strings.Backward);
               begin
                  P (Normalize (Widget.Data_Type (1 .. Last_Dot)) &
                     "To_Unbounded_String (X);", Indent => False);
               end;
            else
               P ("Ada.Strings.Unbounded.To_Unbounded_String (X);",
                  Indent => False);
            end if;
         when mcc.text_entry_types.Bounded_String =>
            Print_Declare_Block;
            if Widget.Data_Type /= null
              and then Widget.Data_Type.all /= "" then
               declare
                  Last_Dot : constant Natural :=
                     Ada.Strings.Fixed.Index (Widget.Data_Type.all, ".",
                                              Going => Ada.Strings.Backward);
               begin
                  P (Normalize (Widget.Data_Type (1 .. Last_Dot)) &
                     "To_Bounded_String (X);", Indent => False);
               end;
            else
               mcc.Msg.Error
                 ("Gui.Widget.Textbox.Generate_Read_Action(BOUNDED_STRING): " &
                  "Data type is required " &
                  "(instantiation of Ada.Strings.Bounded.Generic_Bounded_Length)");
            end if;
      end case;
      P ("exception");
      P ("   when E : others =>");
      P ("      mcc.Msg.Error (E, """ & Widget.Name.all & """);");
      P ("end;");
   end Generate_Read_Action;

   -- display the widget to a window
   procedure Display_Widget
     (Widget    : in out Textbox;
      Container : in out mcc.tki.Container.Container'Class)
   is
   begin
      if Widget.The_Widget = null then
         Widget.The_Widget := new mcc.tki.Widget.Textbox.Textbox;
      end if;

      mcc.tki.Widget.Textbox.Create
        (Obj                  =>
           mcc.tki.Widget.Textbox.Textbox (Widget.The_Widget.all),
         Parent               => Container,
         X                    => Widget.x,
         Y                    => Widget.y,
         Width                => Widget.Width,
         Height               => Widget.Height,
         Horizontal_Scrollbar => Widget.HZ_Scroll,
         Vertical_Scrollbar   => Widget.VR_Scroll);

      if Widget.Color.FG /= mcc.tki.Colors.Default then
         mcc.tki.Widget.Set_Foreground_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.tki.Colors.Named_Color (Widget.Color.FG));
      end if;
      if Widget.Color.BG /= mcc.tki.Colors.Default then
         mcc.tki.Widget.Set_Background_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.tki.Colors.Named_Color (Widget.Color.BG));
      end if;

      Display_Widget (GUI_Widget (Widget), Container);

   exception
      when E : others =>
         mcc.Msg.Error (E, "Can't display " & Widget.Name.all);

   end Display_Widget;

   procedure Set_Properties (Widget : in out Textbox) is
   begin
      -- fill in by copying to global
      mcc.text_entry_types.Default_Base_Type := Widget.Base_Type;

      Textbox_dialog_window.Generate_and_Fill_Window;

      Widget.Properties      :=
        Textbox_dialog_window.Textbox_dialog_window'Access;
      Widget.Name_Entry      := Textbox_dialog_window.entry1'Access;
      Widget.X_Entry         := Textbox_dialog_window.entry2'Access;
      Widget.Y_Entry         := Textbox_dialog_window.entry3'Access;
      Widget.Width_Entry     := Textbox_dialog_window.entry4'Access;
      Widget.Height_Entry    := Textbox_dialog_window.entry5'Access;
      Widget.VR_Scroll_Check := Textbox_dialog_window.vertical'Access;
      Widget.HZ_Scroll_Check := Textbox_dialog_window.horizontal'Access;
      Widget.Color.FG_DD        := Textbox_dialog_window.fg_dd'Access;
      Widget.Color.BG_DD        := Textbox_dialog_window.bg_dd'Access;
      Widget.Data_Item_Entry    := Textbox_dialog_window.dataitem'Access;
      Widget.Base_Type_Dropdown := Textbox_dialog_window.base_dropdown'Access;
      Widget.Data_Type_Entry    := Textbox_dialog_window.datatype'Access;

      Set_Properties (GUI_Widget (Widget));

      if state.Get_Current_Window.Novice_Mode then
         mcc.tki.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Data_Item_Entry.all,
            Text => "disabled for novice");
         mcc.tki.Widget.Text_Entry.Disable (Widget.Data_Item_Entry.all);
         mcc.tki.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Data_Type_Entry.all,
            Text => "disabled for novice");
         mcc.tki.Widget.Text_Entry.Disable (Widget.Data_Type_Entry.all);
      else
         if Widget.Data_Item /= null then
            mcc.tki.Widget.Text_Entry.Set_Text
              (Obj  => Widget.Data_Item_Entry.all,
               Text => Widget.Data_Item.all);
         end if;

         if Widget.Data_Type /= null then
            mcc.tki.Widget.Text_Entry.Set_Text
              (Obj  => Widget.Data_Type_Entry.all,
               Text => Widget.Data_Type.all);
         end if;
      end if;

      --vertical scrollbar?
      if Widget.VR_Scroll then
         mcc.tki.Widget.Button.Check.Select_Check
           (Obj => Widget.VR_Scroll_Check.all);
      else
         mcc.tki.Widget.Button.Check.Unselect_Check
           (Obj => Widget.VR_Scroll_Check.all);
      end if;

      --horizontal scrollbar?
      if Widget.HZ_Scroll then
         mcc.tki.Widget.Button.Check.Select_Check
           (Obj => Widget.HZ_Scroll_Check.all);
      else
         mcc.tki.Widget.Button.Check.Unselect_Check
           (Obj => Widget.HZ_Scroll_Check.all);
      end if;

      mcc.tki.Colors.Fill_Colors (Widget.Color);

   exception
      when E : others =>
         mcc.Msg.Error (E, "Gui.Widget.Textbox.Set_Properties");
   end Set_Properties;

   procedure Apply_Properties (Widget : in out Textbox) is
   begin

      Apply_Properties (GUI_Widget (Widget));

      -- SAG 07.07.2005
      declare
         Data_Itemstr : String :=
            mcc.tki.Widget.Text_Entry.Get_Text (Widget.Data_Item_Entry.all);
      begin
         if Data_Itemstr /= "disabled for novice" then
            Widget.Data_Item := new String'(Data_Itemstr);
         end if;
      end;
      declare
         Data_Type     : String :=
            mcc.tki.Widget.Text_Entry.Get_Text (Widget.Data_Type_Entry.all);
         Base_Type_Pos : Integer;
      begin
         if Data_Type /= "disabled for novice" then
            Widget.Data_Type := new String'(Data_Type);
            Base_Type_Pos    :=
               mcc.tki.Widget.Dropdown.Get_Selected
                 (Widget.Base_Type_Dropdown.all);
            if Base_Type_Pos > 0 then
               Widget.Base_Type :=
                  mcc.text_entry_types.Base_Type'Val (Base_Type_Pos - 1);
            else
               Widget.Base_Type := mcc.text_entry_types.Default_Base_Type;
            end if;
         end if;
      end;

      --Vertical scrollbar?
      Widget.VR_Scroll :=
         mcc.tki.Widget.Button.Check.Is_Checked (Widget.VR_Scroll_Check.all);

      --Horizontal scrollbar?
      Widget.HZ_Scroll :=
         mcc.tki.Widget.Button.Check.Is_Checked (Widget.HZ_Scroll_Check.all);

      mcc.tki.Colors.Read_Colors (Widget.Color);
   end Apply_Properties;

   procedure Check_Properties (Widget : in out Textbox; Ok : out Boolean) is
   begin
      Check_Properties (GUI_Widget (Widget), Ok);

      if not Ok then
         mcc.tki.Bell;
      end if;
   end Check_Properties;

end Gui.Widget.Textbox;
