---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  WIDGET_IO.ADB
--  Description : Read/Write Widgets to file
--
--  Copyright (C) 2000, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with Ada.Characters.Handling;
with gui.Widget;              use type gui.Widget.Widget_Pointer;
with Gui_Enum;                use type Gui_Enum.Keyword;
with File_Helpers;
with Lists_Generic;
with Lists_Generic.Key;
with gui.Widget.Radio_Button;

package body Widget_Io is

   package Listpkg renames Gui.Widget.Widget_List_Package;

   ----------------------------------------------------------
   -- procedure Read_Widgets
   --
   -- initialize widget list
   -- read a keyword
   -- create the appropriate widget or stop on End
   -- read widget and add to list
   -- after stop, verify it said "end widgets"
   ----------------------------------------------------------
   procedure Read_Widgets
     (Window_Name : in String;
      Widget_List : out gui.Widget.Widget_List)
   is
      New_Widget : gui.Widget.Widget_Access;
      Keyword    : Gui_Enum.Keyword;
   begin
      Listpkg.Initialize (Widget_List);

      loop
         File_Helpers.Get_Line;
         Keyword := Gui_Enum.Keyword'Value (File_Helpers.Token (1).all);

         exit when Keyword = Gui_Enum.EndOf;

         New_Widget               := Gui_Enum.Allocate_Widget (Keyword);
         File_Helpers.Token_Index := 2;
         gui.Widget.Read_Widget (New_Widget.all);
         Listpkg.AddToRear (Widget_List, New_Widget);
      end loop;

      Keyword := Gui_Enum.Keyword'Value (File_Helpers.Token (2).all);
      if Keyword /= Gui_Enum.Widgets then
         raise Bad_File;
      end if;
   exception
      when others =>
         raise Bad_File;
   end Read_Widgets;

   procedure Write_Widgets (Widgets : gui.Widget.Widget_List) is
      Traverse : gui.Widget.Widget_Pointer := Listpkg.First (Widgets);
   begin
      File_Helpers.P (Gui_Enum.Img (Gui_Enum.Widgets));
      while not Listpkg.IsPastEnd (Widgets, Traverse)
      loop
         Gui.Widget.Write_Widget (Listpkg.Retrieve (Widgets, Traverse).all);
         Listpkg.GoAhead (Widgets, Traverse);
      end loop;
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.EndOf) & " ");
      File_Helpers.P (Gui_Enum.Img (Gui_Enum.Widgets), Indent => False);
   end Write_Widgets;

   -- helper functions for radio button groups
   function Get_Group (Group : gui.String_Pointer) return String is
   begin
      return Group.all;
   end Get_Group;

   package Group_List_Package is new Lists_Generic (gui.String_Pointer);
   package Group_List_Key_Package is new Group_List_Package.Key (
      GetKey => Get_Group);

   -- mcc 05/23/00
   -- add enumeration type for the radio_groups
   -- along with procedures to read/select.
   procedure Generate_Radio_Groups (Widgets : gui.Widget.Widget_List) is

      -- output an enumeration type for this radio group
      procedure Output_Enumeration (This_Group : in String) is
         Traverse : gui.Widget.Widget_Pointer := Listpkg.First (Widgets);
         Widget   : gui.Widget.Widget_Access;
         Group    : gui.String_Pointer;
         -- put comma before each but first
         First : Boolean := True;
      begin
         File_Helpers.P;
         File_Helpers.P
           ("-- use type " & This_Group & "_Enum as alternative to directly");
         File_Helpers.P
           ("-- accessing buttons in group: " & This_Group);
         File_Helpers.P ("type " & This_Group & "_Enum is (");
         while not Listpkg.IsPastEnd (Widgets, Traverse)
         loop
            Widget := Listpkg.Retrieve (Widgets, Traverse);
            if Widget.all in gui.Widget.Radio_Button.Radio_Button then
               Group :=
                 gui.Widget.Radio_Button.Radio_Button (Widget.all).Group;
               if Ada.Characters.Handling.To_Upper (This_Group) =
                  Ada.Characters.Handling.To_Upper (Group.all)
               then
                  if not First then
                     File_Helpers.P (",", Indent => False);
                  else
                     First := False;
                  end if;
                  File_Helpers.P
                    ("   " & Widget.Name.all & "_Enum", Newline => False);
               end if;
            end if;
            Listpkg.GoAhead (Widgets, Traverse);
         end loop;
         File_Helpers.P (");", Indent => False);
         -- output set and read methods for this group
         File_Helpers.P
           ("procedure Set_Group(Which : in " & This_Group & "_Enum);");
         File_Helpers.P
           ("function Read_Group return " & This_Group & "_Enum;");
      end Output_Enumeration;

      Traverse       : gui.Widget.Widget_Pointer := Listpkg.First (Widgets);
      Group_Position : Group_List_Package.Position;
      Widget         : gui.Widget.Widget_Access;
      Group          : gui.String_Pointer;
      The_List       : Group_List_Package.List;

   begin
      Group_List_Package.Initialize (The_List);
      -- Walk through list accumulating distinct group names
      while not Listpkg.IsPastEnd (Widgets, Traverse)
      loop
         Widget := Listpkg.Retrieve (Widgets, Traverse);
         if Widget.all in gui.Widget.Radio_Button.Radio_Button then
            Group := gui.Widget.Radio_Button.Radio_Button (Widget.all).Group;
            if not Group_List_Key_Package.IsPresent
                     (Group.all,
                      The_List)
            then
               Group_List_Package.AddToRear (The_List, Group);
            end if;
         end if;
         Listpkg.GoAhead (Widgets, Traverse);
      end loop;

      -- Now walk through and output declarations
      Group_Position := Group_List_Package.First (The_List);
      while not Group_List_Package.IsPastEnd (The_List, Group_Position) loop
         Group := Group_List_Package.Retrieve (The_List, Group_Position);
         File_Helpers.P
           (Group.all &
            " : aliased mcc.tki.Widget.Button.Radio.Radio_Group;");
         Group_List_Package.GoAhead (The_List, Group_Position);
         Output_Enumeration (Group.all);
      end loop;
   end Generate_Radio_Groups;

   -- mcc 05/23/00
   -- add enumeration type for the radio_groups
   -- along with procedures to read/select.
   procedure Generate_Radio_Groups_Body (Widgets : gui.Widget.Widget_List) is
      use File_Helpers;

      -- output an enumeration type for this radio group
      procedure Output_Enumeration (This_Group : in String) is
         Traverse : gui.Widget.Widget_Pointer;
         Widget   : gui.Widget.Widget_Access;
         Group    : gui.String_Pointer;
         -- use if for first, elsif for rest in Read_Group
         First : Boolean := True;
      begin
         -- output set and read methods for this group
         NL;
         Set_Indent (1);
         P ("procedure Set_Group(Which : in " & This_Group & "_Enum) is");
         P ("begin");
         Set_Indent (2);
         P ("case Which is");
         Set_Indent (3);
         Traverse := Listpkg.First (Widgets);
         while not Listpkg.IsPastEnd (Widgets, Traverse)
         loop
            Widget := Listpkg.Retrieve (Widgets, Traverse);
            if Widget.all in gui.Widget.Radio_Button.Radio_Button then
               Group :=
                 gui.Widget.Radio_Button.Radio_Button (Widget.all).Group;
               if Ada.Characters.Handling.To_Upper (This_Group) =
                  Ada.Characters.Handling.To_Upper (Group.all)
               then
                  P ("when " & Widget.Name.all & "_Enum =>");
                  P ("   mcc.tki.Widget.Button.Radio.Select_Radio(" &
                     Widget.Name.all & ");");
               end if;
            end if;
            Listpkg.GoAhead (Widgets, Traverse);
         end loop;
         Set_Indent (2);
         P ("end case;");
         Set_Indent (1);
         P ("end Set_Group;");
         NL;
         P ("function Read_Group return " & This_Group & "_Enum is");
         P ("   Selected : mcc.tki.Widget.Button.Radio.Radio_Pointer;");
         P ("   use type mcc.tki.Widget.Button.Radio.Radio_Pointer;");
         P ("begin");
         Set_Indent (2);
         P ("Selected := mcc.tki.Widget.Button.Radio.Get_Selected(" &
            This_Group & ");");
         Traverse := Listpkg.First (Widgets);
         while not Listpkg.IsPastEnd (Widgets, Traverse)
         loop
            Widget := Listpkg.Retrieve (Widgets, Traverse);
            if Widget.all in gui.Widget.Radio_Button.Radio_Button then
               Group :=
                 gui.Widget.Radio_Button.Radio_Button (Widget.all).Group;
               if Ada.Characters.Handling.To_Upper (This_Group) =
                  Ada.Characters.Handling.To_Upper (Group.all)
               then
                  if First then
                     P ("if ", Newline => False);
                     First := False;
                  else
                     P ("elsif ", Newline => False);
                  end if;
                  P ("Selected =  " & Widget.Name.all & "'access then");
                  P ("   return " & Widget.Name.all & "_Enum;");
               end if;
            end if;
            Listpkg.GoAhead (Widgets, Traverse);
         end loop;
         File_Helpers.P ("end if;");
         Set_Indent (1);
         File_Helpers.P ("end Read_Group;");
      end Output_Enumeration;

      Traverse       : gui.Widget.Widget_Pointer := Listpkg.First (Widgets);
      Group_Position : Group_List_Package.Position;
      Widget         : gui.Widget.Widget_Access;
      Group          : gui.String_Pointer;
      The_List       : Group_List_Package.List;

   begin
      Group_List_Package.Initialize (The_List);
      -- Walk through list accumulating distinct group names
      while not Listpkg.IsPastEnd (Widgets, Traverse)
      loop
         Widget := Listpkg.Retrieve (Widgets, Traverse);
         if Widget.all in gui.Widget.Radio_Button.Radio_Button then
            Group := gui.Widget.Radio_Button.Radio_Button (Widget.all).Group;
            if not Group_List_Key_Package.IsPresent
                     (Group.all,
                      The_List)
            then
               Group_List_Package.AddToRear (The_List, Group);
            end if;
         end if;
         Listpkg.GoAhead (Widgets, Traverse);
      end loop;

      -- Now walk through and output declarations
      Group_Position := Group_List_Package.First (The_List);
      while not Group_List_Package.IsPastEnd (The_List, Group_Position) loop
         Group := Group_List_Package.Retrieve (The_List, Group_Position);
         Group_List_Package.GoAhead (The_List, Group_Position);
         Output_Enumeration (Group.all);
      end loop;
   end Generate_Radio_Groups_Body;

   procedure Display_Widgets
     (Window  : in out gui.Window.GUI_Window;
      Widgets : in gui.Widget.Widget_List)
   is
      Traverse : gui.Widget.Widget_Pointer := Listpkg.First (Widgets);
   begin
      while not Listpkg.IsPastEnd (Widgets, Traverse)
      loop
         gui.Widget.Display_Widget
           (Widget    => Listpkg.Retrieve (Widgets, Traverse).all,
            Container => Window.Display_Area);
         Listpkg.GoAhead (Widgets, Traverse);
      end loop;
   end Display_Widgets;

   procedure Put_FG_BG_Color
     (FG_Color, BG_Color : in mcc.tki.Colors.Optional_Color) is
   begin
      File_Helpers.Put (' ' & mcc.tki.Colors.Color_Img (FG_Color));
      File_Helpers.Put (' ' & mcc.tki.Colors.Color_Img (BG_Color));
   end Put_FG_BG_Color;

   procedure Get_FG_BG_Color
     (FG_Color, BG_Color : out mcc.tki.Colors.Optional_Color) is
      Word : Gui.Word_Type;
      Last : Natural;
   begin
      File_Helpers.Get_String (File_Helpers.Token_Index, Word, Last, False);
      FG_Color := mcc.tki.Colors.Optional_Color'Value (Word (1 .. Last));
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      File_Helpers.Get_String (File_Helpers.Token_Index, Word, Last, False);
      BG_Color := mcc.tki.Colors.Optional_Color'Value (Word (1 .. Last));
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
   end Get_FG_BG_Color;

end Widget_Io;
