-------------------------------------------------------------------
--           RAPID - RAPID ADA PORTABLE INTERFACE DESIGNER
--           TCL PEER FOR THE MCC TKI (ToolKit Interface) library
--           Copyright (C) 1999 Martin C. Carlisle.
--
-- RAPID is free software;  you can  redistribute it  and/or modify
-- it under terms of the  GNU General Public License as published
-- by the Free Software  Foundation;  either version 2,  or (at your
-- option) any later version.  RAPID is distributed in the hope that
-- it will be useful, but WITHOUT ANY WARRANTY;  without even the
-- implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
-- PURPOSE.  See the GNU General Public License for more details.
-- You should have  received  a copy of the GNU General Public License
-- distributed with RAPID; see file COPYING.  If not, write to the
-- Free Software Foundation,  59 Temple Place - Suite 330,  Boston,
-- MA 02111-1307, USA.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.  This exception does not apply to executables which
-- are GUI design tools, or that could act as a replacement
-- for RAPID.
----------------------------------------------------------------------
-- Adapted from Ch. 7 of "Graphical Applications with Tcl & Tk"
-- by Eric Foster-Johnson
----------------------------------------------------------------------
-- Change log:
----------------------------------------------------------------------
with Ada.Unchecked_Conversion;
with Ada.Exceptions;
with Ada.Text_IO;
with mcc.Common_Dialogs;
package body mcc.tki.Widget.Dropdown is
   Initialized : Boolean := False;
   procedure Initialize;

   ---------------
   -- Add_Entry --
   ---------------

   procedure Add_Entry
     (Obj      : in out Dropdown;
      Location : in Natural;
      Text     : in String)
   is
   begin
      peer.Eval
        (Obj.My_Peer.Name.all & ".top.frame.list insert " &
         mcc.Img (Location) & " """ & peer.Fix_Quotes (Text) & '"');
   end Add_Entry;

   -----------
   -- Clear --
   -----------

   procedure Clear (Obj : in out Dropdown) is
   begin
      peer.Eval (Obj.My_Peer.Name.all & ".top.frame.list delete 0 end");
   end Clear;

   -----------
   -- Count --
   -----------

   function Count (Obj : in Dropdown) return Natural is
      Result : Integer;
   begin
      peer.Eval
        ("set index [" &
         Obj.My_Peer.Name.all &
         ".top.frame.list" &
         " index end]");
      peer.Get_Value (Name => "index", Result => Result);
      return Result;
   exception
      when others =>
         return 0;
   end Count;

   ------------
   -- Create --
   ------------

   procedure Create
     (Obj         : in out Dropdown;
      Parent      : in mcc.tki.Container.Container'Class;
      X           : in Integer;
      Y           : in Integer;
      Width       : in Natural;
      Height      : in Natural;
      Number_Rows : in Natural)
   is
      type Parent_Access is access constant mcc.tki.Container.Container'Class;
      function Convert is new Ada.Unchecked_Conversion (
         Parent_Access,
         mcc.tki.Container.Container_Pointer);
   begin
      if not Initialized then
         Initialize;
         Initialized := True;
      end if;

      Obj.My_Peer := peer.Create_Peer (mcc.tki.Container.Get_Peer (Parent));
      peer.Eval
        ("DropListCreate " & Obj.My_Peer.Name.all &
         " """" " & mcc.Img (Width) & " " & mcc.Img (Number_Rows) &
         " " & peer.Undot_Name (Obj.My_Peer.Name.all) & " """" ");

      peer.Eval
        ("place " & Obj.My_Peer.Name.all &
         " -anchor nw" &
         " -x " & mcc.Img (X) &
         " -y " & mcc.Img (Y) &
         " -width " & mcc.Img (Width) &
         " -height " & mcc.Img (Height));

      -- Since most of window is subwidget, set its bindings to
      -- that of parent.  This greatly simplifies the implementation
      -- of mouse and key bindings in mcc.tki
      peer.Eval
        ("bindtags " &
         Obj.My_Peer.Name.all &
         ".ent ""[bindtags " &
         Obj.My_Peer.Name.all &
         ".ent] " &
         Obj.My_Peer.Name.all &
         """");
      Obj.Parent := Convert (Parent'Unchecked_Access);
   end Create;

   ------------------------------------------------
   -- procedure Delete_Entry
   --
   -- Delete a range of entries
   ------------------------------------------------
   procedure Delete_Entry
     (Obj   : in out Dropdown;
      Start : in Natural;
      Stop  : in Natural)
   is
   begin
      peer.Eval
        (Obj.My_Peer.Name.all & ".top.frame.list delete " &
         mcc.Img (Start - 1) & ' ' & mcc.Img (Stop - 1));
   end Delete_Entry;

   ------------------
   -- Get_Selected --
   ------------------

   function Get_Selected (Obj : in Dropdown) return Natural is
      Result  : Integer;
      SResult : String (1 .. 50);
      Last    : Natural;
   begin
      peer.Eval
        ("set index [" &
         Obj.My_Peer.Name.all &
         ".top.frame.list" &
         " curselection]");
      peer.Get_Value (Name => "index", Result => Result);
      return Result + 1;
   exception
      when e : others =>
         return 0;
   end Get_Selected;

   -----------------
   -- Select_Item --
   -----------------

   procedure Select_Item (Obj : in out Dropdown; Number : in Natural) is
      use type peer.String_Pointer;
   begin
      if Obj.My_Peer.Name = null then
         Ada.Text_IO.Put_Line
           ("mcc.tki.Widget.Dropdown.Select_Item: Obj.My_Peer.Name is null");
         return;
      end if;
      peer.Eval
        (Obj.My_Peer.Name.all & ".top.frame.list selection clear 0 end ");
      peer.Eval
        (Obj.My_Peer.Name.all &
         ".top.frame.list selection set " &
         Integer'Image (Number - 1));
      peer.Eval
        ("catch {" & ASCII.LF &
         "set selected [list_selected " &
         Obj.My_Peer.Name.all & ".top.frame.list]" & ASCII.LF &
         "if { $selected != """" } {" & ASCII.LF &
      --
      -- Put item into entry widget.
      --
         Obj.My_Peer.Name.all & ".ent configure -state normal" & ASCII.LF &
         Obj.My_Peer.Name.all & ".ent delete 0 end" & ASCII.LF &
         Obj.My_Peer.Name.all & ".ent insert 0 ""$selected""" & ASCII.LF &
         Obj.My_Peer.Name.all & ".ent configure -state disabled" & ASCII.LF &
         Obj.My_Peer.Name.all & ".ent configure -disabledforeground black" &
         ASCII.LF &
         "}" & ASCII.LF &
         "}");
   end Select_Item;

   procedure Initialize is
      LF : constant Character := ASCII.LF;
   begin
      -- Drop-down list code.
      --
      -- The main procedure to use is DropListCreate.
      -- The test_drop_list procedure shows how to use
      -- the drop list. The rest of the procedures are
      -- private to the drop list.
      --

      -- Creates a drop-down list and a few interface widgets
      -- (label, entry and drop-button) from your supplied basename.
      --
      -- basename = base name for frame that holds entry widget.
      -- text = text for label widget.
      -- height = number of visible lines in list.
      -- width = number of columns in list and entry widget.
      -- variable = name of variable that gets the selected value.
      -- initial_value = first value.
      --
      -- Returns name of actual listbox widget,
      -- so you can fill in the listbox.
      --
      peer.Eval
        ("proc DropListCreate { " & LF &
         "basename text width height variable initial_value } {" &
         "upvar #0 $variable var" & LF &
         "set var ""$initial_value""" & LF &
      -- Name of top-level widget to create.
         "set top $basename.top" & LF &

      --
      -- Widgets to enter data.
      --
         "frame $basename -bd 0" & LF &
         "label $basename.lbl -text $text -anchor e" & LF &
         "entry $basename.ent -width $width -state disabled" & LF &
         "$basename.ent insert 0 ""$initial_value""" & LF &
         "DropButton $basename.drop $basename.top $basename.ent" & LF &
         "bind $basename.ent <Return> " &
         """DropListSetVal $basename.ent $variable""" & LF &
         "bind $basename.ent <Key-Escape> ""wm withdraw $top""" & LF &
         "pack $basename.lbl -side left -fill y" & LF &
         "pack $basename.drop -side right -fill y" & LF &
         "pack $basename.ent -side left -expand 1 -fill y" & LF &

      --
      -- Drop-list is a top-level temporary window.
      --
         "toplevel $top -cursor top_left_arrow" & LF &
         "wm overrideredirect $top 1" & LF &
         "wm withdraw $top " & LF &
      -- Create list
         "set frm $top.frame" & LF &
         "frame $frm -bd 4 -relief sunken -width $width" & LF &
      -- export selection 0 allows me to have more than 1
         "listbox $frm.list -height $height " &
         "-selectmode single -exportselection 0 " &
         "-yscrollcommand ""$frm.scrollbar set"" " & LF &
         "bind $frm.list <Key-Escape> ""wm withdraw $top""" & LF &
      -- Create scrollbar
         "scrollbar $frm.scrollbar " &
         "-command ""$frm.list yview"" " & LF &
         "pack $frm.scrollbar -side right -fill y" & LF &
         "pack $frm.list      -side left " & LF &
         "pack $frm -side top " & LF &
         "bind $frm.list <ButtonRelease-1> " &
         """DropListClick $top $basename.ent $variable""" & LF &
         "}");

      -- Returns selected item for a single-select list.
      peer.Eval
        ("proc list_selected { listname } {" & LF &
         "set indx [$listname curselection]" & LF &
         "if { $indx != """" } {" & LF &
         "set item [$listname get $indx]" & LF &
         "return $item" & LF &
         "} else {" & LF &
         "return """";" & LF &
         "}" & LF &
         "}");

      -- Places value in global variable.
      peer.Eval
        ("proc DropListSetVal { entry variable } {" & LF &
         "upvar #0 $variable var" & LF &
         "set value [$entry get]" & LF &
         "if { $value != """" } {" & LF &
         "set var $value" & LF &
         "}" & LF &
         "}");

      -- Handles click on drop list widget.
      peer.Eval
        ("proc DropListClick { basename entry variable } {" & LF &
         "catch {" & LF &
         "set selected [list_selected $basename.frame.list]" & LF &
         "if { $selected != """" } {" & LF &
      --
      -- Put item into entry widget.
      --
         "$entry configure -state normal" & LF &
         "$entry delete 0 end" & LF &
         "$entry insert 0 ""$selected""" & LF &
         "$entry configure -state disabled" & LF &
         "DropListSetVal $entry $variable" & LF &
         "}" & LF &
         "}" & LF &
         "wm withdraw $basename" & LF &
         "}");

      -- Makes drop list visible. Create with DropListCreate.
      peer.Eval
        ("proc ShowDropList { basename associated_widget } {" & LF &
         "set x [winfo rootx $associated_widget]" & LF &
         "set y [winfo rooty $associated_widget]" & LF &
         "set y [expr $y + [winfo height $associated_widget]]" & LF &
         "wm geometry $basename ""+$x+$y""" & LF &
         "update" & LF &
         "wm deiconify $basename" & LF &
         "raise $basename" & LF &
         "focus $basename.frame.list " & LF &
         "}");

      -- Creates a button with a drop-down bitmap.
      peer.Eval
        ("proc DropButton { name toplevel entry } {" & LF &
         "button $name -image dnarrow " &
         "-command ""ShowDropList $toplevel $entry""" & LF &
         "return $name" & LF &
         "}");

      --
      -- Bitmap data for down arrow bitmap.
      --
      peer.Eval
        ("set dnarrow_data """ & LF &
         "#define dnarrow2_width 18" & LF &
         "#define dnarrow2_height 18" & LF &
         "static unsigned char dnarrow2_bits[] = {" & LF & 
"0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xff, 0x00, 0xf8, 0x7f, 0x00," &
         LF & 
"0xf8, 0x7f, 0x00, 0xf0, 0x3f, 0x00, 0xf0, 0x3f, 0x00, 0xe0, 0x1f, 0x00," &
         LF & 
"0xc0, 0x0f, 0x00, 0xc0, 0x0f, 0x00, 0x80, 0x07, 0x00, 0x80, 0x07, 0x00," &
         LF & 
"0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xff, 0x00," &
         LF &
         "0xfc, 0xff, 0x00, 0x00, 0x00, 0x00};" &
         LF &
         """" &
         LF &
         "image create bitmap dnarrow -data $dnarrow_data");
   end Initialize;

end mcc.tki.Widget.Dropdown;
