-------------------------------------------------------------------
--           RAPID - RAPID ADA PORTABLE INTERFACE DESIGNER
--           TCL PEER FOR THE MCC TKI (ToolKit Interface) library
--           Copyright (C) 1999 Martin C. Carlisle.
--
-- RAPID is free software;  you can  redistribute it  and/or modify
-- it under terms of the  GNU General Public License as published
-- by the Free Software  Foundation;  either version 2,  or (at your
-- option) any later version.  RAPID is distributed in the hope that
-- it will be useful, but WITHOUT ANY WARRANTY;  without even the
-- implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
-- PURPOSE.  See the GNU General Public License for more details.
-- You should have  received  a copy of the GNU General Public License
-- distributed with RAPID; see file COPYING.  If not, write to the
-- Free Software Foundation,  59 Temple Place - Suite 330,  Boston,
-- MA 02111-1307, USA.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.  This exception does not apply to executables which
-- are GUI design tools, or that could act as a replacement
-- for RAPID.
------------------------------------------------------------------------------
with Ada.Unchecked_Conversion;
package body mcc.tki.Widget.Listbox is

   ---------------
   -- Add_Entry --
   ---------------

   procedure Add_Entry
     (Obj      : in out Listbox;
      Location : in Natural;
      Text     : in String)
   is
   begin
      peer.Eval
        (Obj.My_Peer.Name.all &
         ".list insert " &
         Integer'Image (Location) &
         " """ &
         peer.Fix_Quotes (Text) &
         '"');
      peer.Eval
        (Obj.My_Peer.Name.all & ".list see " & Integer'Image (Location));
   end Add_Entry;

   -----------
   -- Clear --
   -----------

   procedure Clear (Obj : in out Listbox) is
   begin
      peer.Eval (Obj.My_Peer.Name.all & ".list delete 0 end");
   end Clear;

   -----------
   -- Count --
   -----------

   function Count (Obj : in Listbox) return Natural is
      Result : Integer;
   begin
      peer.Eval
        ("set index [" & Obj.My_Peer.Name.all & ".list" & " index end]");
      peer.Get_Value (Name => "index", Result => Result);
      return Result;
   exception
      when others =>
         return 0;
   end Count;

   ------------
   -- Create --
   ------------

   procedure Create
     (Obj                  : in out Listbox;
      Parent               : in mcc.tki.Container.Container'Class;
      X                    : in Integer;
      Y                    : in Integer;
      Width                : in Natural;
      Height               : in Natural;
      Horizontal_Scrollbar : in Boolean;
      Vertical_Scrollbar   : in Boolean)
   is
      type Parent_Access is access constant mcc.tki.Container.Container'Class;
      function Convert is new Ada.Unchecked_Conversion (
         Parent_Access,
         mcc.tki.Container.Container_Pointer);
   begin
      Obj.My_Peer := peer.Create_Peer (mcc.tki.Container.Get_Peer (Parent));
      peer.Eval ("frame " & Obj.My_Peer.Name.all);
      peer.Eval
        ("listbox " & Obj.My_Peer.Name.all & ".list " &
         " -width " & mcc.Img (Width) &
         " -height " & mcc.Img (Height) &
         " -exportselection 0");

      if Vertical_Scrollbar then
         peer.Eval
           ("scrollbar " & Obj.My_Peer.Name.all & ".vscrl" &
            " -orient vertical -command """ &
            Obj.My_Peer.Name.all & ".list yview""");
         peer.Eval
           (Obj.My_Peer.Name.all & ".list configure" &
            " -yscrollcommand """ &
            Obj.My_Peer.Name.all & ".vscrl set""");
         --Pack options for vertical scrollbar
         peer.Eval
           ("pack " & Obj.My_Peer.Name.all & ".vscrl -side right -fill y");
      end if;

      if Horizontal_Scrollbar then
         peer.Eval
           ("scrollbar " &
            Obj.My_Peer.Name.all &
            ".hscrl" &
            " -orient horizontal -command """ &
            Obj.My_Peer.Name.all &
            ".list xview""");
         peer.Eval
           (Obj.My_Peer.Name.all &
            ".list configure" &
            " -xscrollcommand """ &
            Obj.My_Peer.Name.all &
            ".hscrl set""");
         --Pack options for vertical scrollbar
         peer.Eval
           ("pack " & Obj.My_Peer.Name.all & ".hscrl -side bottom -fill x");
      end if;

      --Create Listbox with no scrollbars
      if (not Horizontal_Scrollbar) and (not Vertical_Scrollbar) then
         --Pack options for listbox
         peer.Eval ("pack " & Obj.My_Peer.Name.all & ".list ");
      else
         --Pack options for listbox
         peer.Eval ("pack " & Obj.My_Peer.Name.all & ".list" & " -side left");
      end if;

      peer.Eval
        ("place " & Obj.My_Peer.Name.all &
         " -anchor nw" &
         " -x " & mcc.Img (X) &
         " -y " & mcc.Img (Y) &
         " -width " & mcc.Img (Width) &
         " -height " & mcc.Img (Height));

      -- Since most of window is subwidget, set its bindings to
      -- that of parent.  This greatly simplifies the implementation
      -- of mouse and key bindings in mcc.tki
      peer.Eval
        ("bindtags " &
         Obj.My_Peer.Name.all &
         ".list ""[bindtags " &
         Obj.My_Peer.Name.all &
         ".list] " &
         Obj.My_Peer.Name.all &
         """");
      Obj.Parent := Convert (Parent'Unchecked_Access);
   end Create;

   ------------------------------------------------
   -- procedure Delete_Entry
   --
   -- Delete a range of entries
   ------------------------------------------------
   procedure Delete_Entry
     (Obj   : in out Listbox;
      Start : in Natural;
      Stop  : in Natural)
   is
   begin
      peer.Eval
        (Obj.My_Peer.Name.all &
         ".list delete " &
         Integer'Image (Start - 1) &
         Integer'Image (Stop - 1));
   end Delete_Entry;

   ------------------
   -- Get_Selected --
   ------------------

   function Get_Selected (Obj : in Listbox) return Natural is
      Result : Integer;
   begin
      peer.Eval
        ("set index [" & Obj.My_Peer.Name.all & ".list" & " curselection]");
      peer.Get_Value (Name => "index", Result => Result);
      return Result + 1;
   exception
      when others =>
         return 0;
   end Get_Selected;

   function Get_Item (Obj : in Listbox; Number : in Natural) return String is
      length : Natural;
   begin
      peer.Eval
        ("set result [" &
         Obj.My_Peer.Name.all &
         ".list" &
         " get " &
         Integer'Image (Number - 1) &
         "]");
      peer.Eval ("set string_length [string length $result]");
      peer.Get_Value (Name => "string_length", Result => length);
      declare
         result : String (1 .. length);
      begin
         peer.Get_Value (Name => "result", Result => result, Last => length);
         return result;
      end;
   exception
      when others =>
         return "";
   end Get_Item;

   -----------------
   -- Select_Item --
   -----------------

   procedure Select_Item (Obj : in out Listbox; Number : in Natural) is
   begin
      peer.Eval (Obj.My_Peer.Name.all & ".list selection clear 0 end ");
      peer.Eval
        (Obj.My_Peer.Name.all &
         ".list selection set " &
         Integer'Image (Number - 1));
   end Select_Item;

end mcc.tki.Widget.Listbox;
