/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <argz.h>
#include <argp.h>
#include "opts.h"
#include "xvasprintf.h"
#include "trim.h"
#ifdef HAVE_LIBREADLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif
#include "gettext-more.h"
#include <curl/curl.h>
#include "reddit.h"
#include "reddit_priv.h"
#include "stories.h"

#include "upvote.h"
#include "downvote.h"
#include "discuss.h"
#include "gosub.h"
#include "return.h"
#include "ls.h"
#include "reload.h"
#include "next.h"
#include "prev.h"
#include "view.h"
#include "share.h"
#include "whats.h"
#include "login.h"
#include "submit.h"

char *htmlentities(const char *);
static int reddit_interactive (struct reddit_state_t *, struct reddit_options_t *);
static int reddit_set_history (struct reddit_state_t*, char *);

#define REDDIT_PROMPT           "reddit> "
#define WARRANTY_KEYWORD        "warranty"
#define QUIT_KEYWORD            "quit"
#define HISTORY_KEYWORD         "history"
#define BROWSER_KEYWORD         "browser"
#define UPVOTE_KEYWORD          "upvote"
#define DOWNVOTE_KEYWORD        "downvote"
#define SHARE_KEYWORD           "share"
#define DISCUSS_KEYWORD         "discuss"
#define NEXT_KEYWORD            "next"
#define PREV_KEYWORD            "prev"
#define WHATS_KEYWORD           "what's"
#define HELP_KEYWORD            "help"
#define VIEW_KEYWORD            "view"
#define GOSUB_KEYWORD           "gosub"
#define RETURN_KEYWORD          "return"
#define LS_KEYWORD              "ls"
#define RELOAD_KEYWORD          "reload"
#define LOGIN_KEYWORD           "login"
#define SUBMIT_KEYWORD           "submit"


void 
reddit_line (struct reddit_state_t *state)
{
  int i = 0;
  int columns = 80;
  if (state->columns != 0)
    columns = state->columns;
  for (;i < columns; i++)
    fprintf (state->out, "-");
  fprintf (state->out, "\n");
}

void 
reddit_banner (struct reddit_state_t *state, char *title)
{
  char *edition = title;
  if (edition == NULL)
    edition = _("Canadian Edition!");
  fprintf (state->out, "    _o    ____  ____  ___   ___   _  ___\n");
  fprintf (state->out, "  _/_     |__/  |___  |  \\  |  \\  |   |\n");
  fprintf (state->out, " (o_o)    |  \\  |___  |__/  |__/  |   |  [%s]\n",
           edition);
  fprintf (state->out, "\n");
  fprintf (state->out, "     [%s]    [%s]    [%s]    [%s]    [%s]\n", 
           state->whats == REDDIT_WHATS_HOT ? _("what's hot") : _("hot"), 
           state->whats == REDDIT_WHATS_NEW ? _("what's new") : _("new"), 
           state->whats == REDDIT_WHATS_CONTROVERSIAL ? 
           _("most controversial") :_("controversial"), 
           state->whats == REDDIT_WHATS_TOP ? _("top scoring") : _("top"),
           state->whats == REDDIT_WHATS_SAVED ? _("*saved*") : _("saved")
           );
  fprintf (state->out, "\n");
  reddit_line (state);
}

static void
reddit_welcome (FILE *fileptr)
{
  fprintf (fileptr, "%s %s\n", PACKAGE, VERSION);
  fprintf (fileptr, "Copyright (C) 2011 Ben Asselstine\n");
  fprintf (fileptr, "%s\n", _("REDDIT is a registered trademark "
                              "of Advance Magazine Publishers Inc."));
  fprintf (fileptr, "%s\n", 
           _("This is free software with ABSOLUTELY NO WARRANTY."));
  fprintf (fileptr, _("For warranty details type `%s'.\n"), 
           _(WARRANTY_KEYWORD));
}

static int
reddit_history(struct reddit_state_t *state)
{
#ifdef HAVE_LIBREADLINE
  if (!history_is_stifled())
    fprintf (state->out, "%d\n", -1);
  else
    fprintf (state->out, "%d\n", history_max_entries);
#else
  fprintf (state->out, "0\n");
#endif
  return 0;
}

static int
reddit_browser (struct reddit_state_t *state)
{
  fprintf (state->out, "%s\n", state->browser_binary);
  return 0;
}

static int
reddit_set_browser (struct reddit_state_t *state, char *value)
{
  free (state->browser_binary);
  FILE *fileptr = fopen (value, "r");
  if (fileptr)
    {
      state->browser_binary = strdup (value);
      fclose (fileptr);
    }
  else
    fprintf (state->out, _("The program %s does not exist.\n"), value);
  return 0;
}


static int
reddit_warranty (struct reddit_state_t *state)
{
  fprintf (state->out, "%s %s\n", PACKAGE, VERSION);
  fprintf (state->out, "Copyright (C) 2011 Ben Asselstine\n");
  fprintf (state->out, "\n");
  fprintf (state->out, _("\
    This program is free software; you can redistribute it and/or modify\n\
    it under the terms of the GNU General Public License as published by\n\
    the Free Software Foundation; either version 2 of the License, or\n\
    (at your option) any later version.\n\
\n\
    This program is distributed in the hope that it will be useful,\n\
    but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n\
    GNU General Public License for more details.\n\
\n\
    You should have received a copy of the GNU General Public License \n\
    along with this program. If not, write to\n\
\n\
    The Free Software Foundation, Inc.\n\
    51 Franklin Street, Fifth Floor\n\
    Boston, MA 02110-1301  USA\n\n\
"));
  return 0;
}

static int 
get_command (char *const cmds[], struct argp* const parsers[], char *line, char **value, struct argp **argp)
{
  char *ptr = line;
  int retval = getsubopt (&ptr, cmds, value);
  if (retval >= 0)
    {
      if (parsers[retval] == NULL)
        {
          if (*value)
            {
              char *newvalue = strdup (*value);
              *value = newvalue;
            }
          return retval;
        }
    }

  *value = NULL;
  int i = 0;
  char *cmd = NULL;
  retval = sscanf (line, "%ms", &cmd);
  for (i = 0; cmds[i] != NULL; i++)
    if (retval == 1 && strcasecmp (cmd, cmds[i]) == 0)
      break;
  free (cmd);
  *argp = parsers[i];
  return i;
}

static void
reddit_unknown_command (struct reddit_state_t *state)
{
  fprintf (state->out, _("Unknown command `%s'\n"), state->command);
}

static void 
make_command_line (char *cmd, int *argc, char ***argv)
{
  char *s1 = strdup (cmd);
  if (s1)
    {
      char *term;
      *argv = NULL;
      *argc = 0;
      for (term = strtok (s1, " \t"); term != NULL; term = strtok (NULL, " \t"))
        {
          *argv = (char **) realloc (*argv, sizeof (char *) * ((*argc) + 1));
          if (*argv)
            (*argv)[*argc] = strdup (term);
          (*argc)++;
        }
      *argv = (char **) realloc (*argv, sizeof (char *) * ((*argc) + 1));
      if (*argv)
        (*argv)[(*argc)] = 0;
      free (s1);
    }
}

int
argp_help_check (int argc, char **argv)
{
  int i = 0;
  for (; i < argc; i++)
    {
      if (strcmp (argv[i], "-?") == 0)
        return 1;
      if (strcmp (argv[i], "--help") == 0 || strcmp (argv[i], "--usage") == 0)
        return 1;
    }
  return 0;
}

static void 
reddit_help (struct reddit_state_t *state, char * const cmds[], struct argp* const parsers[])
{
  fprintf (state->out, _("List of Commands:\n"));
  int i = 0;
  for (; cmds[i] != NULL; i++)
    {
      if (strcmp (cmds[i], HELP_KEYWORD) == 0 ||
          strcmp (cmds[i], BROWSER_KEYWORD) == 0 ||
          strcmp (cmds[i], HISTORY_KEYWORD) == 0)
        continue;
      fprintf(state->out, "\t%s %s\n", cmds[i], 
              parsers[i] ? parsers[i]->args_doc : "");
    }
}

void
reddit_reshow_stories (struct reddit_state_t *state)
{
  reddit_show_stories (state);
  if (!reddit_is_in_subreddit (state))
    reddit_banner (state, state->subreddit);
  else
    reddit_banner (state, NULL);
}

static int
reddit_parse_command (struct reddit_state_t *state, char *line, int *quit)
{
  int err = 0;
  enum 
    {
      HISTORY = 0,
      BROWSER,
      LOGIN,
      SUBMIT,
      VIEW,
      UPVOTE,
      DOWNVOTE,
      SHARE,
      DISCUSS,
      WHATS,
      NEXT,
      PREV,
      GOSUB, 
      GOBACK, 
      LS,
      RELOAD,
      HELP,
      WARRANTY,
      QUIT,
      THE_END
    };
  char *const cmds[] = 
    {
      [HISTORY] = HISTORY_KEYWORD,
      [BROWSER] = BROWSER_KEYWORD,
      [LOGIN] = LOGIN_KEYWORD,
      [SUBMIT] = SUBMIT_KEYWORD,
      [VIEW] = VIEW_KEYWORD,
      [UPVOTE] = UPVOTE_KEYWORD,
      [DOWNVOTE] = DOWNVOTE_KEYWORD,
      [SHARE] = SHARE_KEYWORD,
      [DISCUSS] = DISCUSS_KEYWORD,
      [WHATS] = WHATS_KEYWORD,
      [NEXT] = NEXT_KEYWORD,
      [PREV] = PREV_KEYWORD,
      [GOSUB] = GOSUB_KEYWORD,
      [GOBACK] = RETURN_KEYWORD,
      [LS] = LS_KEYWORD,
      [RELOAD] = RELOAD_KEYWORD,
      [HELP] = HELP_KEYWORD,
      [WARRANTY] = WARRANTY_KEYWORD,
      [QUIT] = QUIT_KEYWORD,
      [THE_END] = NULL
    };
  struct argp * const parsers[] =
    {
      [HISTORY] = NULL,
      [BROWSER] = NULL,
      [VIEW] = &reddit_view_argp,
      [LOGIN] = &reddit_login_argp,
      [SUBMIT] = &reddit_submit_argp,
      [UPVOTE] = &reddit_upvote_argp,
      [DOWNVOTE] = &reddit_downvote_argp,
      [SHARE] = &reddit_share_argp,
      [DISCUSS] = &reddit_discuss_argp,
      [WHATS] = &reddit_whats_argp,
      [NEXT] = &reddit_next_argp,
      [PREV] = &reddit_prev_argp,
      [GOSUB] = &reddit_gosub_argp,
      [GOBACK] = &reddit_return_argp,
      [LS] = &reddit_ls_argp,
      [RELOAD] = &reddit_reload_argp,
      [HELP] = NULL,
      [WARRANTY] = NULL,
      [QUIT] = NULL,
      [THE_END] = NULL
    };

  if (strcmp (line, "\n") == 0 || strcmp (line, "\r\n") == 0 || *line == 0)
    return 0;

  state->command = line;
  int argc = 0;
  char **argv = NULL;
  //make_argv (line, &argc, &argv);
  make_command_line (line, &argc, &argv);
  char *value = NULL;
  struct argp *argp = NULL;
  int command = get_command (cmds, parsers, line, &value, &argp);
  switch (command)
    {
    case HISTORY:
      if (!value)
        err = reddit_history (state);
      else
        err = reddit_set_history (state, value);
      break;
    case BROWSER:
      if (!value)
        err = reddit_browser (state);
      else
        err = reddit_set_browser (state, value);
      break;
    case WARRANTY:
      if (!value)
        err = reddit_warranty (state);
      else
        reddit_unknown_command (state);
      break;
    case UPVOTE:
      if (!value && argp)
        err = reddit_upvote_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case DOWNVOTE:
      if (!value && argp)
        err = reddit_downvote_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case VIEW:
      if (!value && argp)
        err = reddit_view_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case SHARE:
      if (!value && argp)
        err = reddit_share_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case GOSUB:
      if (!value && argp)
        err = reddit_gosub_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case GOBACK:
      if (!value && argp)
        err = reddit_return_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case HELP:
      if (!value)
        reddit_help (state, cmds, parsers);
      else
        reddit_unknown_command (state);
      break;
    case LOGIN:
      if (!value && argp)
        err = reddit_login_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      if (err)
        err=0;
      break;
    case SUBMIT:
      if (!value && argp)
        err = reddit_submit_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case NEXT:
      if (!value && argp)
        err = reddit_next_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case PREV:
      if (!value && argp)
        err = reddit_prev_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case WHATS:
      if (!value && argp)
        err = reddit_whats_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case DISCUSS:
      if (!value && argp)
        err = reddit_discuss_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case LS:
      if (!value && argp)
        err = reddit_ls_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case RELOAD:
      if (!value && argp)
        err = reddit_reload_parse_argp (state, argc, argv, argp);
      else
        reddit_unknown_command (state);
      break;
    case QUIT:
      if (!value)
        {
          if (quit)
            *quit = 1;
        }
      else
        reddit_unknown_command (state);
      break;
    default:
      break;
    }
  free (value);
  int i = 0;
  for (i = 0; i < argc; i++)
    free (argv[i]);
  free (argv);
  return err;
}

static int
reddit_parse_untrimmed_command (struct reddit_state_t *state, char *line, int *quit)
{
  if (strcmp (line, "\n") == 0 || strcmp (line, "\r\n") == 0 || *line == 0)
    return 0;
  char *trimmed_line = trim (line);
  int err = reddit_parse_command (state, trimmed_line, quit);
  free (trimmed_line);
  return err;
}

#ifndef HAVE_LIBREADLINE
static char *
myreadline (char *prompt)
{
  char *line = NULL;
  size_t len = 0;
  fprintf (stdout, "%s", prompt);
  fflush (stdout);
  ssize_t bytesread = getline (&line, &len, stdin);
  if (bytesread == -1)
    return NULL;
  char *tmp = strrchr (line, '\n');
  if (tmp)
    tmp[0] = '\0';
  return line;
}

static int
reddit_set_history (struct reddit_state_t *state, char *value)
{
  fprintf (state->out, _("Error: %s was not compiled with GNU readline.  "
                      "A history of commands cannot be kept.\n"), PACKAGE);
  return 0;
}

#else 

static int
reddit_set_history (struct reddit_state_t *state, char *value)
{
  char *end = NULL;
  long int new_history_length = strtol (value, &end, 0);
  if ((end == NULL) || (end == value))
    fprintf (state->out, _("Error `%s' isn't a number.\n"), value);
  else if (new_history_length < -1)
    fprintf (state->out, _("Error `%s' is an invalid value.\n"), value);
  else
    {
      if (new_history_length == -1 && history_is_stifled())
        unstifle_history();
      else if (new_history_length == -1 && !history_is_stifled())
        ;
      else
        stifle_history (new_history_length);
    }
  return 0;
}
#endif

static int
reddit_interactive (struct reddit_state_t *state, struct reddit_options_t *arguments)
{
#ifdef HAVE_LIBREADLINE
      using_history ();
#endif
  int err = 0;
  int quit = 0;
  char *line;
  while (1)
    {
#ifdef HAVE_LIBREADLINE
      line = readline (REDDIT_PROMPT);
      if (line)
	add_history (line);
      rl_get_screen_size (NULL, &state->columns);
#else
      line = myreadline (REDDIT_PROMPT);
#endif

      err = reddit_parse_untrimmed_command (state, line, &quit);

      free (line);
      if (quit)
        break;
      if (err)
	break;

    }
  return err;
}

char *
reddit_construct_url (struct reddit_state_t *state, char *page)
{
  char *mypage = page;
  if (mypage == NULL)
    mypage = state->page;
  char *whats = NULL;
  if (state->whats != REDDIT_WHATS_HOT)
    whats = reddit_whats_to_string (state->whats);
  char *url;
  if (state->whats == REDDIT_WHATS_SAVED)
    url = xasprintf ("http://%s/saved/%s", REDDIT_SITE, 
                     mypage ? mypage : "");
  else
    url = xasprintf ("http://%s/%s%s%s%s%s%s", REDDIT_SITE,
                     state->subreddit ? "r/" : "",
                     state->subreddit ? state->subreddit : "",
                     state->subreddit ? "/" : "",
                     whats ? whats : "",
                     whats ? "/" : "",
                     mypage ? mypage : "");
  return url;
}

int 
reddit_load_and_show_stories (struct reddit_state_t *state, char *page, int *response)
{
  int err = 0;
  struct reddit_story_t *stories = NULL;
  char *url = reddit_construct_url (state, page);
  int num_stories = reddit_load_stories (state, url, &stories, response);

  if (num_stories <= 0)
    {
      free (url);
      free (stories);
      if (num_stories == 0)
        err = -1;
      else
        err = num_stories;
    }
  else
    {
      free (url);
      reddit_free_stories (state->stories, state->num_stories);
      state->num_stories = num_stories;
      state->stories = stories;
      free (state->page);
      if (page)
        state->page = strdup (page);
      else
        state->page = NULL;
      reddit_show_stories (state);
    }
  if (reddit_is_in_subreddit (state))
    reddit_banner (state, state->subreddit);
  else
    reddit_banner (state, NULL);
  return err;
}

int
reddit_update_modhash (struct reddit_state_t *state)
{
  struct reddit_story_t *stories = NULL;
  char *url = reddit_construct_url (state, NULL);
  int num_stories = 
    reddit_load_stories (state, url, &stories, NULL);
  free (url);
  if (num_stories > 0)
    {
      reddit_free_stories (state->stories, state->num_stories);
      state->stories = stories;
      state->num_stories = num_stories;
    }
  else
    return -1;
  return 0;
}

int 
reddit (struct reddit_options_t *arguments)
{
  struct reddit_state_t state;
  int err = 0;

  remove (REDDIT_COOKIEJAR);
  memset (&state, 0, sizeof (state));
  state.links_binary = g_find_program_in_path ("links");
  if (state.links_binary == NULL)
    {
      fprintf (stderr, _("Error: Couldn't find the `links' program in your "
                         "path.  It is a required to run this software.\n"));
      return 0;
    }
  state.browser_binary = strdup (state.links_binary);

  curl_global_init (CURL_GLOBAL_ALL);
  state.curl = curl_easy_init();
  curl_easy_setopt (state.curl, CURLOPT_COOKIEFILE, REDDIT_COOKIEJAR);
  curl_easy_setopt (state.curl, CURLOPT_COOKIEJAR, REDDIT_COOKIEJAR);

  if (arguments->quiet == 0)
    reddit_welcome (stdout);

  state.out = stdout;

  if (arguments->username)
    {
      struct reddit_login_options_t login_opts;
      memset (&login_opts, 0, sizeof (login_opts));
      login_opts.username = arguments->username;
      err = reddit_login (&state, &login_opts);
    }

#ifdef HAVE_LIBREADLINE
  rl_initialize ();
  rl_reset_screen_size ();
  rl_get_screen_size (NULL, &state.columns);
#endif

  if (!err)
    reddit_load_and_show_stories (&state, NULL, NULL);

  if (!err)
    err = reddit_interactive (&state, arguments);
  curl_easy_cleanup(state.curl);
  curl_global_cleanup();
  free (state.username);
  free (state.links_binary);
  free (state.browser_binary);
  return err;
}

