/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <argp.h>
#include <argz.h>
#include "story.h"
#include "stories.h"
#include "whats.h"
#include "gosub.h"
#include "reddit_priv.h"
#include "xvasprintf.h"
#include "trim.h"
#include "gettext-more.h"

int 
argp_parse_story_id (struct argp_state *state, char *arg, struct reddit_state_t *r)
{
  char *end = NULL;
  unsigned long val = strtoul (arg, &end, 0);
  if (end == NULL || end == arg || (int)val < 0)
    argp_failure (state, 0, 0, _("`%s' isn't a story id."), arg);
  else
    {
      if (r && r->num_stories)
        {
          if (val >= r->stories[0].story_number && 
              val <= r->stories[r->num_stories-1].story_number)
            return val;
          else
            {
              argp_failure 
                (state, 0, 0, 
                 _("That story number isn't between %d and %d."),
                 r->stories[0].story_number,
                 r->stories[r->num_stories-1].story_number);
              argp_state_help (state, state->err_stream, ARGP_HELP_SEE);
            }
        }
      else
        return val;
    }
  return -1;
}

char * 
argp_parse_story_help_filter (int key, const char *text, struct reddit_state_t *s)
{
  if (key == ARGP_KEY_HELP_POST_DOC)
    {
      if (s && s->num_stories)
        return xasprintf 
          (_("Valid story numbers are between %d and %d inclusive.\n"),
           s->stories[0].story_number, 
           s->stories[s->num_stories-1].story_number);
    }
  return (char *) text;
}

char *
argp_parse_email (struct argp_state *state, char *arg, struct reddit_state_t *r)
{
  return arg;
}

static int 
reddit_load_rank_from_story_chunk (struct reddit_state_t *state, gchar *data)
{
  int rank = -1;
  int num_fields = 0;
  gchar **fields = g_regex_split_simple 
      ("<span class=\"rank\" style=\"width:....ex;\"\\>", data, 0, 0);

  num_fields = g_strv_length (fields) ;
  if (num_fields <= 1)
    return -1;
  else
    rank = atoi (fields[1]);
  g_strfreev (fields);
  return rank;
}

static char *
reddit_load_id_from_story_chunk (struct reddit_state_t *state, gchar *data)
{
  char *id = NULL;
  int retval = sscanf (data, "%ms", &id);
  if (retval != 1)
    return NULL;
  return id;
}

static int
reddit_load_score_from_story_chunk (struct reddit_state_t *state, char *data, int *err)
{
  int score = 0;
  gchar **fields = 
    g_regex_split_simple ("\\<div class=\"score unvoted\"\\>", data, 0, 0);
  int num_fields = g_strv_length (fields) ;
  if (num_fields <= 1)
    *err = 1;
  else
    score = atoi (fields[1]);
  g_strfreev (fields);
  return score;
}

static char *
reddit_load_link_from_story_chunk (struct reddit_state_t *state, gchar *data)
{
  char *link = NULL;
  gchar **fields;
  if (state->logged_in)
    fields = g_regex_split_simple ("<a class=\"title loggedin \" href=\"", data, 0, 0);
  else
    fields = g_regex_split_simple ("<a class=\"title \" href=\"", data, 0, 0);
  int num_fields = g_strv_length (fields);
  if (num_fields <= 1)
    {
      g_strfreev (fields);
      return NULL;
    }
  else
    {
      int retval = sscanf (fields[1], "%ms", &link);
      if (retval != 1)
        {
          g_strfreev (fields);
          return NULL;
        }
      char *quote = strchr (link, '\"');
      if (quote)
        *quote = '\0';
    }
  g_strfreev (fields);
  return link;
}

static char *
reddit_load_title_from_story_chunk (struct reddit_state_t *state, gchar *data)
{
  char *title = NULL;
  gchar **fields;
  if (state->logged_in)
    fields = g_regex_split_simple ("<a class=\"title loggedin \" href=\"", data, 0, 0);
  else
    fields = g_regex_split_simple ("<a class=\"title \" href=\"", data, 0, 0);
  int num_fields = g_strv_length (fields);
  if (num_fields <= 1)
    {
      g_strfreev (fields);
      return NULL;
    }
  else
    {
      char *gt = strchr (fields[1], '>');
      if (!gt)
        {
          g_strfreev (fields);
          return NULL;
        }
      char *lt = strchr (gt, '<');
      if (!lt)
        {
          g_strfreev (fields);
          return NULL;
        }
      *lt = '\0';
      char *text = htmltotext (state, ++gt);
      title = trim (text);
    }
  g_strfreev (fields);
  return title;
}

static char *
reddit_load_provider_from_story_chunk (struct reddit_state_t *state, gchar *data)
{
  char *provider = NULL;
  gchar **fields = 
    g_regex_split_simple ("<span class=\"domain\"\\>", data, 0, 0);
  int num_fields = g_strv_length (fields);
  if (num_fields <= 1)
    {
      g_strfreev (fields);
      return NULL;
    }
  else
    {
      char *gt = strchr (fields[1], '>');
      if (!gt)
        {
          g_strfreev (fields);
          return NULL;
        }
      char *lt = strchr (gt, '<');
      if (!lt)
        {
          g_strfreev (fields);
          return NULL;
        }
      *lt = '\0';
      provider = strdup (++gt);
    }
  g_strfreev (fields);
  return provider;
}

static char *
reddit_load_ago_from_story_chunk (struct reddit_state_t *state, gchar *data)
{
  char *ago = NULL;
  gchar **fields = 
    g_regex_split_simple ("<p class=\"tagline\"", data, 0, 0);
  int num_fields = g_strv_length (fields);
  if (num_fields <= 1)
    {
      g_strfreev (fields);
      return NULL;
    }
  else
    {
      char *gt = strchr (fields[1], '>');
      if (!gt)
        {
          g_strfreev (fields);
          return NULL;
        }
      char *lt = strchr (gt, '<');
      if (!lt)
        {
          g_strfreev (fields);
          return NULL;
        }
      *lt = '\0';
      char *text = htmltotext (state, ++gt);
      ago = trim (text);
      free (text);
    }
  g_strfreev (fields);
  return ago;
}

static char *
reddit_load_submitter_from_story_chunk (struct reddit_state_t *state, gchar *data)
{
  char *submitter = NULL;
  gchar **fields = 
    g_regex_split_simple (" class=\"author ", data, 0, 0);
  int num_fields = g_strv_length (fields);
  if (num_fields <= 1)
    {
      g_strfreev (fields);
      return NULL;
    }
  else
    {
      char *gt = strchr (fields[1], '>');
      if (!gt)
        {
          g_strfreev (fields);
          return NULL;
        }
      char *lt = strchr (gt, '<');
      if (!lt)
        {
          g_strfreev (fields);
          g_strfreev (fields);
          return NULL;
        }
      *lt = '\0';
      submitter = strdup (++gt);
    }
  g_strfreev (fields);
  return submitter;
}

static char *
reddit_load_subreddit_from_story_chunk (struct reddit_state_t *state, gchar *data)
{
  char *subreddit = NULL;
  gchar **fields = 
    g_regex_split_simple (" class=\"subreddit hover\"", data, 0, 0);
  int num_fields = g_strv_length (fields);
  if (num_fields <= 1)
    {
      g_strfreev (fields);
      return NULL;
    }
  else
    {
      char *gt = strchr (fields[1], '>');
      if (!gt)
        {
          g_strfreev (fields);
          return NULL;
        }
      char *lt = strchr (gt, '<');
      if (!lt)
        {
          g_strfreev (fields);
          return NULL;
        }
      *lt = '\0';
      subreddit = strdup (++gt);
    }
  g_strfreev (fields);
  return subreddit;
}

static int
reddit_load_number_of_comments_from_story_chunk (struct reddit_state_t *state, gchar *data)
{
  int num_comments = -1;
  gchar **fields = 
    g_regex_split_simple ("<a class=\"comments\"", data, 0, 0);
  int num_fields = g_strv_length (fields);
  if (num_fields <= 1)
    {
      g_strfreev (fields);
      return -1;
    }
  else
    {
      char *gt = strchr (fields[1], '>');
      if (!gt)
        {
          g_strfreev (fields);
          return -1;
        }
      num_comments = atoi (++gt);
    }
  g_strfreev (fields);
  return num_comments;
}

static char * 
reddit_load_votehash_from_story_chunk (struct reddit_state_t *state, gchar *data)
{
  char *votehash = NULL;
  gchar **fields = 
    g_regex_split_simple ("<div class=\"arrow up\" onclick=\"\\$\\(this\\).vote\\('", 
                          data, 0, 0);
  int num_fields = g_strv_length (fields);
  if (num_fields <= 1)
    {
      g_strfreev (fields);
      return NULL;
    }
  else
    {
      char *quote = strchr (fields[1], '\'');
      if (!quote)
        {
          g_strfreev (fields);
          return NULL;
        }
      *quote = 0;
      if (fields[1][0] != '<')
        votehash = strdup (fields[1]);
    }
  g_strfreev (fields);
  return votehash;
}

struct reddit_story_t
reddit_load_story_from_chunk (struct reddit_state_t *state, gchar *data, int *err, int *skip)
{
  struct reddit_story_t story;
  memset (&story, 0, sizeof (struct reddit_story_t));
  int rank = reddit_load_rank_from_story_chunk (state, data);

  if (rank <= 0)
    {
      *skip = 1;
      return story;
    }
  story.story_number = rank;

  char *id = reddit_load_id_from_story_chunk (state, data);
  if (id == NULL)
    {
      *err = -101;
      return story;
    }
  story.id = id;

  story.votehash = reddit_load_votehash_from_story_chunk (state, data);

  story.score = reddit_load_score_from_story_chunk (state, data, err);
  if (err && *err)
    {
      if (err)
        *err = -102;
      return story;
    }

  char *link = reddit_load_link_from_story_chunk (state, data);
  if (!link)
    {
      if (err)
        *err = -102;
      return story;
    }
  story.link = link;

  char *title = reddit_load_title_from_story_chunk (state, data);
  if (!title)
    {
      if (err)
        *err = -103;
      return story;
    }
  story.title = title;

  char *provider = reddit_load_provider_from_story_chunk (state, data);
  if (!provider)
    {
      if (err)
        *err = -104;
      return story;
    }
  story.provider = provider;

  char *ago = reddit_load_ago_from_story_chunk (state, data);
  if (!ago)
    {
      if (err)
        *err = -105;
      return story;
    }
  story.ago = ago;

  char *submitter = reddit_load_submitter_from_story_chunk (state, data);
  if (!submitter)
    story.submitter = strdup (_("[deleted]"));
  else
    story.submitter = submitter;

  if (!reddit_is_in_subreddit (state))
    {
      // the subreddit doesn't appear when we're already in one.
      char *subreddit = reddit_load_subreddit_from_story_chunk (state, data);
      if (!subreddit)
        {
          if (err)
            *err = -107;
          return story;
        }
      story.subreddit = subreddit;
    }

  if (state->whats != REDDIT_WHATS_NEW)
    {
      int comments = reddit_load_number_of_comments_from_story_chunk (state, 
                                                                      data);
      if (comments < 0)
        comments = 0;
      story.num_comments = comments;
    }

  return story;
}

//from kde.  gpl.
static size_t 
get_word(char *string)
{
  int tab_size = 2;
  register int i = 0, word_len = 0;

  if (!string[0])
    return 0;
  while (isspace(string[i]))
    {
      if ('\t' == string[i])
        {
          if (0 == tab_size)
            string[i] = ' ';
          else  word_len += tab_size-1;
        }
      else if ('\n' == string[i])
        string[i]=' ';
      word_len++;
      i++;
    }
  while (string[i] && !isspace(string[i++]))
    word_len++;
  return word_len;
}

//from kde. gpl.
char *
word_wrap (char *string, size_t line_len)
{
  size_t len,                         /* length of current word */
         current_len = 0;             /* current length of line */
  size_t start_line = 0;              /* index of beginning if line */

  while (0 != (len = get_word(&string[current_len + start_line])))
    {
      if (current_len + len < line_len)
        current_len += len;
      else
        {
          string[start_line+current_len] = '\n';
          start_line += current_len + 1;
          current_len = 0;
        }
    }
  return string;
}

static void
reddit_format_title (char *title, char **lines, size_t *lines_len, int wrap)
{
  char *argz = NULL;
  size_t argz_len = 0;
  char *prev = NULL;
  char *line;
  argz_create_sep (title, '\n', &argz, &argz_len);
  while ((line = argz_next (argz, argz_len, prev)))
    {
      prev = line;
      char *trimmed_line = trim (line);
      argz_add (lines, lines_len, trimmed_line);
      free (trimmed_line);
    }
  free (argz);
  argz_stringify (*lines, *lines_len, '\n');
  char *text = strdup (*lines);
  text = (char *) realloc (text, strlen (text) * 2);
  text = word_wrap (text, wrap);
  free (*lines);
  *lines = NULL;
  *lines_len = 0;
  argz_create_sep (text, '\n', lines, lines_len);
  return;
}

static char *
format_score (int score)
{
  char *s = NULL;
  if (score < 10)
    s = xasprintf (" %d  ", score);
  else if (score < 100)
    s = xasprintf (" %2d ", score);
  else if (score < 1000)
    s = xasprintf ("%3d ", score);
  else if (score < 10000)
    s = xasprintf ("%4d", score);
  else
    s = xasprintf ("%d", score);
  return s;
}

void 
reddit_show_story (struct reddit_state_t *state, struct reddit_story_t *story)
{
  char *argz = NULL;
  size_t argz_len = 0;
  char *title  = xasprintf ("%s (%s)", story->title, story->provider);
  int columns = 67;
  if (state->columns > 0)
    columns = state->columns - 13;
  reddit_format_title (title, &argz, &argz_len, columns);
  free (title);
  char *prev = NULL;
  char *line;
  int i = 0;
  int last_line = argz_count (argz, argz_len) - 1;
  while ((line = argz_next (argz, argz_len, prev)))
    {
      prev = line;
      if (i == last_line)
        fprintf (state->out, "       ↑   %s\n", line);
      else
        fprintf (state->out, "           %s\n", line);
      i++;
    }
  free (argz);
  char *formatted_score = format_score (story->score);
  fprintf (state->out, "%-4d  %s    -%s %s %s %s\n", story->story_number, 
           formatted_score, story->ago, story->submitter, 
           !reddit_is_in_subreddit (state) ? "to": "",
           !reddit_is_in_subreddit (state) ? story->subreddit: "");
  free (formatted_score);
  if (state->whats == REDDIT_WHATS_NEW)
    fprintf (state->out, "       ↓\n");
  else
    {
      if (story->num_comments == 1)
        fprintf (state->out, "       ↓      * %d comment\n", story->num_comments);
      else if (story->num_comments >= 0)
        fprintf (state->out, "       ↓      * %d comments\n", story->num_comments);
      else
        fprintf (state->out, "       ↓\n");
    }
  fprintf (state->out, "\n\n");
}

void 
reddit_free_story (struct reddit_story_t *story, int free_ptr_too)
{
  free (story->id);
  free (story->link);
  free (story->title);
  free (story->provider);
  free (story->submitter);
  free (story->subreddit);
  free (story->ago);
  free (story->votehash);
  if (free_ptr_too)
    free (story);
}
