/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "xvasprintf.h"
#include "downvote.h"
#include "upvote.h"
#include "gettext-more.h"
#include "read-file.h"
#include "stories.h"


static struct argp_option options[] = 
{
    {0}
};

static error_t 
parse_opt (int key, char *arg, struct argp_state *state)
{
  struct reddit_downvote_options_t *opt;
  opt = (struct reddit_downvote_options_t*) state->input;
  switch (key)
    {
    case ARGP_KEY_ARG:
        {
          if (opt->story_id == -1 && !opt->parsing_errors)
            {
              opt->story_id = argp_parse_story_id (state, arg, opt->state);
              if (opt->story_id == -1)
                opt->parsing_errors = 1;
            }
          else if (!opt->parsing_errors)
            {
              argp_failure (state, 0, 0, 
                            _("You may only downvote one story at a time."));
              opt->parsing_errors = 1;
            }
        }
      break;
    case ARGP_KEY_NO_ARGS:
      if (state->next <=1)
        {
          argp_failure (state, 0, 0, 
                        _("You must specify a story number to downvote."));
          argp_state_help (state, state->err_stream, ARGP_HELP_SEE);
          opt->parsing_errors = 1;
        }
      break;
    case ARGP_KEY_INIT:
      opt->story_id = -1;
      opt->parsing_errors = 0;
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

static char * 
help_filter (int key, const char *text, void *in)
{
  struct reddit_downvote_options_t *o = (struct reddit_downvote_options_t*) in;
  return argp_parse_story_help_filter (key, text, o ? o->state : NULL);
}

struct argp reddit_downvote_argp = { options, parse_opt, "STORY-NUMBER",N_("Decreases the number of people who liked a given story.  Tallies your vote against the story."), 0, help_filter };

int 
reddit_downvote_parse_argp (struct reddit_state_t *state, int argc, char **argv, struct argp *argp)
{
  int err = 0;
  struct reddit_downvote_options_t downvote_options;
  downvote_options.state = state;
  err = argp_parse (argp, argc, argv, ARGP_NO_EXIT, 0, &downvote_options);
  if (!err && downvote_options.story_id != -1 && !downvote_options.parsing_errors)
    return reddit_downvote (state, &downvote_options);
  return 0;
}

int 
reddit_downvote_story (struct reddit_state_t *state, struct reddit_downvote_options_t *options, struct reddit_story_t *story, int *resp, char **errmsg)
{
  return reddit_vote_story (state, -1, story, resp, errmsg);
}

int 
reddit_downvote (struct reddit_state_t *state, struct reddit_downvote_options_t *options)
{
  int err = 0;
  if (state->logged_in == 0)
    {
      fprintf (state->out, _("You need to login to vote on a story."
                             "  Login using the `login' command.\n"));
      return 0;
    }
  struct reddit_story_t *story = 
    reddit_find_story_with_id (state, options->story_id);
  int response = 0;
  char *errmsg = NULL;
  if (story)
    err = reddit_downvote_story (state, options, story, &response, &errmsg);

  if (!err)
    reddit_update_modhash (state);
  if (!err)
    fprintf (state->out, _("The story has been downvoted.\n"));
  else
    {
      fprintf (state->out, _("Error: Couldn't downvote the story "
                             "HTTP /1.1 %d err=%d)\n"), response, err);
      if (errmsg)
        {
          fprintf (state->out, _("Maybe this gobbledegook will be useful "
                                 "in diagnosing what went wrong:\n"
                                 "%s\n"), errmsg);
          free (errmsg);
        }
    }
  return err;
}
